/*

Copyright (C) 2012 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.

 */

package com.clustercontrol.systemlog.service;

import java.util.ArrayList;
import java.util.Comparator;
import java.util.Date;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.TreeMap;
import java.util.Map.Entry;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.commons.util.HinemosProperties;
import com.clustercontrol.systemlog.bean.SyslogMessage;

public class SystemLogSuppressor {

	private static final Log log = LogFactory.getLog(SystemLogSuppressor.class);

	private final Map<Long, Map<String, ReceiverStat>> _recentSyslog;
	private int gcCounter = 0;
	private int recentSyslogCounter = 0;

	private static final int _maxRecentSyslog;
	private static final long _maxLifetimeSyslogMsec;

	static {
		// 30 [msg/sec] * 60 [sec] * 15 [min] = 27000
		_maxRecentSyslog = Integer.parseInt(HinemosProperties.getProperty("monitor.systemlog.recent.maxsize", "27000"));
		// 15 [min] * 60 [sec] * 1000 [msec] = 900000 [msec]
		_maxLifetimeSyslogMsec = Long.parseLong(HinemosProperties.getProperty("monitor.systemlog.recent.lifetime", "900000"));
	}

	public SystemLogSuppressor() {
		_recentSyslog = new TreeMap<Long, Map<String, ReceiverStat>>(
				new Comparator<Long>() {
					@Override
					public int compare(Long key1, Long key2) {
						// syslog日付の降順に並ぶように実装
						long diff = key2 - key1;
						if (diff > Integer.MAX_VALUE) {
							return Integer.MAX_VALUE;
						} else if (diff < Integer.MIN_VALUE) {
							return Integer.MIN_VALUE;
						} else {
							return (int)diff;
						}
					}
				}
				);
	}

	public synchronized boolean isSuppress(String receiverId, SyslogMessage syslog) {
		if (++gcCounter > _maxRecentSyslog) {
			gcRecentSyslog();
			gcCounter = 0;
		}

		if (log.isDebugEnabled()) {
			log.debug("checking receiverId = " + receiverId + ", syslog = " + syslog + ")");
		}

		ReceiverStat counter = null;
		if (_recentSyslog.containsKey(syslog.date)) {
			// 該当時間帯に出力したsyslogが存在するか
			Map<String, ReceiverStat> syslogMap = _recentSyslog.get(syslog.date);

			if (syslogMap.containsKey(syslog.rawSyslog)) {
				// 該当時間に出力された同一のsyslogを受信している場合、
				// 各receiverの受信統計情報を取得する
				counter = syslogMap.get(syslog.rawSyslog);
			} else {
				if (recentSyslogCounter < _maxRecentSyslog) {
					// 該当時間帯に出力された同一のsyslogを受信していない場合、
					// そのsyslogの受信統計情報を新規生成する
					counter = new ReceiverStat();
					syslogMap.put(syslog.rawSyslog, counter);
					recentSyslogCounter++;
				} else {
					// 抑制判定用のsyslogストアが一定量を超えたら、syslogストアを更新しない
					// よって、生存期間で空きが出るまで、重複して監視される
					log.warn("too many syslog. skip update list of suppression : " + syslog);
					return false;
				}
			}
		} else {
			if (recentSyslogCounter < _maxRecentSyslog) {
				// 該当時間に出力されたsyslogが存在しない場合、
				// その時間の統計情報一覧を新規生成する
				Map<String, ReceiverStat> syslogMap = new HashMap<String, ReceiverStat>();
				counter = new ReceiverStat();
				syslogMap.put(syslog.rawSyslog, counter);
				_recentSyslog.put(syslog.date, syslogMap);
				recentSyslogCounter++;
			} else {
				// 抑制判定用のsyslogストアが一定量を超えたら、syslogストアを更新しない
				// よって、生存期間で空きが出るまで、重複して監視される
				log.warn("too many syslog. skip update list of suppression : " + syslog);
				return false;
			}
		}

		// 該当syslogの受信統計情報から抑制するかどうかを返す
		return counter.isSuppress(receiverId);
	}

	private void gcRecentSyslog() {
		long now = System.currentTimeMillis();
		long thresTime = now - _maxLifetimeSyslogMsec;

		log.info("cleaning recent syslog map. (garbage syslog before " + new Date(thresTime) + ")");

		List<Long> removeList = new ArrayList<Long>();

		recentSyslogCounter = 0;
		for (Iterator<Entry<Long, Map<String, ReceiverStat>>> it = _recentSyslog.entrySet().iterator(); it.hasNext();) {
			Entry<Long, Map<String, ReceiverStat>> entry = it.next();

			if (log.isDebugEnabled()) {
				log.debug("checking recent syslog. (date = " + new Date(entry.getKey()) + ", syslog = " + entry.getKey() + ")");
			}

			// 生存期間を超えたsyslog情報は破棄する
			// (万が一、これを超えて同一のsyslog情報が届いた場合、抑制されずに通知される)
			if (entry.getKey() < thresTime) {
				if (log.isDebugEnabled()) {
					log.debug("removing recent syslog. (date = " + new Date(entry.getKey()) + ")");
				}
				removeList.add(entry.getKey());
			} else {
				recentSyslogCounter += entry.getValue().size();
			}
		}

		for (Long key : removeList) {
			_recentSyslog.remove(key);
		}
	}

	private class ReceiverStat {
		private final Map<String, Integer> counter = new HashMap<String, Integer>();

		public boolean isSuppress(String receiverId) {
			int receiverCount = 0;
			int maxCount = 0;

			// receiverが初めてそのsyslogを受信した場合
			if (! counter.containsKey(receiverId)) {
				counter.put(receiverId, 0);
			}

			// 各receiverの統計情報を確認し、最大受信数とreceiverの受信数を取得
			for (String key : counter.keySet()) {
				if (counter.get(key) > maxCount) {
					maxCount = counter.get(key);
				}
				if (key.equals(receiverId)) {
					receiverCount = counter.get(key);
				}
			}

			// 該当receiverの受信数をカウントアップする
			counter.put(receiverId, receiverCount + 1);

			if (receiverCount < maxCount) {
				// 既に監視している系が存在するため、抑制対象となる
				return true;
			} else {
				// 監視している系が存在しないため、監視対象となる
				return false;
			}
		}
	}

}
