/*

Copyright (C) 2006 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.

 */

package com.clustercontrol.notify.util;

import java.io.File;
import java.io.FileInputStream;
import java.io.UnsupportedEncodingException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.Locale;
import java.util.Map;
import java.util.Properties;
import java.util.StringTokenizer;
import java.util.Vector;

import javax.mail.AuthenticationFailedException;
import javax.mail.Message;
import javax.mail.MessagingException;
import javax.mail.Session;
import javax.mail.Transport;
import javax.mail.internet.AddressException;
import javax.mail.internet.InternetAddress;
import javax.mail.internet.MimeMessage;
import javax.mail.internet.MimeUtility;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.bean.PriorityConstant;
import com.clustercontrol.notify.bean.NotifyRequestMessage;
import com.clustercontrol.notify.bean.OutputBasicInfo;
import com.clustercontrol.notify.mail.bean.MailTemplateInfo;
import com.clustercontrol.notify.mail.session.MailTemplateControllerBean;
import com.clustercontrol.notify.model.NotifyMailInfoEntity;
import com.clustercontrol.util.Messages;
import com.clustercontrol.util.StringBinder;
import com.clustercontrol.util.apllog.AplLogger;
import com.sun.mail.smtp.SMTPAddressFailedException;

/**
 * メールを送信するクラス<BR>
 * 
 * @version 3.0.0
 * @since 3.0.0
 */
public class SendMail implements Notifier {
	/** ログ出力のインスタンス。 */
	private static Log m_log = LogFactory.getLog(SendMail.class);

	public static final String _configFilePath;
	public static final String _configFilePathDefault = System.getProperty("hinemos.manager.etc.dir") + File.separator + "mail-service.properties";

	public static final Properties _properties = new Properties();

	/** 日時フォーマット。 */
	private static final String SUBJECT_DATE_FORMAT = "yyyy-MM-dd HH:mm:ss";

	/** 接続ユーザ名 */
	private static final String _loginUser;
	/** 接続パスワード */
	private static final String _loginPassword;

	/** 差出人アドレス。 */
	private static final String _fromAddress;
	/** 差出人個人名。 */
	private static final String _fromPersonalName;
	/** 返信の送信先アドレス。 */
	private static final String _replyToAddress;
	/** 返信の送信先個人名。 */
	private static final String _replyToPersonalName;
	/** エラー送信先アドレス。 */
	private static final String _errorsToAddress;
	/** 再送回数。 */
	private static final int _transportTries;
	/** 再送スリープ */
	private static final int _transportTriesInterval; // デフォルトは10秒。
	/** ロケール情報。 */
	private static final Locale m_local = Locale.getDefault();

	/** アドレスロケール */
	public static final String _charsetAddress;
	public static final String _charsetAddressDefault = "UTF-8";

	/** メール件名ロケール */
	public static final String _charsetSubject;
	public static final String _charsetSubjectDefault = "UTF-8";

	/**`メール本文ロケール */
	public static final String _charsetContent;
	public static final String _charsetContentDefault = "UTF-8";

	static {
		_configFilePath = System.getProperty("hinemos.log4j.file", _configFilePathDefault);

		FileInputStream fin = null;
		try {
			fin = new FileInputStream(_configFilePath);
			_properties.load(fin);
		} catch (Exception e) {
			m_log.warn("config file loading failure. (" + _configFilePath + ")", e);
		} finally {
			if (fin != null) {
				try {
					fin.close();
				} catch (Exception e) {
					m_log.warn("file closing failure.", e);
				}
			}
		}

		/**
		 * メールの設定をプロパティファイルから取得
		 */
		_loginUser = _properties.getProperty("hinemos.mail.transport.user");
		_loginPassword = _properties.getProperty("hinemos.mail.transport.password");

		_fromAddress = _properties.getProperty("hinemos.mail.from.address");
		_fromPersonalName = _properties.getProperty("hinemos.mail.from.personal.name");
		_replyToAddress = _properties.getProperty("hinemos.mail.reply.to.address");
		_replyToPersonalName = _properties.getProperty("hinemos.mail.reply.personal.name");
		_errorsToAddress = _properties.getProperty("hinemos.mail.errors.to.address");

		_transportTries = Integer.parseInt(_properties.getProperty("hinemos.mail.tranport.tries", "1"));
		_transportTriesInterval = Integer.parseInt(_properties.getProperty("hinemos.mail.tranport.tries.interval", "10000"));

		_charsetAddress = _properties.getProperty("hinemos.mail.charset.address", _charsetAddressDefault);
		_charsetSubject = _properties.getProperty("hinemos.mail.charset.subject", _charsetSubjectDefault);
		_charsetContent = _properties.getProperty("hinemos.mail.charset.content", _charsetContentDefault);

		m_log.info("initialized mail sender : from_address = " + _fromAddress
				+ ", From = " + _fromPersonalName + " <" + _replyToAddress + ">"
				+ ", Reply-To = " + _replyToPersonalName + " <" + _replyToAddress + ">"
				+ ", Errors-To = " + _errorsToAddress
				+ ", tries = " + _transportTries
				+ ", tries-interval = " + _transportTriesInterval
				+ ", Charset [address:subject:content] = [" + _charsetAddress + ":" + _charsetSubject + ":" + _charsetContent + "]");
	}

	/**
	 * メールの送信を行います。
	 * 
	 * @param outputInfo 出力・通知情報
	 */
	@Override
	public synchronized void notify(NotifyRequestMessage message) {

		sendMail(message.getOutputInfo(), message.getNotifyId());
	}

	/**
	 * メールの送信を行います。
	 * 
	 */
	private void sendMail(OutputBasicInfo outputInfo, String notifyId) {

		if (m_log.isDebugEnabled()) {
			m_log.debug("sendMail() " + outputInfo);
		}

		try {
			NotifyMailInfoEntity mailInfo = QueryUtil.getNotifyMailInfoPK(notifyId, outputInfo.getPriority());

			// メールの件名を指定
			String subject = getSubject(outputInfo, mailInfo);

			// メールの内容を指定
			String content = getContent(outputInfo, mailInfo);

			/**
			 * メール送信
			 */
			String address = mailInfo.getMailAddress();
			StringTokenizer t = new StringTokenizer(address, ";");
			ArrayList<String> addressList = new ArrayList<String>();
			while (t.hasMoreTokens()) {
				addressList.add(t.nextToken());
			}
			String[] addresses = addressList.toArray(new String[0]);

			if (addresses == null || addresses.length <= 0) {
				m_log.debug("sendMail() : mail address is empty");
				return;
			}

			try {
				this.sendMail(addresses, subject, content);
			} catch (AuthenticationFailedException e) {
				String detailMsg = "cannot connect to the mail server due to an Authentication Failure";
				m_log.warn("sendMail() " + e.getMessage() + " : " + detailMsg + " : "
						+ e.getClass().getSimpleName() + ", " + e.getMessage(), e);
				internalErrorNotify(notifyId, "007", detailMsg);
			} catch (SMTPAddressFailedException e) {
				String detailMsg = e.getMessage() + "(SMTPAddressFailedException)";
				m_log.warn("sendMail() " + e.getMessage() + " : " + detailMsg + " : "
						+ e.getClass().getSimpleName() + ", " + e.getMessage(), e);
				internalErrorNotify(notifyId, "007", detailMsg);
			} catch (MessagingException e) {
				String detailMsg = e.getCause() != null ? e.getMessage() + "\nCause : " + e.getCause().getMessage() : e.getMessage();
				m_log.warn("sendMail() " + e.getMessage() + " : " + detailMsg + " : "
						+ e.getClass().getSimpleName() + ", " + e.getMessage(), e);
				internalErrorNotify(notifyId, "007", detailMsg);
			}
		} catch (Exception e1) {
			String detailMsg = e1.getCause() != null ? e1.getMessage() + "\nCause : " + e1.getCause().getMessage() : e1.getMessage();
			m_log.warn("sendMail() " + e1.getMessage() + " : " + detailMsg + detailMsg + " : "
					+ e1.getClass().getSimpleName() + ", " + e1.getMessage(), e1);
			internalErrorNotify(notifyId, "007", detailMsg);
		}
	}

	/**
	 * メールを送信します。
	 * 
	 * <p>
	 * 下記の情報は、ファイルより取得します。
	 * <p>
	 * <ul>
	 * <li>差出人アドレス</li>
	 * <li>差出人個人名</li>
	 * <li>返信の送信先アドレス</li>
	 * <li>返信の送信先個人名</li>
	 * <li>エラー送信先アドレス</li>
	 * </ul>
	 * 
	 * @param address
	 *            送信先アドレス
	 * @param source
	 *            出力内容
	 * @return 送信に成功した場合、<code> true </code>
	 * @throws MessagingException
	 * @throws NamingException
	 * @throws UnsupportedEncodingException
	 */
	public void sendMail(String[] address, String subject, String content)
			throws MessagingException, UnsupportedEncodingException {

		if (address == null || address.length <= 0) {
			// 何もせず終了
			return;
		}

		// JavaMail Sessionリソース検索
		Session session = Session.getDefaultInstance(_properties);

		Message mineMsg = new MimeMessage(session);

		// 送信元メールアドレスと送信者名を指定
		if (_fromAddress != null && _fromPersonalName != null) {
			mineMsg.setFrom(new InternetAddress(_fromAddress, _fromPersonalName, _charsetAddress));
		} else if (_fromAddress != null && _fromPersonalName == null) {
			mineMsg.setFrom(new InternetAddress(_fromAddress));
		}
		// REPLY-TOを指定
		if (_replyToAddress != null && _replyToPersonalName != null) {
			InternetAddress reply[] = { new InternetAddress(_replyToAddress, _replyToPersonalName, _charsetAddress) };
			mineMsg.setReplyTo(reply);
			mineMsg.reply(true);
		} else if (_replyToAddress != null && _replyToPersonalName == null) {
			InternetAddress reply[] = { new InternetAddress(_replyToAddress) };
			mineMsg.setReplyTo(reply);
			mineMsg.reply(true);
		}

		// ERRORS-TOを指定
		if (_errorsToAddress != null) {
			mineMsg.setHeader("Errors-To", _errorsToAddress);
		}

		// 送信先メールアドレスを指定
		InternetAddress[] toAddress = this.getAddress(address);
		if (toAddress == null || toAddress.length <= 0) {
			// 何もせず終了
			return;
		}
		mineMsg.setRecipients(javax.mail.Message.RecipientType.TO, toAddress);

		// メールの件名を指定
		mineMsg.setSubject(MimeUtility.encodeText(subject, _charsetSubject, "B"));

		// メールの内容を指定
		mineMsg.setContent(content, "text/plain; charset=" + _charsetContent);

		// 送信日付を指定
		mineMsg.setSentDate(new Date());

		// 再送信フラグがtrueかつ再送回数以内の場合
		for (int i = 0; i < _transportTries; i++) {
			Transport transport = null;
			try {
				// メール送信
				transport = session.getTransport();
				transport.connect(_loginUser, _loginPassword);
				transport.sendMessage(mineMsg, mineMsg.getAllRecipients());
				break;
			} catch (AuthenticationFailedException e) {
				throw e;
			} catch (SMTPAddressFailedException e) {
				throw e;
			} catch (MessagingException me) {
				if (i < _transportTries) {
					m_log.info("sendMail() : retry sendmail. " + me.getMessage());
					try {
						Thread.sleep(_transportTriesInterval);
					} catch (InterruptedException e) { }
				} else {
					throw me;
				}
			} finally {
				if (transport != null) {
					transport.close();
				}
			}
		}
	}

	/**
	 * 引数で指定された送信先アドレスの<code> InternetAddress </code>オブジェクトを返します。
	 * 
	 * @param addressList
	 *            送信先アドレスの文字列配列
	 * @return <code> InternetAddress </code>オブジェクトの配列
	 */
	private InternetAddress[] getAddress(String[] addressList) {
		InternetAddress toAddress[] = null;
		Vector<InternetAddress> list = new Vector<InternetAddress>();
		if (addressList != null) {
			for (String address : addressList) {
				try {
					list.add(new InternetAddress(address));
				} catch (AddressException e) {
					m_log.info("getAddress() : "
							+ e.getClass().getSimpleName() + ", "
							+ address + ", "
							+ e.getMessage());
				}
			}
			if (list.size() > 0) {
				toAddress = new InternetAddress[list.size()];
				list.copyInto(toAddress);
			}
		}
		return toAddress;
	}

	/**
	 * メール件名を返します。
	 * 
	 * @param source
	 *            出力内容
	 * @param mailInfo
	 *            通知内容
	 * @return メール件名
	 */
	public String getSubject(OutputBasicInfo source,
			NotifyMailInfoEntity mailInfo) {

		String subject = null;
		try {
			if (mailInfo != null
					&& mailInfo.getMailTemplateInfoEntity() != null
					&& mailInfo.getMailTemplateInfoEntity().getMailTemplateId() != null) {
				MailTemplateInfo templateData
				= new MailTemplateControllerBean().getMailTemplateInfo(
						mailInfo.getMailTemplateInfoEntity().getMailTemplateId());
				Map<String, String> param = NotifyUtil.createParameter(source, mailInfo.getNotifyInfoEntity());
				StringBinder binder = new StringBinder(param);
				subject = binder.bindParam(templateData.getSubject());
			} else {
				subject = Messages.getString("mail.subject", m_local) + "("
						+ PriorityConstant.typeToString(source.getPriority())
						+ ")";
			}
		} catch (Exception e) {
			m_log.warn("getSubject() : "
					+ e.getClass().getSimpleName() + ", " + e.getMessage(), e);
			// 例外発生時のメールサブジェクト
			return "Hinemos Notification";
		}

		return subject;
	}

	/**
	 * メール本文を返します。
	 * 
	 * @param source
	 *            出力内容
	 * @param mailInfo
	 *            通知内容
	 * @return メール本文
	 */
	public String getContent(OutputBasicInfo source, NotifyMailInfoEntity mailInfo) {

		StringBuffer buf = new StringBuffer();
		SimpleDateFormat sdf = new SimpleDateFormat(SUBJECT_DATE_FORMAT);

		try {
			if (mailInfo != null
					&& mailInfo.getMailTemplateInfoEntity() != null
					&& mailInfo.getMailTemplateInfoEntity().getMailTemplateId() != null) {
				MailTemplateInfo mailData
				= new MailTemplateControllerBean().getMailTemplateInfo(
						mailInfo.getMailTemplateInfoEntity().getMailTemplateId());
				Map<String, String> param = NotifyUtil.createParameter(source,
						mailInfo.getNotifyInfoEntity());
				StringBinder binder = new StringBinder(param);
				buf.append(binder.bindParam(mailData.getBody() + "\n"));
			} else {
				buf.append(Messages.getString("generation.time", m_local)
						+ " : "
						+ sdf.format(source.getGenerationDate()) + "\n");
				buf.append(Messages.getString("application", m_local) + " : "
						+ source.getApplication() + "\n");
				buf.append(Messages.getString("priority", m_local) + " : "
						+ PriorityConstant.typeToString(source.getPriority())
						+ "\n");
				buf.append(Messages.getString("message", m_local) + " : "
						+ source.getMessage() + "\n");
				buf.append(Messages.getString("scope", m_local) + " : "
						+ source.getScopeText() + "\n");
			}
		} catch (Exception e) {
			m_log.warn("getContent() : "
					+ e.getClass().getSimpleName() + ", " + e.getMessage(), e);
			// 例外発生時のメール本文
			return "An error occurred creating message.";
		}

		// 改行コードをLFからCRLFに変更する。
		// 本来はJavaMailが変換するはずだが、変換されないこともあるので、
		// 予め変更しておく。
		String ret = buf.toString().replaceAll("\r\n", "\n").replaceAll("\n", "\r\n");

		return ret;
	}

	/**
	 * 通知失敗時の内部エラー通知を定義します
	 */
	@Override
	public void internalErrorNotify(String notifyId, String msgID, String detailMsg) {
		AplLogger apllog = new AplLogger("NOTIFY", "notify");
		String[] args = { notifyId };
		// 通知失敗メッセージを出力
		apllog.put("SYS", msgID, args, detailMsg);
	}
}
