package com.clustercontrol.notify.model;

import java.io.Serializable;
import javax.persistence.*;

import com.clustercontrol.bean.HinemosModuleConstant;
import com.clustercontrol.commons.util.JpaTransactionManager;


import java.sql.Timestamp;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;


/**
 * The persistent class for the cc_notify_history database table.
 * 
 */
@Entity
@Table(name="cc_notify_history")
@Cacheable(true)
public class NotifyHistoryEntity implements Serializable {
	private static final long serialVersionUID = 1L;
	private NotifyHistoryEntityPK id;
	private Timestamp lastNotify;
	private Integer priority;
	private NotifyInfoEntity notifyInfoEntity;

	@Deprecated
	public NotifyHistoryEntity() {
	}

	public NotifyHistoryEntity(NotifyHistoryEntityPK pk,
			NotifyInfoEntity notifyInfoEntity) {
		this.setId(pk);
		EntityManager em = new JpaTransactionManager().getEntityManager();
		em.persist(this);
		
		// ジョブからの通知はメモリリークにつながるので、
		// relationを定義しない。
		if (!HinemosModuleConstant.JOB.equals(pk.getPluginId())) {
			this.relateToNotifyInfoEntity(notifyInfoEntity);
		}
	}

	public NotifyHistoryEntity(NotifyInfoEntity notifyInfoEntity,
			String facilityId,
			String pluginId,
			String monitorId,
			String subKey) {
		this(new NotifyHistoryEntityPK(facilityId,
				pluginId,
				monitorId,
				notifyInfoEntity.getNotifyId(),
				subKey), notifyInfoEntity);
	}

	@EmbeddedId
	public NotifyHistoryEntityPK getId() {
		return this.id;
	}

	public void setId(NotifyHistoryEntityPK id) {
		this.id = id;
	}


	@Column(name="last_notify")
	public Timestamp getLastNotify() {
		return this.lastNotify;
	}

	public void setLastNotify(Timestamp lastNotify) {
		this.lastNotify = lastNotify;
	}


	public Integer getPriority() {
		return this.priority;
	}

	public void setPriority(Integer priority) {
		this.priority = priority;
	}


	//bi-directional many-to-one association to NotifyInfoEntity
	@ManyToOne(fetch=FetchType.LAZY)
	@JoinColumn(name="notify_id", insertable=false, updatable=false)
	public NotifyInfoEntity getNotifyInfoEntity() {
		return this.notifyInfoEntity;
	}

	@Deprecated
	public void setNotifyInfoEntity(NotifyInfoEntity notifyInfoEntity) {
		this.notifyInfoEntity = notifyInfoEntity;
	}

	/**
	 * NotifyInfoEntityオブジェクト参照設定<BR>
	 * 
	 * NotifyInfoEntity設定時はSetterに代わりこちらを使用すること。
	 * 
	 * JPAの仕様(JSR 220)では、データ更新に伴うrelationshipの管理はユーザに委ねられており、
	 * INSERTやDELETE時に、そのオブジェクトに対する参照をメンテナンスする処理を実装する。
	 * 
	 * JSR 220 3.2.3 Synchronization to the Database
	 * 
	 * Bidirectional relationships between managed entities will be persisted
	 * based on references held by the owning side of the relationship.
	 * It is the developer’s responsibility to keep the in-memory references
	 * held on the owning side and those held on the inverse side consistent
	 * with each other when they change.
	 */
	public void relateToNotifyInfoEntity(NotifyInfoEntity notifyInfoEntity) {
		this.setNotifyInfoEntity(notifyInfoEntity);
		if (notifyInfoEntity != null) {
			List<NotifyHistoryEntity> list = notifyInfoEntity.getNotifyHistoryEntities();
			if (list == null) {
				list = new ArrayList<NotifyHistoryEntity>();
			} else {
				for(NotifyHistoryEntity entity : list){
					if (entity.getId().equals(this.getId())) {
						return;
					}
				}
			}
			list.add(this);
			notifyInfoEntity.setNotifyHistoryEntities(list);
		}
	}

	/**
	 * 削除前処理<BR>
	 * 
	 * JPAの仕様(JSR 220)では、データ更新に伴うrelationshipの管理はユーザに委ねられており、
	 * INSERTやDELETE時に、そのオブジェクトに対する参照をメンテナンスする処理を実装する。
	 * 
	 * JSR 220 3.2.3 Synchronization to the Database
	 * 
	 * Bidirectional relationships between managed entities will be persisted
	 * based on references held by the owning side of the relationship.
	 * It is the developer’s responsibility to keep the in-memory references
	 * held on the owning side and those held on the inverse side consistent
	 * with each other when they change.
	 */
	public void unchain() {

		// NotifyInfoEntity
		if (this.notifyInfoEntity != null) {
			List<NotifyHistoryEntity> list = this.notifyInfoEntity.getNotifyHistoryEntities();
			if (list != null) {
				Iterator<NotifyHistoryEntity> iter = list.iterator();
				while(iter.hasNext()) {
					NotifyHistoryEntity entity = iter.next();
					if (entity.getId().equals(this.getId())){
						iter.remove();
						break;
					}
				}
			}
		}
	}

}