/*

 Copyright (C) 2006 NTT DATA Corporation

 This program is free software; you can redistribute it and/or
 Modify it under the terms of the GNU General Public License
 as published by the Free Software Foundation, version 2.

 This program is distributed in the hope that it will be
 useful, but WITHOUT ANY WARRANTY; without even the implied
 warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
 PURPOSE.  See the GNU General Public License for more details.

 */

package com.clustercontrol.monitor.run.composite;

import java.util.ArrayList;
import java.util.List;

import org.eclipse.jface.dialogs.IDialogConstants;
import org.eclipse.jface.viewers.DoubleClickEvent;
import org.eclipse.jface.viewers.IDoubleClickListener;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.swt.SWT;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.Table;
import org.eclipse.ui.PlatformUI;

import com.clustercontrol.dialog.ValidateResult;
import com.clustercontrol.monitor.run.dialog.StringValueInfoCreateDialog;
import com.clustercontrol.monitor.run.util.StringValueInfoManager;
import com.clustercontrol.monitor.run.viewer.StringValueListTableViewer;
import com.clustercontrol.ws.monitor.MonitorInfo;
import com.clustercontrol.ws.monitor.MonitorStringValueInfo;

/**
 * 文字列監視の判定情報一覧コンポジットクラス<BR>
 * 
 * @version 3.0.0
 * @since 2.1.0
 */
public class StringValueListComposite extends Composite {

	/** テーブルビューアー。 */
	private StringValueListTableViewer m_tableViewer = null;

	/** テーブル定義情報。 */
	private ArrayList m_tableDefine = null;

	/**
	 * インスタンスを返します。
	 * <p>
	 * 初期処理を呼び出し、コンポジットを配置します。
	 * 
	 * @param parent 親のコンポジット
	 * @param style スタイル
	 * 
	 * @see org.eclipse.swt.SWT
	 * @see org.eclipse.swt.widgets.Composite#Composite(Composite parent, int style)
	 * @see #initialize()
	 */
	public StringValueListComposite(Composite parent, int style) {
		super(parent, style);

		this.initialize();
	}

	/**
	 * インスタンスを返します。
	 * <p>
	 * 初期処理を呼び出し、コンポジットを配置します。
	 * 
	 * @param parent 親のコンポジット
	 * @param style スタイル
	 * @param tableDefine 判定情報一覧のテーブル定義情報（{@link com.clustercontrol.bean.TableColumnInfo}のリスト）
	 * 
	 * @see org.eclipse.swt.SWT
	 * @see org.eclipse.swt.widgets.Composite#Composite(Composite parent, int style)
	 * @see com.clustercontrol.bean.TableColumnInfo#TableColumnInfo(java.lang.String, int, int, int)
	 * @see com.clustercontrol.monitor.run.action.GetStringFilterTableDefine
	 * @see #initialize()
	 */
	public StringValueListComposite(Composite parent, int style, ArrayList tableDefine) {
		super(parent, style);

		this.m_tableDefine = tableDefine;

		this.initialize();
	}

	/**
	 * コンポジットを配置します。
	 */
	private void initialize() {
		GridLayout layout = new GridLayout(1, true);
		this.setLayout(layout);
		layout.marginHeight = 0;
		layout.marginWidth = 0;

		Table table = new Table(this, SWT.SINGLE | SWT.H_SCROLL | SWT.V_SCROLL | SWT.FULL_SELECTION);
		table.setHeaderVisible(true);
		table.setLinesVisible(true);

		GridData gridData = new GridData();
		gridData.horizontalAlignment = GridData.FILL;
		gridData.verticalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		gridData.grabExcessVerticalSpace = true;
		table.setLayoutData(gridData);

		// テーブルビューアの作成
		this.m_tableViewer = new StringValueListTableViewer(table);
		this.m_tableViewer.createTableColumn(this.m_tableDefine);
		this.m_tableViewer.addDoubleClickListener(new IDoubleClickListener() {
			@Override
			public void doubleClick(DoubleClickEvent event) {
				MonitorStringValueInfo info = getFilterItem();
				if (info != null) {
					Shell shell = PlatformUI.getWorkbench().getActiveWorkbenchWindow().getShell();
					StringValueInfoCreateDialog dialog = new StringValueInfoCreateDialog(shell, info.getIdentifier());
					if (dialog.open() == IDialogConstants.OK_ID) {
						dialog.getInputData();
						Table table = getTableViewer().getTable();
						int selectIndex = table.getSelectionIndex();
						StringValueInfoManager.getInstance().modify(dialog.getInputData());
						table.setSelection(selectIndex);
						update();
					}
				}
			}
		});

		gridData = new GridData();
		gridData.horizontalAlignment = GridData.FILL;
		gridData.verticalAlignment = GridData.FILL;
	}

	/**
	 * このコンポジットが利用するテーブルビューアーを返します。
	 * 
	 * @return テーブルビューアー
	 */
	public StringValueListTableViewer getTableViewer() {
		return this.m_tableViewer;
	}

	/**
	 * 現在選択されているアイテムを返します。
	 * <p>
	 * 選択されていない場合は、<code>null</code>を返します。
	 * 
	 * @return 選択アイテム
	 */
	public MonitorStringValueInfo getFilterItem() {
		StructuredSelection selection = (StructuredSelection) this.m_tableViewer.getSelection();

		if (selection == null) {
			return null;
		} else {
			return (MonitorStringValueInfo) selection.getFirstElement();
		}
	}

	/**
	 * 引数で指定された監視情報の値を、各項目に設定します。
	 * 
	 * @param info 設定値として用いる監視情報
	 */
	public void setInputData(MonitorInfo info) {

		if(info != null){
			// 文字列監視判定情報設定
			StringValueInfoManager.getInstance().initialize(info);
			// データ取得
			Object[] list = StringValueInfoManager.getInstance().get();

			// テーブル更新
			this.m_tableViewer.setInput(list);
		}
	}

	/**
	 * コンポジットを更新します。<BR>
	 * 判定情報一覧を取得し、テーブルビューアーにセットします。
	 * 
	 * @see com.clustercontrol.monitor.run.util.StringValueInfoManager#get()
	 * @see com.clustercontrol.monitor.run.viewer.StringValueListTableViewer
	 */
	@Override
	public void update() {
		// データ取得
		Object[] list = StringValueInfoManager.getInstance().get();

		// テーブル更新
		this.m_tableViewer.setInput(list);
	}

	/**
	 * 引数で指定された監視情報に、入力値を設定します。
	 * <p>
	 * 入力値チェックを行い、不正な場合は認証結果を返します。
	 * 不正ではない場合は、<code>null</code>を返します。
	 * 
	 * @param monitorInfo 入力値を設定する監視情報
	 * @return 検証結果
	 * 
	 * @see #setValidateResult(String, String)
	 */
	public ValidateResult createInputData(MonitorInfo monitorInfo) {

		ArrayList<MonitorStringValueInfo> list = StringValueInfoManager.getInstance().getMonitorStringValueInfo();

		if(list != null && list.size() > 0){
			String MonitorTypeId = monitorInfo.getMonitorTypeId();
			String MonitorId = monitorInfo.getMonitorId();


			for(int index=0; index<list.size(); index++){
				MonitorStringValueInfo info = list.get(index);
				info.setMonitorTypeId(MonitorTypeId);
				info.setMonitorId(MonitorId);
			}
			List<MonitorStringValueInfo> monitorStringValueInfoList
			= monitorInfo.getStringValueInfo();
			monitorStringValueInfoList.clear();
			monitorStringValueInfoList.addAll(list);
		}
		return null;
	}

	/* (非 Javadoc)
	 * @see org.eclipse.swt.widgets.Control#setEnabled(boolean)
	 */
	@Override
	public void setEnabled(boolean enabled) {
		this.m_tableViewer.getTable().setEnabled(enabled);
	}

	/**
	 * 無効な入力値の情報を設定します。
	 * 
	 * @param id ID
	 * @param message メッセージ
	 * @return 認証結果
	 */
	protected ValidateResult setValidateResult(String id, String message) {

		ValidateResult validateResult = new ValidateResult();
		validateResult.setValid(false);
		validateResult.setID(id);
		validateResult.setMessage(message);

		return validateResult;
	}
}