/*

 Copyright (C) 2006 NTT DATA Corporation

 This program is free software; you can redistribute it and/or
 Modify it under the terms of the GNU General Public License
 as published by the Free Software Foundation, version 2.

 This program is distributed in the hope that it will be
 useful, but WITHOUT ANY WARRANTY; without even the implied
 warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
 PURPOSE.  See the GNU General Public License for more details.

 */
package com.clustercontrol.winservice.util;

import intel.management.wsman.ManagedInstance;
import intel.management.wsman.ManagedReference;
import intel.management.wsman.WsmanConnection;
import intel.management.wsman.WsmanException;
import intel.management.wsman.WsmanUtils;

import java.net.Inet6Address;
import java.net.InetAddress;
import java.net.UnknownHostException;
import java.util.Date;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.http.conn.ssl.SSLSocketFactory;

import com.clustercontrol.fault.HinemosUnknown;
import com.clustercontrol.util.Messages;

/**
 * WinRMを使用したWindowsサービス監視実行クラス
 * 
 * @version 4.0.0
 * @since 4.0.0
 */
public class RequestWinRM {

	private static Log m_log = LogFactory.getLog( RequestWinRM.class ) ;

	private static final String URI_WIN32_SERVICE = "http://schemas.microsoft.com/wbem/wsman/1/wmi/root/cimv2/Win32_Service";
	private static final String STATE_RUNNING = "Running";

	// Windowsサービス名
	private String m_serviceName;

	private String m_message;
	private String m_messageOrg;
	private long m_date;

	private WsmanConnection m_con;
	private String m_url;
	private String m_state;

	/**
	 * デフォルトコンストラクタ
	 * @param serviceName
	 */
	public RequestWinRM(String serviceName) {
		this.m_serviceName = serviceName;
	}

	/**
	 * WinRMへアクセスして指定のWindowsサービスの状態がRunningであることを確認する
	 * 
	 * @param ipAddress
	 * @param user
	 * @param userPassword
	 * @param port
	 * @param protocol
	 * @param timeout
	 * @param retries
	 * @return
	 */
	public boolean polling (
			String ipAddress,
			String user,
			String userPassword,
			int port,
			String protocol,
			int timeout,
			int retries
	) throws HinemosUnknown, WsmanException {
		m_log.debug("polling() " +
				"ipAddress = " + ipAddress +
				",user = " + user +
				",userPassword = " + userPassword +
				",port = " + port +
				",protocol = " + protocol +
				",timeout = " + timeout +
				",retries = " + retries);


		// XMLのTransformerFactoryの設定
		m_log.debug("polling() javax.xml.transform.TransformerFactory = " + System.getProperty("javax.xml.transform.TransformerFactory"));
		System.setProperty("javax.xml.transform.TransformerFactory", "com.sun.org.apache.xalan.internal.xsltc.trax.TransformerFactoryImpl");

		// URLの作成
		try{
			InetAddress address = InetAddress.getByName(ipAddress);
			if (address instanceof Inet6Address){
				m_url = protocol + "://[" + ipAddress + "]:" + port + "/wsman";
			}
			else{
				m_url = protocol + "://" + ipAddress + ":" + port + "/wsman";
			}
		} catch (UnknownHostException e) {
			m_log.warn("ipAddress is not valid : " + ipAddress);
			throw new HinemosUnknown("ipAddress is not valid : " + ipAddress);
		}
		m_log.debug("polling() url = " + m_url);

		// コネクションと認証の設定
		m_con = WsmanConnection.createConnection(m_url);
		m_con.setAuthenticationScheme("basic");
		m_con.setUsername(user);
		m_con.setUserpassword(userPassword);
		m_con.setTimeout(timeout);

		// HTTP監視で使用しているライブラリ common-httpclient の HostnameVerifier を使用する
		m_con.setHostnameVerifier(SSLSocketFactory.STRICT_HOSTNAME_VERIFIER);

		// URIの設定
		ManagedReference ref = m_con.newReference(URI_WIN32_SERVICE);
		ref.addSelector("Name", m_serviceName);

		// 判定
		int count = 0;
		WsmanException lastException = null;
		while (count < retries) {
			try{
				// 接続
				ManagedInstance inst = ref.get();
				if(m_log.isDebugEnabled()){
					m_log.debug(WsmanUtils.getXML(inst));
				}

				// 状態の取得
				Object stateObj = inst.getProperty("State");
				if(stateObj != null){
					m_state = stateObj.toString();
				}else{
					count++;
					continue;
				}

				// 状態の判定
				if(STATE_RUNNING.equalsIgnoreCase(m_state)){

					// [OK]
					m_message = m_serviceName + " Service is " + STATE_RUNNING;
					m_messageOrg = m_serviceName + " Service is " + STATE_RUNNING;
					m_date = System.currentTimeMillis();

					break;
				}
				else{
					// [NG]
					m_message = m_serviceName + " Service is not " + STATE_RUNNING;
					m_messageOrg = m_serviceName + " Service is another state : " + m_state;
					m_date = System.currentTimeMillis();

					return false;
				}

			} catch (WsmanException e) {
				m_log.warn("polling()" + count + " " + e.getMessage() + ", " + e.getReason());
				m_log.debug("polling() ", e);

				lastException = e; // 最後の例外を返却
				count++;
				continue;

			} finally {

				if(m_con != null){
					m_con = null;
				}
			}
		}

		// リトライ全部失敗の場合はNG
		if(count == retries){

			// 不明
			m_message = "WinRM Access Error . ";
			m_messageOrg = "WinRM Access Error . ";
			if(lastException != null){
				m_messageOrg = m_messageOrg + " : " + lastException.getMessage();
			}
			m_date = System.currentTimeMillis();

			if(lastException != null){
				m_log.warn("winservice message=" + lastException.getMessage() +
						", reason=" + lastException.getReason());
				if(lastException.getMessage() == null){
					throw new HinemosUnknown(Messages.getString("message.winservice.3") + " : " + lastException.getReason());
				}else{
					if(lastException.getMessage().indexOf("HTTP response code: 401") != -1){
						throw new HinemosUnknown(Messages.getString("message.winservice.2"));
					}
				}
				throw lastException;
			}
			else{
				// ここには到達しないはず。
				throw new HinemosUnknown("winservice unknown");
			}
		}

		// [OK]の場合のみ到達
		return true;
	}

	/**
	 * @return メッセージを戻します。
	 */
	public String getMessage() {
		m_log.debug("getMessage() message = " + m_message);
		return m_message;
	}

	/**
	 * 
	 * @return
	 */
	public String getMessageOrg() {
		m_log.debug("getMessageOrg() messageOrg = " + m_messageOrg);
		return m_messageOrg;
	}

	/**
	 * @return 取得日時を戻します。
	 */
	public long getDate() {
		m_log.debug("getDate() date = " + new Date(m_date));
		return m_date;
	}


	/**
	 * 
	 * @param args
	 */
	public static void main(String[] args) {
		try{
			RequestWinRM winRM = new RequestWinRM("SNMP");
			winRM.polling(
					"172.26.98.119",
					"Administrator",
					"Hinemos24",
					5985,
					"http",
					3000,
					5);

			System.out.println("MSG = " + winRM.getMessage());
			System.out.println("MSG_ORG = " + winRM.getMessageOrg());

			System.out.println(System.getProperty("javax.xml.transform.TransformerFactory"));

		} catch (Exception e) {
			System.out.println(e.getMessage());
			e.printStackTrace();
		}
	}
}
