/*

Copyright (C) 2011 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.

 */

package com.clustercontrol.systemlog.service.queue;

import java.net.InetAddress;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.CopyOnWriteArrayList;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

public class Channel {
	private static Log _log = LogFactory.getLog(Channel.class);

	private final ConcurrentHashMap<InetAddress, NodeQueue> _queueMap;
	private final CopyOnWriteArrayList<NodeQueue> _queueList;

	private final Object _lock = new Object();

	private int _lastQueueIndex;  // 最後に参照したQueueのインデックス保持

	public Channel(){
		_queueMap = new ConcurrentHashMap<InetAddress, NodeQueue>();
		_queueList = new CopyOnWriteArrayList<NodeQueue>();
		_lastQueueIndex = -1;
	}

	/**
	 * Queueを新規作成する
	 * 管理が煩雑となるため、Queueを削除することはない
	 * @param facilityId
	 */
	private void putQueue(InetAddress ipAddress){
		synchronized (_lock) {
			if(_queueMap.contains(ipAddress)){
				// 既にある
			} else {
				NodeQueue queue = new NodeQueue(ipAddress);

				_queueMap.put(ipAddress, queue);
				_queueList.add(queue);
			}
		}
	}

	/**
	 * メッセージを格納する
	 * @param ipAddress IPアドレス
	 * @param message メッセージ
	 */
	public void addMessage(InetAddress ipAddress, byte[] buffer){
		if(_log.isDebugEnabled()){
			_log.debug("add message : ipAddress = " + ipAddress + ", message = " + buffer.length + "byte");
		}

		NodeQueue queue = _queueMap.get(ipAddress);

		if(queue == null){
			// 初めてのIPアドレスの場合はQueueを新規作成する
			putQueue(ipAddress);
			queue = _queueMap.get(ipAddress);
		}

		queue.add(buffer);
	}

	/**
	 * ファシリティ毎に用意されたキューを順次読み最初に見つかったメッセージを返す
	 * 全部のキューを走査してもメッセージがない場合は、nullを返す
	 * @return 見つかったメッセージ（1件も存在しない場合はnull）
	 */
	protected byte[] getNextMessage(){
		int currentQueueIndex = _lastQueueIndex;

		for(int i = 0; i < _queueList.size(); i++){
			currentQueueIndex++;

			// 最後のインデックスに到達した場合はループさせる
			if(currentQueueIndex >= _queueList.size()){
				currentQueueIndex = 0;
			}

			// キューのサイズが0でなければ値を取得して返す
			// isEmpty()で実装すべき
			// size()メソッドは処理時間が大幅に増える
			//			if(_queueList.get(currentQueueIndex).size() != 0){
			if(!_queueList.get(currentQueueIndex).isEmpty()){
				_lastQueueIndex = currentQueueIndex;
				return _queueList.get(currentQueueIndex).poll();
			}
		}

		return null;
	}
}
