/*

Copyright (C) 2010 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.

 */

package com.clustercontrol.selfcheck.util;

import java.util.ArrayList;
import java.util.List;
import java.util.Set;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.poller.bean.PollerProtocolConstant;
import com.clustercontrol.poller.impl.SnmpPollerImpl;
import com.clustercontrol.sharedtable.DataTable;
import com.clustercontrol.sharedtable.TableEntry;

/**
 * ファイルシステムの利用状況を確認する処理の実装クラス
 */
public class FileSystemPoller {

	private static Log m_log = LogFactory.getLog( FileSystemPoller.class );

	private String mountPoint = null;

	private int snmpPort = 161;
	private String snmpVersion = "2c";
	private String snmpCommunity = "public";
	private int snmpRetries = 0;
	private int snmpTimeout = 3000;

	private static final String SNMP_POLLING_IPADDRESS = "127.0.0.1";
	private static final String POLLING_TARGET_OID = ".1.3.6.1.2.1.25.2.3.1";
	private static final String STORAGE_DESCR_OID = ".1.3.6.1.2.1.25.2.3.1.3";
	private static final String STORAGE_AllOCATION_UNIT_OID = ".1.3.6.1.2.1.25.2.3.1.4";
	private static final String STORAGE_SIZE_OID = ".1.3.6.1.2.1.25.2.3.1.5";
	private static final String STORAGE_USED_OID = ".1.3.6.1.2.1.25.2.3.1.6";

	/**
	 * コンストラクタ
	 * @param mountPoint 対象となるファイルシステムのマウントポイント
	 * @param fileSystemMaxUsage ファイルシステム使用率の上限値
	 * @param snmpPort SNMPポート番号
	 * @param snmpVersion SNMPバージョン
	 * @param snmpCommunity SNMPコミュニティ名
	 * @param snmpRetries SNMPリトライ回数
	 * @param snmpTimeout SNMPタイムアウト[msec]
	 */
	public FileSystemPoller(String mountPoint, int snmpPort, String snmpVersion, String snmpCommunity, int snmpRetries, int snmpTimeout) {
		this.mountPoint = mountPoint;
		this.snmpPort = snmpPort;
		this.snmpVersion = snmpVersion;
		this.snmpCommunity = snmpCommunity;
		this.snmpRetries = snmpRetries;
		this.snmpTimeout = snmpTimeout;
	}

	/**
	 * ファイルシステムの全体サイズを取得します（MB単位）。
	 */
	public int getFileSystemTotal(){
		return getFileSystemSize(STORAGE_SIZE_OID);
	}

	/**
	 * ファイルシステムの使用サイズを取得します（MB単位）。
	 */
	public int getFileSystemUsage(){
		return getFileSystemSize(STORAGE_USED_OID);
	}

	// 指定のファイルシステムの使用サイズを取得します（MB単位）。
	private int getFileSystemSize(String oid){
		/** ローカル変数 */
		int version = 0;

		SnmpPollerImpl poller = null;
		List<String> oidList = null;
		DataTable dataTable = null;
		Set<TableEntry> mibValues = null;

		String storageSizeOid = null;
		String allocationUnitsOid = null;
		String key = null;
		int index = 0;
		long size = 0;
		long sizeMByte = 0;
		long unit = 0;

		/** メイン処理 */
		if (m_log.isDebugEnabled()) m_log.debug("start snmp polling for getting usage of file system. (mountPoint = " + mountPoint + ")");

		// SNMPのバージョン変換
		if("1".equals(snmpVersion)){
			version = 0;
		} else if("2c".equals(snmpVersion)){
			version = 1;
		} else {
			// それ以外の場合はv2cに設定
			version = 1;
		}

		try {
			// 収集対象のOID
			oidList = new ArrayList<String>();
			oidList.add(POLLING_TARGET_OID);

			// ポーラを生成してポーリングを実行
			poller = new SnmpPollerImpl();
			dataTable = poller.polling(
					SNMP_POLLING_IPADDRESS,
					snmpPort,
					version,
					snmpCommunity,
					snmpRetries,
					snmpTimeout,
					oidList,
					false);

			// 収集されたMIB値の中から「HOST-RESOURCES-MIB::hrStorageDescr」のものを抽出
			mibValues = dataTable.getValueSetStartWith(getEntryKey(STORAGE_DESCR_OID));

			// ターゲットのファイルシステムの使用量を取得できるOIDを特定
			for (TableEntry entry : mibValues) {
				// 取得したMIBのリストの中からマウントポイントに一致するもの取得する
				if ( mountPoint.equals(entry.getValue().toString()) ) {
					key = entry.getKey();
					index = Integer.parseInt(key.substring(key.lastIndexOf('.') + 1));

					storageSizeOid = oid + "." + index;
					allocationUnitsOid = STORAGE_AllOCATION_UNIT_OID + "." + index;
				}
			}

			if (storageSizeOid != null && allocationUnitsOid != null) {
				TableEntry sizeEntry = dataTable.getValue(getEntryKey(storageSizeOid));
				size = (Long)sizeEntry.getValue();

				TableEntry unitEntry = dataTable.getValue(getEntryKey(allocationUnitsOid));
				unit = (Long)unitEntry.getValue();

				// Byte単位をMByte単位に変換する
				sizeMByte = (size * unit) / (1024 * 1024);

				if (m_log.isDebugEnabled()) m_log.debug("successful of snmp polling for getting usage of file system. (mountPoint = " + mountPoint + ")");
				return (int)sizeMByte;
			}

		} catch (Exception e) {
			if (m_log.isWarnEnabled()) m_log.warn("failure in getting a size of a file system with snmp polling.", e);
		}

		// ポーリング結果を取得できなかった場合
		return -1;
	}

	/**
	 * OIDをTableEntryのキーに変換する
	 */
	private String getEntryKey(String oidString){
		return PollerProtocolConstant.PROTOCOL_SNMP + "." + oidString;
	}

	/**
	 * 単体試験用
	 */
	public static void main(String[] args) {
		String mountPoint = "/";

		FileSystemPoller poller = new FileSystemPoller(mountPoint, 161, "2c", "public", 1, 3000);

		int total = poller.getFileSystemTotal();
		int usage = poller.getFileSystemUsage();

		System.out.println("File System Usage : " + usage + ", File System Total : " + total);
	}
}
