/*

Copyright (C) 2010 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.

 */

package com.clustercontrol.selfcheck.monitor;

import java.util.Date;

import javax.management.ObjectName;
import javax.naming.InitialContext;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.jboss.jmx.adaptor.rmi.RMIAdaptor;

import com.clustercontrol.bean.PriorityConstant;
import com.clustercontrol.notify.bean.OutputBasicInfo;
import com.clustercontrol.util.Messages;

/**
 * メッセージングキューの蓄積量を確認する処理の実装クラス
 */
public class JMSQueueMonitor extends SelfCheckMonitorBase {

	private static Log m_log = LogFactory.getLog( JMSQueueMonitor.class );

	private static final String LOOKUP_NAME = "jmx/invoker/RMIAdaptor";

	private static final String OBJECT_NAME_PREFIX1 = "jboss.messaging.destination:service=";
	private static final String OBJECT_NAME_PREFIX2 = ",name=";

	private String type = null;
	private String objectShortName = null;
	private String objectName = null;
	private String attribute = null;
	private int threshold = 0;

	private static final String monitorIdPrefix = "SYS_JMS";
	private String monitorId = "";
	private String application = "SELFCHECK (Java Messaging Service)";

	/**
	 * コンストラクタ
	 * @param type QUEUE あるいは TOPIC
	 * @param objectShortName メッセージングキュー/トピック名（clustercontrolNotifyStatusなど）
	 */
	public JMSQueueMonitor(String type, String objectShortName, String attribute, int threshold) {
		this.type = type;
		this.objectShortName = objectShortName;
		this.objectName = OBJECT_NAME_PREFIX1 + this.type + OBJECT_NAME_PREFIX2 + objectShortName;
		this.attribute = attribute;
		this.threshold = threshold;

		this.monitorId = monitorIdPrefix + "_" + objectShortName;
	}

	/**
	 * セルフチェック処理名
	 */
	@Override
	public String toString() {
		return "monitoring jms object - " + objectShortName;
	}

	/**
	 * 監視項目ID
	 */
	@Override
	public String getMonitorId() {
		return monitorId;
	}

	/**
	 * メッセージングキューへの疎通確認
	 * @return 通知情報（アプリケーション名は未格納）
	 */
	@Override
	public OutputBasicInfo execute() {
		/** ローカル変数 */
		InitialContext initCtx = null;
		RMIAdaptor rmiAdapter = null;

		OutputBasicInfo notifyInfo = null;

		Integer count = 0;
		boolean avaiable = false;

		/** メイン処理 */
		if (m_log.isDebugEnabled())
			m_log.debug("monitoring jms. (objectName = " + objectShortName + ", attribute = " + attribute + ", threshold=" + threshold + ")");
		try {
			// JMSのオブジェクトをJNDI経由で取得し、属性値の値を取得できるかを確認する
			initCtx = new InitialContext();
			rmiAdapter = (RMIAdaptor)initCtx.lookup(LOOKUP_NAME);
			count = (Integer)rmiAdapter.getAttribute(new ObjectName(objectName), attribute);

			if (count != null) {
				avaiable = true;
			}
		} catch (Exception e) {
			if (m_log.isWarnEnabled())
				m_log.warn("failure to access a jms object. (objectName = " + objectShortName + ", attribute = " + attribute + ")", e);
			avaiable = false;
		}

		if (avaiable) {
			if (count <= threshold) {
				if (m_log.isDebugEnabled())
					m_log.debug("size of jms message is low. (objectName = " + objectShortName + ", attribute = " + attribute + ", count = " + count + ", threshold = " + threshold + ")");

				// set result, but do not notify (priority == info)
				notifyInfo = new OutputBasicInfo();
				notifyInfo.setPluginId(PLUGIN_ID);
				notifyInfo.setMonitorId(monitorId);
				notifyInfo.setPriority(PriorityConstant.TYPE_INFO);
			} else {
				m_log.warn("size of jms message is high. (objectName = " + objectShortName + ", attribute = " + attribute + ", count = " + count + ", threshold = " + threshold + ")");
				notifyInfo = new OutputBasicInfo();

				String[] msgAttr1 = { type, objectShortName, attribute, new Integer(count).toString(), new Integer(threshold).toString() };
				String[] msgAttr2 = { type, objectShortName, attribute, new Integer(count).toString(), new Integer(threshold).toString() };

				notifyInfo.setPluginId(PLUGIN_ID);
				notifyInfo.setMonitorId(monitorId);
				notifyInfo.setApplication(application);
				notifyInfo.setMessageId("001");
				notifyInfo.setMessage(Messages.getString("message.selfcheck.notify.jms.failure.msg", msgAttr1));
				notifyInfo.setMessageOrg(Messages.getString("message.selfcheck.notify.jms.failure.origmsg", msgAttr2));
				notifyInfo.setFacilityId(FACILITY_ID);
				notifyInfo.setScopeText(FACILITY_TEXT);
				notifyInfo.setPriority(PriorityConstant.TYPE_WARNING);
				notifyInfo.setGenerationDate(new Date().getTime());
			}
		} else {
			if (m_log.isInfoEnabled())
				m_log.info("skipped monitoring jms. (objectName = " + objectShortName + ", attribute = " + attribute + ", threshold=" + threshold + ")");
		}

		return notifyInfo;
	}

}
