/*

Copyright (C) 2010 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.

 */

package com.clustercontrol.selfcheck.monitor;

import java.util.Date;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.bean.PriorityConstant;
import com.clustercontrol.notify.bean.OutputBasicInfo;
import com.clustercontrol.selfcheck.util.FileSystemPoller;
import com.clustercontrol.util.Messages;

/**
 * ファイルシステムの利用状況を確認する処理の実装クラス
 */
public class FileSystemMonitor extends SelfCheckMonitorBase {

	private static Log m_log = LogFactory.getLog( FileSystemMonitor.class );

	private String mountPoint = null;
	private int thresholdPer = 0;

	private static final String monitorIdPrefix = "SYS_FS";
	private String monitorId = "";
	private String application = "SELFCHECK (FileSystem)";

	private int snmpPort = 161;
	private String snmpVersion = "2c";
	private String snmpCommunity = "public";
	private int snmpRetries = 0;
	private int snmpTimeout = 3000;

	/**
	 * コンストラクタ
	 * @param mountPoint 対象となるファイルシステムのマウントポイント
	 * @param fileSystemUsagePer ファイルシステム使用率の上限値
	 * @param snmpPort SNMPポート番号
	 * @param snmpVersion SNMPバージョン
	 * @param snmpCommunity SNMPコミュニティ名
	 * @param snmpRetries SNMPリトライ回数
	 * @param snmpTimeout SNMPタイムアウト[msec]
	 */
	public FileSystemMonitor(String mountPoint, int fileSystemUsagePer, int snmpPort, String snmpVersion, String snmpCommunity, int snmpRetries, int snmpTimeout) {
		/** ローカル変数 */
		String monitorIdPostfix = "";

		/** メイン処理 */
		this.mountPoint = mountPoint;
		this.thresholdPer = fileSystemUsagePer;

		this.snmpPort = snmpPort;
		this.snmpVersion = snmpVersion;
		this.snmpCommunity = snmpCommunity;
		this.snmpRetries = snmpRetries;
		this.snmpTimeout = snmpTimeout;

		monitorIdPostfix = mountPoint.replace('/', '_');
		this.monitorId = monitorIdPrefix + monitorIdPostfix;
	}

	/**
	 * セルフチェック処理名
	 */
	@Override
	public String toString() {
		return "monitoring a usage of a file system - " + mountPoint;
	}

	/**
	 * 監視項目ID
	 */
	@Override
	public String getMonitorId() {
		return monitorId;
	}

	/**
	 * ファイルシステムの使用率が上限値以下であるかを確認する処理
	 * @return 通知情報（アプリケーション名は未格納）
	 */
	@Override
	public OutputBasicInfo execute() {
		/** ローカル変数 */
		int fileSystemUsage = 0;
		int fileSystemTotal = 0;
		double fileSystemUsagePer = 0;

		OutputBasicInfo notifyInfo = null;

		/** メイン処理 */
		if (m_log.isDebugEnabled()) m_log.debug("monitoring file system usage. (mountPoint = " + mountPoint + ")");

		// 利用可能なヒープ容量をMByte単位で取得する
		try {
			fileSystemUsage = new FileSystemPoller(mountPoint, snmpPort, snmpVersion, snmpCommunity, snmpRetries, snmpTimeout).getFileSystemUsage();
			fileSystemTotal = new FileSystemPoller(mountPoint, snmpPort, snmpVersion, snmpCommunity, snmpRetries, snmpTimeout).getFileSystemTotal();
		} catch (Exception e) {
			m_log.warn("filesystem usage collection failure. (mountPoint = " + mountPoint + ", threshold = " + thresholdPer + " [%])", e);
		}

		fileSystemUsagePer = (double)fileSystemUsage / (double)fileSystemTotal * 100.0;

		if (fileSystemUsage == -1 || fileSystemTotal == -1) {
			if (m_log.isInfoEnabled()) m_log.info("skipped monitoring file system usage. (mountPoint = " + mountPoint + ")");
		} else {
			if (fileSystemUsagePer <= thresholdPer) {
				if (m_log.isDebugEnabled())
					m_log.debug("usage of file system is low. (mountPoint = " + mountPoint
							+ ", usage = " + String.format("%.2f", fileSystemUsagePer) + " [%], threshold = " + thresholdPer + " [%])");

				// set result, but do not notify (priority == info)
				notifyInfo = new OutputBasicInfo();
				notifyInfo.setPluginId(PLUGIN_ID);
				notifyInfo.setMonitorId(monitorId);
				notifyInfo.setPriority(PriorityConstant.TYPE_INFO);
			} else {
				m_log.warn("usage of file system is high. (mountPoint = " + mountPoint
						+ ", usage = " + String.format("%.2f", fileSystemUsagePer) + " [%], threshold = " + thresholdPer + " [%])");
				notifyInfo = new OutputBasicInfo();

				String[] msgAttr1 = { mountPoint, String.format("%.2f", fileSystemUsagePer), new Integer(thresholdPer).toString() };
				String[] msgAttr2 = { mountPoint, String.format("%.2f", fileSystemUsagePer), new Integer(thresholdPer).toString() };

				notifyInfo.setPluginId(PLUGIN_ID);
				notifyInfo.setMonitorId(monitorId);
				notifyInfo.setApplication(application);
				notifyInfo.setMessageId("001");
				notifyInfo.setMessage(Messages.getString("message.selfcheck.notify.filesystem.failure.msg", msgAttr1));
				notifyInfo.setMessageOrg(Messages.getString("message.selfcheck.notify.filesystem.failure.origmsg", msgAttr2));
				notifyInfo.setFacilityId(FACILITY_ID);
				notifyInfo.setScopeText(FACILITY_TEXT);
				notifyInfo.setPriority(PriorityConstant.TYPE_WARNING);
				notifyInfo.setGenerationDate(new Date().getTime());
			}
		}

		return notifyInfo;
	}

}
