/*

Copyright (C) 2006 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.

 */

package com.clustercontrol.notify.util;

import java.sql.Timestamp;

import javax.ejb.CreateException;
import javax.ejb.EJBException;
import javax.ejb.FinderException;
import javax.naming.NamingException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.monitor.bean.StatusExpirationConstant;
import com.clustercontrol.notify.bean.NotifyRequestMessage;
import com.clustercontrol.notify.bean.OutputBasicInfo;
import com.clustercontrol.notify.ejb.entity.NotifyStatusInfoLocal;
import com.clustercontrol.notify.ejb.entity.NotifyStatusInfoPK;
import com.clustercontrol.notify.ejb.entity.NotifyStatusInfoUtil;
import com.clustercontrol.notify.monitor.ejb.entity.StatusInfoLocal;
import com.clustercontrol.notify.monitor.ejb.entity.StatusInfoPK;
import com.clustercontrol.notify.monitor.ejb.entity.StatusInfoUtil;
import com.clustercontrol.notify.util.DependDbNotifier;

/**
 * ステータス情報を更新するクラス<BR>
 *
 * @version 3.2.0
 * @since 3.0.0
 */
public class OutputStatus implements DependDbNotifier {
	/** ログ出力のインスタンス */
	private static Log m_log = LogFactory.getLog(OutputStatus.class);

	// 無期限収集の場合に設定される終了予定時刻（9999.12.31 23:59:59.000 に相当）
	private final long EXPIRATION_DATE_MAX = 253402268399000l;

	@Override
	public void notify(NotifyRequestMessage message) throws Exception {
		updateStatus(message.getOutputInfo(), message.getNotifyId());
	}

	/**
	 * ステータス情報を更新します。
	 * 更新対象のステータスが存在しない場合は新規に生成します。
	 * 
	 * @param outputInfo 通知対象情報
	 * @param notifyId 通知ID
	 */
	private void updateStatus(OutputBasicInfo outputInfo, String notifyId) {
		long outputDateTime = System.currentTimeMillis();

		NotifyStatusInfoLocal notifyStatusInfo;
		try {
			notifyStatusInfo = NotifyStatusInfoUtil.getLocalHome().findByPrimaryKey(
					new NotifyStatusInfoPK(notifyId, outputInfo.getPriority()));

			// 有効期限制御フラグを設定（わかりずらい仕様のため要注意）
			// cc_status_info の expirationFlg は、有効期限制御を行うか否かの2値ではなく、
			// 行う場合はどの重要度に置き換えるのかの情報も合わせて管理する。
			Integer expirationFlg = null;
			if (StatusExpirationConstant.TYPE_DELETE == notifyStatusInfo.getStatusInvalidFlg()) {
				expirationFlg = new Integer(notifyStatusInfo.getStatusInvalidFlg());
			} else if (StatusExpirationConstant.TYPE_UPDATE == notifyStatusInfo.getStatusInvalidFlg()) {
				// FIXME
				// 有効期間経過後の扱いが、更新されていない旨のメッセージに置き換える場合は、
				// 置換え後の重要度を設定する
				expirationFlg = new Integer(notifyStatusInfo.getStatusUpdatePriority());
			}

			// 有効期限日時を設定
			long expirationDateTime = EXPIRATION_DATE_MAX;


			if (expirationFlg != null) {
				if (notifyStatusInfo.getStatusValidPeriod() > 0) {
					// StatusValidPeriod は分単位
					expirationDateTime = outputDateTime + notifyStatusInfo.getStatusValidPeriod() * 60 * 1000l;
				}
			}

			this.outputStatusInfo(outputInfo, expirationFlg, expirationDateTime, outputDateTime);
		} catch (FinderException e) {
			m_log.error(e.getMessage(), e);
		} catch (NamingException e) {
			m_log.error(e.getMessage(), e);
		}
	}

	/**
	 * ステータス情報を出力します。<BR>
	 * 同じステータス情報が存在しない場合は、ステータス情報を作成します。
	 * 同じステータス情報がすでに存在する場合は、ステータス情報を更新します。
	 * 
	 * @param logOutput ログ出力情報
	 * @param outputDate 受信日時
	 */
	private void outputStatusInfo(OutputBasicInfo outputInfo, int expirationFlg, long expirationDateTime, long outputDateTime) {
		StatusInfoLocal outputStatus = null;

		try {
			// エンティティ情報の検索
			StatusInfoPK pk = new StatusInfoPK(
					outputInfo.getMonitorId(),
					outputInfo.getSubKey(),
					outputInfo.getPluginId(),
					outputInfo.getFacilityId());
			outputStatus = StatusInfoUtil.getLocalHome().findByPrimaryKey(pk);

			// ステータス情報の更新
			try {
				outputStatus.setApplication(outputInfo.getApplication());
				outputStatus.setMessageId(outputInfo.getMessageId());
				outputStatus.setMessage(outputInfo.getMessage());

				// 重要度が変更されていた場合、出力日時を更新する
				if (outputStatus.getPriority().intValue() != outputInfo.getPriority()) {
					outputStatus.setGenerationDate(new java.sql.Timestamp(outputInfo.getGenerationDate()));
				}

				outputStatus.setPriority(new Integer(outputInfo.getPriority()));
				outputStatus.setOutputDate(new Timestamp(outputDateTime));
				outputStatus.setExpirationFlg(expirationFlg);
				outputStatus.setExpirationDate(new Timestamp(expirationDateTime));
			} catch (EJBException e) {
				// NOT NULL のカラムにnullを送信した場合など
				m_log.error(e.getMessage(), e);
			}
		} catch (NamingException e) {
			m_log.error(e.getMessage(), e);
		} catch (FinderException e) {
			// 検索条件に合致するエンティティが存在しないため新規に生成
			try {
				StatusInfoUtil.getLocalHome().create(
						outputInfo.getFacilityId(),
						outputInfo.getMonitorId(),
						outputInfo.getSubKey(),
						outputInfo.getPluginId(),
						outputInfo.getApplication(),
						outputInfo.getMessageId(),
						outputInfo.getMessage(),
						new Integer(outputInfo.getPriority()),
						new Timestamp(outputInfo.getGenerationDate()),
						new Timestamp(outputDateTime),
						expirationFlg,
						new Timestamp(expirationDateTime));
			} catch (NamingException e2) {
				m_log.error(e2.getMessage(), e2);
			} catch (CreateException e2) {
				m_log.error(e2.getMessage(), e2);
			}
		}
	}

	/**
	 * 通知失敗時の内部エラー通知を定義します
	 */
	@Override
	public void internalErrorNotify(String notifyId, String msgID, String detailMsg) throws Exception {
		//FIXME
		// 何もしない
	}
}
