/*

 Copyright (C) 2011 NTT DATA Corporation

 This program is free software; you can redistribute it and/or
 Modify it under the terms of the GNU General Public License
 as published by the Free Software Foundation, version 2.

 This program is distributed in the hope that it will be
 useful, but WITHOUT ANY WARRANTY; without even the implied
 warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
 PURPOSE.  See the GNU General Public License for more details.

 */


package com.clustercontrol.notify.util;

import javax.ejb.CreateException;
import javax.naming.NamingException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.bean.HinemosModuleConstant;
import com.clustercontrol.commons.util.HinemosProperties;
import com.clustercontrol.monitor.run.bean.MonitorInfo;
import com.clustercontrol.monitor.run.bean.MonitorStringValueInfo;
import com.clustercontrol.notify.bean.OutputBasicInfo;
import com.clustercontrol.notify.ejb.session.NotifyControllerLocal;
import com.clustercontrol.notify.ejb.session.NotifyControllerUtil;
import com.clustercontrol.repository.bean.FacilityTreeAttributeConstant;
import com.clustercontrol.repository.ejb.session.RepositoryControllerLocal;
import com.clustercontrol.repository.ejb.session.RepositoryControllerUtil;
import com.clustercontrol.systemlog.service.MessageInfo;

public class LogDirectNotifier implements LogNotifier {
	/** ログ出力のインスタンス  */
	private final static Log m_log = LogFactory.getLog(LogDirectNotifier.class);

	/** オリジナルメッセージ出力文字数 */
	public static final String _LOG_LINE_MAX_LENGTH_KEY = "log.line.max.length";
	public static final int _LOG_LINE_MAX_LENGTH_DEFAULT = 256;

	private final static int LOG_LINE_LENGTH;

	static {
		// intialize log line
		String logLineLengthStr = HinemosProperties.getProperty(_LOG_LINE_MAX_LENGTH_KEY);
		int logLineLength = _LOG_LINE_MAX_LENGTH_DEFAULT;
		try {
			if (logLineLengthStr != null) {
				logLineLength = Integer.parseInt(logLineLengthStr);
			}
		} catch (NumberFormatException e) {
			m_log.warn("invalid configuration. (" + _LOG_LINE_MAX_LENGTH_KEY + " = " + logLineLengthStr + ")");
		}
		LOG_LINE_LENGTH = logLineLength;
		m_log.info("NotifyDirectSender initialized : LOG_LINE_LENGTH = " + LOG_LINE_LENGTH);
	}

	@Override
	public void put(String multiId, String message, MessageInfo logmsg,
			MonitorInfo monitorInfo, MonitorStringValueInfo rule, String logFacilityId) {
		/*
		 * multiIdはメソッドの引数にすべきではない。
		 * 利用する直前にhinemos.propertiesから取得すべき。
		 */

		//マッチした場合
		// メッセージ作成（未登録ノードの時は、送信元ホスト名をスコープ名にする。）
		m_log.debug("call makeMessage.");
		OutputBasicInfo logOutput  = makeMessage(multiId, message, logmsg,
				monitorInfo, rule, logFacilityId, logmsg.getHostName());
		m_log.debug("called makeMessage.");

		//メッセージ送信処理
		try {
			NotifyControllerLocal m_notifyController = NotifyControllerUtil.getLocalHome().create();

			m_log.debug("Process:" + rule.getDescription() + ":" + message);
			m_notifyController.notify(logOutput, monitorInfo.getNotifyGroupId());
			m_log.debug("sended.");
		} catch (CreateException e) {
			m_log.error("initialize : " + e.getMessage(), e);
		} catch (NamingException e) {
			m_log.error("initialize : " + e.getMessage(), e);
		}
	}

	/**
	 * 引数で指定された情報より、ログ出力メッセージを生成し返します。
	 * 
	 * @param msg メッセージ
	 * @param logmsg syslogメッセージ情報
	 * @param moninfo 監視項目情報
	 * @param filterInfo フィルタ情報
	 * @param facilityID ファシリティID
	 * @param nodeName ノード名
	 * @return ログ出力メッセージ
	 * 
	 * @since 1.0.0
	 */
	public static OutputBasicInfo makeMessage(
			String multiId,
			String msg,
			MessageInfo logmsg,
			MonitorInfo monInfo,
			MonitorStringValueInfo filterInfo,
			String facilityId,
			String nodeName) {

		m_log.debug("Make ObjectMsg");

		OutputBasicInfo output = new OutputBasicInfo();

		output.setMonitorId(filterInfo.getMonitorId());
		output.setFacilityId(facilityId);

		if (HinemosModuleConstant.MONITOR_LOGFILE.equals(monInfo.getMonitorTypeId())) {
			output.setPluginId(HinemosModuleConstant.MONITOR_LOGFILE);
		} else if (HinemosModuleConstant.MONITOR_SYSTEMLOG.equals(monInfo.getMonitorTypeId())) {
			output.setPluginId(HinemosModuleConstant.MONITOR_SYSTEMLOG);
		} else {
			m_log.error("makeMessage " + monInfo.getMonitorTypeId());
		}

		// 通知抑制を監視項目の「パターンマッチ表現」単位にするため、通知抑制用サブキーを設定する。
		output.setSubKey(filterInfo.getPattern());

		if(FacilityTreeAttributeConstant.UNREGISTEREFD_SCOPE.equals(facilityId)){
			//未登録ノードの場合には送信元のホスト名を表示する。
			output.setScopeText(nodeName);
		}else{
			// ファシリティのパスを設定する]
			try {
				RepositoryControllerLocal repositoryCtrl = RepositoryControllerUtil.getLocalHome().create();
				m_log.debug("call getFacilityPath.");
				String facilityPath = repositoryCtrl.getFacilityPath(facilityId, null);
				m_log.debug("called getFacilityPath.");

				output.setScopeText(facilityPath);
			} catch (Exception e) {
				m_log.error("cannot get facility path.(facilityId = " + facilityId + ")");
			}
		}

		output.setApplication(monInfo.getApplication());
		output.setMessageId(filterInfo.getMessageId());

		// メッセージに#[LOG_LINE]のみが入力されている場合は、
		// オリジナルメッセージ(文字数上限：log_line_length)をメッセージに出力する
		if("#[LOG_LINE]".equals(filterInfo.getMessage())) {
			// オリジナルメッセージの文字数が上限を超えていない場合
			if (logmsg.getMessage().length() < LOG_LINE_LENGTH + 1) {
				output.setMessage(logmsg.getMessage());
			}
			// オリジナルメッセージの文字数が上限を超えている場合
			else {
				// 先頭からorg_msg_length分文字を出力させる
				output.setMessage(logmsg.getMessage().substring(0, LOG_LINE_LENGTH));
			}
		} else {
			output.setMessage(filterInfo.getMessage());
		}

		output.setMessageOrg(msg);
		output.setPriority(filterInfo.getPriority());
		output.setGenerationDate(logmsg.getGenerationDate());

		if(multiId != null && !"".equals(multiId)){
			output.setMultiId(multiId);
		}

		return output;
	}
}
