/*

 Copyright (C) 2009 NTT DATA Corporation

 This program is free software; you can redistribute it and/or
 Modify it under the terms of the GNU General Public License
 as published by the Free Software Foundation, version 2.

 This program is distributed in the hope that it will be
 useful, but WITHOUT ANY WARRANTY; without even the implied
 warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
 PURPOSE.  See the GNU General Public License for more details.

 */

package com.clustercontrol.notify.queue;

import java.util.ArrayList;
import java.util.concurrent.ConcurrentHashMap;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.commons.util.SendQueue;
import com.clustercontrol.notify.bean.NotifyRequestMessage;
import com.clustercontrol.notify.bean.QueueConstant;

/**
 * ログ出力情報（通知グループIDと通知IDとログ出力基本情報）を順次NotifyExecutorを利用して通知処理します。
 * Message-Driven Bean 実装を利用する場合の SendQueue に相当します。
 */
public class NotifyQueueSender {
	private static Log m_log = LogFactory.getLog( NotifyQueueSender.class );

	// 通知種別毎にインスタンスを保持
	// getInstance()呼び出しの度に新規にインスタンスを生成しなくてもよいように保持する
	private static volatile ConcurrentHashMap<Integer, NotifyQueueSender> _instanceMap = null;

	// _instanceMapインスタンスへの書き込みを制御するロック
	private static final Object _instanceMapWriteLock = new Object();

	// キュー名
	private final String _queueName;

	// 送信先のQueue
	private volatile SendQueue _queue = null;
	private static final Object queueLock = new Object();

	/**
	 * 通知種別とキュー名の対応マップを初期化します
	 * @throws NotifyQueueException
	 */
	private static void init() throws NotifyQueueException {
		synchronized (_instanceMapWriteLock) {
			if(_instanceMap != null){
				return;
			}

			// マップのインスタンスを生成する。一度生成すると再度生成されることはない。
			_instanceMap = new ConcurrentHashMap<Integer, NotifyQueueSender>();

			m_log.debug("init");
			// キュー名やJMS利用の設定を行う
			for (ArrayList<Object> queue : QueueConstant.getQueueList()) {
				// インスタンスを生成しマップに登録する
				// ここで登録されたインスタンスが使いまわされる
				_instanceMap.put((Integer)queue.get(0),
						new NotifyQueueSender((String)queue.get(1)));
			}
		}
	}
	/**
	 * インスタンスを生成します。
	 * @param queueName キュー名
	 * @throws NotifyQueueException
	 */
	public static NotifyQueueSender getInstance(int notifyType) throws NotifyQueueException{
		// 通知種別毎に生成され、マップに登録されているインスタンスを返す
		// マップがまだ生成されていない場合は、生成する
		synchronized (_instanceMapWriteLock) {
			if(_instanceMap == null){
				NotifyQueueSender.init();
			}
		}

		return _instanceMap.get(notifyType);
	}

	// このクラスのインスタンスは外部から生成できない
	private NotifyQueueSender(String queueName){
		_queueName = queueName;

		try {
			synchronized (queueLock) {
				_queue = new SendQueue(_queueName);
			}
		} catch (Exception e) {
			m_log.warn("queue sender initialize failure.", e);
		}
	}

	/**
	 * メッセージを NotifyExecutor の該当キューに挿入します
	 * @param info ログ出力情報（通知グループIDと通知IDとログ出力基本情報）
	 * @throws Exception
	 */
	public void send(NotifyRequestMessage info, Integer deliveryMode) throws NotifyQueueException {
		m_log.debug("send() start. " + "Queue=" + _queueName	+ " " + info);

		try {

			try{
				// JMSを利用してメッセージ送信（SendQueueはThreadsafeでないため、排他制御）
				synchronized (queueLock) {
					_queue.put(info, deliveryMode);
				}
			} catch (Exception e) {
				m_log.warn("send queue failure.", e);
				synchronized (queueLock) {
					_queue = new SendQueue(_queueName);
					_queue.put(info, deliveryMode);
				}
			}

		} catch (Exception e) {
			new NotifyQueueException("message send failed. " + info, e);
		}
		m_log.debug("send() end.   " + "Queue=" + _queueName	+ " " + info);
	}
}
