/*

 Copyright (C) 2008 NTT DATA Corporation

 This program is free software; you can redistribute it and/or
 Modify it under the terms of the GNU General Public License
 as published by the Free Software Foundation, version 2.

 This program is distributed in the hope that it will be
 useful, but WITHOUT ANY WARRANTY; without even the implied
 warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
 PURPOSE.  See the GNU General Public License for more details.

 */

package com.clustercontrol.notify.mail.ejb.session;

import java.rmi.RemoteException;
import java.security.Principal;
import java.util.ArrayList;
import java.util.HashMap;
import javax.ejb.CreateException;
import javax.ejb.EJBException;
import javax.ejb.SessionContext;

import com.clustercontrol.fault.HinemosUnknown;
import com.clustercontrol.fault.InvalidSetting;
import com.clustercontrol.fault.MailTemplateDuplicate;
import com.clustercontrol.fault.MailTemplateNotFound;
import com.clustercontrol.notify.mail.bean.MailTemplateInfo;
import com.clustercontrol.notify.mail.ejb.entity.MailTemplateInfoData;
import com.clustercontrol.notify.mail.factory.AddMailTemplate;
import com.clustercontrol.notify.mail.factory.DeleteMailTemplate;
import com.clustercontrol.notify.mail.factory.ModifyMailTemplate;
import com.clustercontrol.notify.mail.factory.SelectMailTemplate;
import com.clustercontrol.notify.util.NotifyValidator;

/**
 * メールテンプレート機能の管理を行う Session Bean <BR>
 * クライアントからの Entity Bean へのアクセスは、Session Bean を介して行います。
 *
 * @ejb.bean name="MailTemplateController"
 *           jndi-name="MailTemplateController"
 *           type="Stateless"
 *           transaction-type="Container"
 *           view-type="local"
 * 
 * @ejb.transaction type="Required"
 * 
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=MailTemplateInfoLocal"
 * 
 * @ejb.permission
 *     unchecked="true"
 *     method-intf="LocalHome"
 * 
 * @ejb.permission
 *     unchecked="true"
 *     method-intf="Local"
 */
public abstract class MailTemplateControllerBean implements javax.ejb.SessionBean {

	/** コンテキスト情報。 */
	@SuppressWarnings("unused")
	private SessionContext m_context;

	/**
	 * コンテキスト情報を設定します。<BR>
	 * Session Bean の生成時に行う処理を実装します。
	 * @see javax.ejb.SessionBean#setSessionContext(javax.ejb.SessionContext)
	 */
	@Override
	public void setSessionContext(SessionContext ctx) throws EJBException, RemoteException {
		m_context = ctx;
	}

	/**
	 * メールテンプレート情報をマネージャに登録します。<BR>
	 * 
	 * @ejb.interface-method
	 * 
	 * @param info 作成対象のメールテンプレート情報
	 * @return 作成に成功した場合、<code> true </code>
	 * @throws HinemosUnknown
	 * @throws MailTemplateDuplicate
	 * 
	 * @see com.clustercontrol.notify.factory.AddMailTemplate#add(MailTemplateInfoData)
	 */
	public boolean addMailTemplate(MailTemplateInfo data) throws HinemosUnknown, MailTemplateDuplicate,InvalidSetting {

		Principal pri = m_context.getCallerPrincipal();

		// メールテンプレート情報を登録
		boolean ret = false;

		try{
			//入力チェック
			NotifyValidator.validateMailTemplateInfo(data);

			AddMailTemplate mailTemplate = new AddMailTemplate();
			ret = mailTemplate.add(data, pri.getName());
		}catch(HinemosUnknown e){
			m_context.setRollbackOnly();
			throw e;
		} catch (MailTemplateDuplicate e) {
			m_context.setRollbackOnly();
			throw e;
		} catch (CreateException e) {
			m_context.setRollbackOnly();
			throw new HinemosUnknown(e.getMessage(), e);
		} catch (InvalidSetting e) {
			m_context.setRollbackOnly();
			throw e;
		}

		return ret;
	}

	/**
	 * マネージャ上のメールテンプレート情報を変更します。<BR>
	 * 
	 * @ejb.interface-method
	 * 
	 * @param info 変更対象のメールテンプレート情報
	 * @return 変更に成功した場合、<code> true </code>
	 * @throws HinemosUnknown
	 * @throws MailTemplateNotFound
	 * 
	 * @see com.clustercontrol.notify.factory.ModifyMailTemplate#modify(MailTemplateInfoData)
	 * @see com.clustercontrol.notify.bean.MailTemplateInfoData
	 */
	public boolean modifyMailTemplate(MailTemplateInfo data) throws HinemosUnknown, MailTemplateNotFound,InvalidSetting {

		Principal pri = m_context.getCallerPrincipal();

		// メールテンプレート情報を更新
		boolean ret = false;

		try{
			//入力チェック
			NotifyValidator.validateMailTemplateInfo(data);

			ModifyMailTemplate mailTemplate = new ModifyMailTemplate();
			ret = mailTemplate.modify(data, pri.getName());
		}catch(HinemosUnknown e){
			m_context.setRollbackOnly();
			throw e;
		} catch (MailTemplateNotFound e) {
			m_context.setRollbackOnly();
			throw e;
		} catch (InvalidSetting e) {
			m_context.setRollbackOnly();
			throw e;
		}

		return ret;
	}

	/**
	 * メールテンプレート情報をマネージャから削除します。<BR>
	 * 
	 * @ejb.interface-method
	 * 
	 * @param mailTemplateId 削除対象のメールテンプレートID
	 * @return 削除に成功した場合、<code> true </code>
	 * @throws HinemosUnknown
	 * 
	 * @see com.clustercontrol.notify.factory.DeleteMailTemplate#delete(String)
	 */
	public boolean deleteMailTemplate(String mailTemplateId) throws HinemosUnknown {

		// メールテンプレート情報を削除
		DeleteMailTemplate mailTemplate = new DeleteMailTemplate();

		boolean ret = false;

		try{
			ret = mailTemplate.delete(mailTemplateId);
		} catch(HinemosUnknown e){
			m_context.setRollbackOnly();
			throw e;
		}

		return ret;
	}

	/**
	 * 引数で指定されたメールテンプレート情報を返します。
	 * 
	 * @ejb.interface-method
	 * 
	 * @param mailTemplateId 取得対象のメールテンプレートID
	 * @return メールテンプレート情報
	 * @throws HinemosUnknown
	 * @throws MailTemplateNotFound
	 * 
	 * @see com.clustercontrol.notify.factory.SelectMailTemplate#getMailTemplateInfo(String)
	 */
	public MailTemplateInfo getMailTemplateInfo(String mailTemplateId) throws MailTemplateNotFound, HinemosUnknown {

		// メールテンプレート情報を取得
		SelectMailTemplate mailTemplate = new SelectMailTemplate();
		MailTemplateInfo info = mailTemplate.getMailTemplateInfo(mailTemplateId);

		return info;
	}

	/**
	 * メールテンプレートID一覧を取得します。<BR>
	 * 
	 * 戻り値のArrayListにはMailTemplateId(String)が順に
	 * 格納されています。
	 * 
	 * @ejb.interface-method
	 * 
	 * @jboss.method-attributes
	 *     read-only="true"
	 * 
	 * @return メールテンプレートID一覧
	 * @throws MailTemplateNotFound
	 * @throws HinemosUnknown
	 * 
	 * @see com.clustercontrol.notify.factory.SelectMailTemplate#getMailTemplateIdList()
	 */
	public ArrayList<String> getMailTemplateIdList() throws HinemosUnknown, MailTemplateNotFound {

		// メールテンプレートID一覧を取得
		SelectMailTemplate mailTemplate = new SelectMailTemplate();
		ArrayList<String> list = mailTemplate.getMailTemplateIdList();

		return list;
	}

	/**
	 * メールテンプレート情報一覧を取得します。<BR>
	 * 
	 * 戻り値はArrayListのArrayListであり以下のように格納される。
	 * 
	 * <Pre>
	 * ArrayList info = new ArrayList();
	 *  info.add(mailTemplate.getMailTemplateId());
	 *  info.add(mailTemplate.getDescription());
	 *	info.add(mailTemplate.getRegUser());
	 *	info.add(mailTemplate.getRegDate() == null ? null:new Date(mailTemplate.getRegDate().getTime()));
	 *	info.add(mailTemplate.getUpdateUser());
	 *	info.add(mailTemplate.getUpdateDate() == null ? null:new Date(mailTemplate.getUpdateDate().getTime()));
	 * 
	 *  list.add(info);
	 * </Pre>
	 * 
	 * 
	 * @ejb.interface-method
	 * 
	 * @return メールテンプレート情報一覧（Objectの2次元配列）
	 * @throws MailTemplateNotFound
	 * @throws HinemosUnknown
	 * 
	 * @see com.clustercontrol.notify.factory.SelectMailTemplate#getMailTemplateList()
	 */
	public ArrayList<MailTemplateInfo> getMailTemplateList() throws HinemosUnknown, MailTemplateNotFound {

		// メールテンプレート一覧を取得
		SelectMailTemplate mailTemplate = new SelectMailTemplate();
		ArrayList<MailTemplateInfo> list = mailTemplate.getMailTemplateList();

		return list;
	}

	/**
	 * メールテンプレート情報のマップを取得します。<BR>
	 * 
	 * @ejb.interface-method
	 * 
	 * @jboss.method-attributes
	 *     read-only="true"
	 * 
	 * @return メールテンプレート情報マップ
	 * @throws HinemosUnknown
	 * @throws MailTemplateNotFound
	 * 
	 * @see com.clustercontrol.notify.factory.SelectMailTemplate#getMailTemplateMap()
	 */
	public HashMap<String, MailTemplateInfo> getMailTemplateMap() throws MailTemplateNotFound, HinemosUnknown {

		// メールテンプレート一覧を取得
		SelectMailTemplate mailTemplate = new SelectMailTemplate();
		HashMap<String, MailTemplateInfo> map = mailTemplate.getMailTemplateMap();

		return map;
	}
}
