/*

Copyright (C) 2006 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.

 */

package com.clustercontrol.monitor.run.factory;

import java.rmi.RemoteException;
import java.util.Collection;
import java.util.Date;

import javax.ejb.CreateException;
import javax.ejb.EJBLocalHome;
import javax.ejb.FinderException;
import javax.naming.NamingException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.bean.ValidConstant;
import com.clustercontrol.commons.scheduler.TriggerConstant;
import com.clustercontrol.commons.scheduler.TriggerScheduler;
import com.clustercontrol.commons.scheduler.TriggerSchedulerException;
import com.clustercontrol.commons.scheduler.TriggerSchedulerFactory;
import com.clustercontrol.monitor.run.bean.QuartzConstant;
import com.clustercontrol.monitor.run.ejb.entity.MonitorInfoLocal;
import com.clustercontrol.monitor.run.ejb.entity.MonitorInfoUtil;
import com.clustercontrol.monitor.run.ejb.session.MonitorRunManagementUtil;

/**
 * スケジュールを登録するクラス<BR>
 *
 * @version 4.0.0
 * @since 2.0.0
 */
public class ModifySchedule {

	/** ログ出力のインスタンス。 */
	private static Log m_log = LogFactory.getLog( ModifySchedule.class );

	/**
	 * DBに登録されている全ての監視項目の設定をスケジューラに登録します。
	 * @throws TriggerSchedulerException
	 * 
	 * @since 4.0.0
	 */
	public void updateScheduleAll() throws TriggerSchedulerException{
		m_log.debug("updateScheduleAll()");

		Throwable exception = null;

		Collection<MonitorInfoLocal> beanList = null;
		try {
			beanList = MonitorInfoUtil.getLocalHome().findAll();
		} catch (FinderException e) {
			String msg = "updateScheduleAll() " + e.getClass().getSimpleName() + ", " + e.getMessage();
			m_log.warn(msg, e);
			exception = e;
		} catch (NamingException e) {
			String msg = "updateScheduleAll() " + e.getClass().getSimpleName() + ", " + e.getMessage();
			m_log.warn(msg, e);
			exception = e;
		}

		for (MonitorInfoLocal bean : beanList) {
			try {
				updateSchedule(bean);
			} catch (TriggerSchedulerException e) {
				m_log.warn("updateScheduleAll() scheduleJob : monitorId = " + bean.getMonitorId() + ", " + e.getClass().getSimpleName() + ", " + e.getMessage());
				// 次の設定を処理するため、throwはしない。
				exception = e;
			}
		}

		if(exception != null){
			throw new TriggerSchedulerException("An error occurred while scheduling the trigger.", exception);
		}
	}

	/**
	 * スケジューラに監視情報のジョブを登録します。
	 * <p>
	 * <ol>
	 * <li>監視対象IDと監視項目IDを指定し、監視を実行するメソッドのジョブを作成します。</li>
	 * <li>呼びだすメソッドの引数として、下記項目をセットします。</li>
	 * <dl>
	 *  <dt>引数</dt>
	 *  <dd>監視項目ID</dd>
	 * </dl>
	 * <li>スケジューラにジョブとトリガを登録します。</li>
	 * </ol>
	 * 
	 * @param monitorId 監視項目ID
	 * @throws TriggerSchedulerException
	 * @since 2.0.0
	 */
	protected void updateSchedule(String monitorId) throws TriggerSchedulerException {
		m_log.debug("updateSchedule() : monitorId=" + monitorId);

		MonitorInfoLocal bean = null;
		try {
			bean = MonitorInfoUtil.getLocalHome().findByPrimaryKey(monitorId);
		} catch (FinderException e) {
			String msg = "updateSchedule() scheduleJob : monitorId = " + monitorId + e.getClass().getSimpleName() + ", " + e.getMessage();
			m_log.warn(msg, e);
			throw new TriggerSchedulerException(msg, e);
		} catch (NamingException e) {
			String msg = "updateSchedule() scheduleJob : monitorId = " + monitorId + e.getClass().getSimpleName() + ", " + e.getMessage();
			m_log.warn(msg, e);
			throw new TriggerSchedulerException(msg, e);
		}

		updateSchedule(bean);
	}

	private void updateSchedule(MonitorInfoLocal bean) throws TriggerSchedulerException {
		String monitorId = bean.getMonitorId();
		String monitorTypeId = bean.getMonitorTypeId();

		//JobDetailに呼び出すメソッドの引数を設定
		// 監視対象IDを設定
		Object[] jdArgs = new Object[3];
		Class[] jdArgsType = new Class[3];
		jdArgs[0] = monitorTypeId;
		jdArgsType[0] = String.class;
		// 監視項目IDを設定
		jdArgs[1] = monitorId;
		jdArgsType[1] = String.class;
		// 監視判定タイプを設定
		jdArgs[2] = Integer.valueOf(bean.getMonitorType());
		jdArgsType[2] = Integer.class;

		if(TriggerScheduler.TRIGGER_TYPE_SIMPLE.equals(bean.getTriggerType())){
			int interval = bean.getRunInterval();

			try {
				m_log.debug("Schedule SimpleTrigger. monitorId = " + monitorId);
				// SimpleTrigger でジョブをスケジューリング登録
				// 監視も収集も無効の場合、登録後にポーズするようにスケジュール
				TriggerSchedulerFactory.getScheduler(TriggerConstant.MONITOR_INSTANCE_NAME).scheduleEjbLocalInvokerJobWithSimpleTrigger(
						monitorId,
						monitorTypeId,
						MonitorRunManagementUtil.getLocalHome(),
						QuartzConstant.METHOD_NAME,
						jdArgs,
						jdArgsType,
						null,
						true,
						ValidConstant.typeToBoolean(bean.getMonitorFlg()) || ValidConstant.typeToBoolean(bean.getCollectorFlg()),
						calcSimpleTriggerStartTime(interval, bean.getDelayTime()),
						bean.getRunInterval());
			} catch (TriggerSchedulerException e) {
				m_log.warn("updateSchedule() scheduleJob : monitorId = " + monitorId + ", " + e.getClass().getSimpleName() + ", " + e.getMessage());
				throw e;
			} catch (NamingException e) {
				String msg = "updateSchedule() scheduleJob : monitorId = " + monitorId + ", " + e.getClass().getSimpleName() + ", " + e.getMessage();
				m_log.warn(msg, e);
				throw new TriggerSchedulerException(msg, e);
			}
		} else if(TriggerScheduler.TRIGGER_TYPE_CRON.equals(bean.getTriggerType())) {
			try {
				m_log.debug("Schedule CronTrigger. monitorId = " + monitorId);
				// CronTrigger でジョブをスケジューリング登録
				// 監視も収集も無効の場合、登録後にポーズするようにスケジュール
				TriggerSchedulerFactory.getScheduler(TriggerConstant.MONITOR_INSTANCE_NAME).scheduleEjbLocalInvokerJobWithCronTrigger(
						monitorId,
						monitorTypeId,
						MonitorRunManagementUtil.getLocalHome(),
						QuartzConstant.METHOD_NAME,
						jdArgs,
						jdArgsType,
						null,
						true,
						ValidConstant.typeToBoolean(bean.getMonitorFlg()) || ValidConstant.typeToBoolean(bean.getCollectorFlg()),
						getCronString(bean.getRunInterval(), bean.getDelayTime()));
			} catch (TriggerSchedulerException e) {
				m_log.warn("updateSchedule() scheduleJob : monitorId = " + monitorId + ", " + e.getClass().getSimpleName() + ", " + e.getMessage());
				throw e;
			} catch (NamingException e) {
				String msg = "updateSchedule() scheduleJob : monitorId = " + monitorId + ", " + e.getClass().getSimpleName() + ", " + e.getMessage();
				m_log.warn(msg, e);
				throw new TriggerSchedulerException(msg, e);
			}
		} else if(TriggerScheduler.TRIGGER_TYPE_NONE.equals(bean.getTriggerType())) {
			// スケジュール登録しない
		} else {
			m_log.warn("Invalid TRIGGER_TYPE. monitorTypeId = " + monitorTypeId + ", + monitorId = " + monitorId);
		}
	}


	/**
	 * 引数で指定された監視情報をQuartzから削除します。
	 * 
	 * @param monitorTypeId 監視対象ID
	 * @param monitorId 監視項目ID
	 * @throws NamingException
	 * @throws SchedulerException
	 * @throws RemoteException
	 * @throws CreateException
	 * 
	 * @see com.clustercontrol.monitor.run.bean.QuartzConstant
	 * @see com.clustercontrol.commons.util.QuartzUtil#getQuartzManager()
	 * @see com.clustercontrol.quartzmanager.ejb.session.QuartzManager#deleteSchedule(java.lang.String, java.lang.String)
	 */
	protected void deleteSchedule(String monitorTypeId, String monitorId) throws TriggerSchedulerException {
		m_log.debug("deleteSchedule() : type =" + monitorTypeId + ", id=" + monitorId);

		try {
			//ジョブ削除
			TriggerSchedulerFactory.getScheduler(TriggerConstant.MONITOR_INSTANCE_NAME).deleteJob(monitorId, monitorTypeId);
		} catch (TriggerSchedulerException e) {
			m_log.warn("deleteSchedule() deleteJob : " + e.getClass().getSimpleName() + ", " + e.getMessage());
			throw e;
		}
	}

	/**
	 * Cron形式のスケージュール定義を返します。
	 * 
	 * @param schedule スケジュールカレンダ
	 * @return Cron形式スケジュール
	 */
	private String getCronString(int interval, int delayTime){
		String cronString = null;
		if(interval > 0 && interval < 3600){
			int minute = interval / 60;

			// Quartzのcron形式（例： 30 */1 * * * ? *）
			cronString = delayTime + " */" + minute + " * * * ? *";
		} else if(interval > 0 && interval >= 3600){
			int hour = interval / 3600;

			// Quartzのcron形式（例： 30 0 */1 * * ? *）
			cronString = delayTime + " 0 */" + hour + " * * ? *";
		}

		m_log.debug("getCronString() interval = " + interval + ", delayTime = " + delayTime + ", cronString = " + cronString);
		return cronString;
	}

	/**
	 * スケジュール開始時刻を求めます。
	 */
	private long calcSimpleTriggerStartTime(int interval, int delayTime){
		// 再起動時も常に同じタイミングでQuartzのTriggerが起動されるようにstartTimeを設定する
		long now = System.currentTimeMillis() + 1000l; // pauseFlag が trueの場合、起動させないように実行開始時刻を少し遅らせる。
		long intervalMilliSecond = interval * 1000;

		// 1) 現在時刻の直前で、監視間隔の倍数となる時刻を求める
		//   例）22:32:05 に 5分間間隔の設定を追加する場合は、22:35:00
		long roundout = (now / intervalMilliSecond + 1) * intervalMilliSecond;

		// 2) 1)の時刻にDelayTimeを秒数として足したものをstartTimeとして設定する
		long startTime = roundout + delayTime * 1000;

		// 3) もう一つ前の実行タイミングが（現在時刻+5秒）より後の場合は、そちらをstartTimeとする
		if((System.currentTimeMillis() + 5 * 1000l) < (startTime - intervalMilliSecond)){
			m_log.debug("reset time before : " + new Date(startTime));
			startTime = startTime - intervalMilliSecond;
			m_log.debug("reset time after : " + new Date(startTime));
		}

		return startTime;
	}
}
