/*

 Copyright (C) 2006 NTT DATA Corporation

 This program is free software; you can redistribute it and/or
 Modify it under the terms of the GNU General Public License
 as published by the Free Software Foundation, version 2.

 This program is distributed in the hope that it will be
 useful, but WITHOUT ANY WARRANTY; without even the implied
 warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
 PURPOSE.  See the GNU General Public License for more details.

 */

package com.clustercontrol.monitor.run.factory;

import java.util.Collection;

import javax.ejb.CreateException;
import javax.ejb.EJBException;
import javax.ejb.FinderException;
import javax.ejb.RemoveException;
import javax.naming.InitialContext;
import javax.naming.NamingException;
import javax.transaction.HeuristicMixedException;
import javax.transaction.HeuristicRollbackException;
import javax.transaction.InvalidTransactionException;
import javax.transaction.NotSupportedException;
import javax.transaction.RollbackException;
import javax.transaction.SystemException;
import javax.transaction.Transaction;
import javax.transaction.TransactionManager;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.fault.HinemosUnknown;
import com.clustercontrol.fault.MonitorNotFound;
import com.clustercontrol.fault.NotifyNotFound;
import com.clustercontrol.bean.HinemosModuleConstant;
import com.clustercontrol.commons.bean.SettingUpdateInfo;
import com.clustercontrol.commons.scheduler.TriggerSchedulerException;
import com.clustercontrol.commons.util.HinemosProperties;
import com.clustercontrol.custom.util.CustomManagerUtil;
import com.clustercontrol.monitor.run.ejb.entity.MonitorInfoLocal;
import com.clustercontrol.monitor.run.ejb.entity.MonitorInfoUtil;
import com.clustercontrol.notify.ejb.entity.MonitorStatusLocal;
import com.clustercontrol.notify.ejb.entity.MonitorStatusPK;
import com.clustercontrol.notify.ejb.entity.MonitorStatusUtil;
import com.clustercontrol.notify.ejb.entity.NotifyHistoryLocal;
import com.clustercontrol.notify.ejb.entity.NotifyHistoryUtil;
import com.clustercontrol.notify.ejb.session.NotifyControllerLocal;
import com.clustercontrol.notify.ejb.session.NotifyControllerUtil;
import com.clustercontrol.notify.util.MonitorResultStatusUpdater;
import com.clustercontrol.performance.dao.CalculatedDataDAO;
import com.clustercontrol.util.apllog.AplLogger;

/**
 * 監視情報を削除する抽象クラス<BR>
 * <p>
 * 監視種別（真偽値，数値，文字列）の各クラスで継承してください。
 *
 * @version 4.0.0
 * @since 2.0.0
 */
abstract public class DeleteMonitor {
	/** ログ出力のインスタンス。 */
	private static Log m_log = LogFactory.getLog( DeleteMonitor.class );

	/** 監視情報のローカルコンポーネント。 */
	protected MonitorInfoLocal m_monitor;

	/** 監視対象ID */
	protected String m_monitorTypeId;

	/** 監視項目ID */
	protected String m_monitorId;

	/** 監視設定削除時に性能データも共に削除するか否かのフラグ*/
	private static boolean deleteCascadeFlg = true;
	
	static {
		String deleteCascadeFlgStr = HinemosProperties.getProperty(
				"monitor.common.delete.cascade.perfdata", "on");
		if("off".equals(deleteCascadeFlgStr)){
			deleteCascadeFlg = false;
		} else{
			deleteCascadeFlg = true;
		}
		
		m_log.info("deleteCascadeFlg = " + deleteCascadeFlg);
	}
	
	/**
	 * トランザクションを開始し、監視情報を削除します。
	 * 
	 * @param monitorTypeId 監視対象ID
	 * @param monitorId 監視項目ID
	 * @return 削除に成功した場合、</code> true </code>
	 * @throws RemoveException
	 * @throws NamingException
	 * @throws NotSupportedException
	 * @throws HeuristicMixedException
	 * @throws HeuristicRollbackException
	 * @throws RollbackException
	 * @throws InvalidTransactionException
	 * @throws IllegalStateException
	 * @throws SystemException
	 * @throws CreateException
	 * @throws HinemosUnknown
	 * @throws NotifyNotFound
	 * @throws MonitorNotFound
	 * @throws TriggerSchedulerException
	 * @see #deleteMonitorInfo()
	 */
	public boolean delete(String monitorTypeId, String monitorId) throws RemoveException, NamingException, NotSupportedException, HeuristicMixedException, HeuristicRollbackException, RollbackException, InvalidTransactionException, IllegalStateException, SystemException, CreateException, FinderException, NotifyNotFound, HinemosUnknown, MonitorNotFound, TriggerSchedulerException{

		m_monitorTypeId = monitorTypeId;
		m_monitorId = monitorId;

		TransactionManager tm = null;
		Transaction oldTx = null;
		boolean result = false;

		try
		{
			// TransactionManagerを取得
			InitialContext context = new InitialContext();
			tm = (TransactionManager)context.lookup("java:/TransactionManager");

			// 実行中のスレッドがトランザクションに関連付けられている場合は、トランザクションを退避
			if(tm.getTransaction() != null){
				oldTx = tm.suspend();
			}

			try{
				// トランザクション開始
				if(tm.getTransaction() == null){
					tm.begin();
				}

				// 監視情報を削除
				result = deleteMonitorInfo();

				if(result){
					// コミット
					tm.commit();

					if (HinemosModuleConstant.MONITOR_CUSTOM.equals(monitorTypeId)) {
						// read-committedのため、commit後に外部コンポーネントに通知する
						CustomManagerUtil.broadcastConfigured();
					}
					if (HinemosModuleConstant.MONITOR_SYSTEMLOG.equals(monitorTypeId)) {
						SettingUpdateInfo.getInstance().setSystemLogMonitorUpdateTime(System.currentTimeMillis());
					}
					if (HinemosModuleConstant.MONITOR_SNMPTRAP.equals(monitorTypeId)) {
						SettingUpdateInfo.getInstance().setSnmptrapMonitorUpdateTime(System.currentTimeMillis());
					}
				}

			} catch(NotSupportedException e){
				outputLog(e, "delete()");
				throw e;
			} catch (RollbackException e) {
				outputLog(e, "delete()");
				throw e;
			} catch (HeuristicMixedException e) {
				outputLog(e, "delete()");
				throw e;
			} catch(HeuristicRollbackException e){
				outputLog(e, "delete()");
				throw e;
			} catch(SystemException e){
				outputLog(e, "delete()");
				throw e;
			} catch (CreateException e) {
				outputLog(e, "delete()");
				throw e;
			}

			finally{
				// トランザクション関連の例外が発生した場合は、ロールバック
				if(tm.getTransaction() != null){
					if(!result){
						tm.rollback();
					}
				}
			}
		}
		finally{

			// 一時停止していたトランザクションを再開
			if(oldTx != null){
				try{
					tm.resume(oldTx);

				} catch(InvalidTransactionException e){
					outputLog(e, "delete()");
					throw e;
				} catch(IllegalStateException e){
					outputLog(e, "delete()");
					throw e;
				} catch(SystemException e){
					outputLog(e, "delete()");
					throw e;
				}
			}
		}
		return result;
	}

	/**
	 * スケジューラから削除する
	 * @throws NamingException
	 * @throws TriggerSchedulerException
	 */
	protected void deleteSchedule() throws NamingException, TriggerSchedulerException {
		// Quartzに登録(runInterval = 0 -> スケジュール起動を行わない監視)
		if(m_monitor.getRunInterval() > 0){
			try {
				//ジョブ削除
				new ModifySchedule().deleteSchedule(m_monitorTypeId, m_monitorId);
			} catch (TriggerSchedulerException e) {
				m_log.warn("deleteQuartz() deleteJob : " + e.getClass().getSimpleName() + ", " + e.getMessage());
				throw e;
			}
		}
	}

	/**
	 * チェック条件情報を削除します。
	 * <p>
	 * 各監視管理のサブクラスで実装します。
	 * 
	 * @return 削除に成功した場合、</code> true </code>
	 * @throws FinderException
	 * @throws RemoveException
	 * @throws NamingException
	 * @throws NotifyNotFound
	 */
	protected abstract boolean deleteCheckInfo() throws FinderException, RemoveException, NamingException, HinemosUnknown, NotifyNotFound;

	/**
	 * 監視情報を削除します。
	 * <p>
	 * <ol>
	 * <li>監視対象IDと監視項目IDより、監視情報を取得します。</li>
	 * <li>チェック条件情報を削除します。各監視管理のサブクラスで実装します（{@link #deleteCheckInfo()}）。</li>
	 * <li>判定情報を削除します。各監視種別（真偽値，数値，文字列）のサブクラスで実装します（{@link #deleteJudgementInfo()}）。</li>
	 * <li>監視情報を削除します。</li>
	 * <li>Quartzから監視情報を削除します。</li>
	 * </ol>
	 * 
	 * @return 削除に成功した場合、</code> true </code>
	 * @throws RemoveException
	 * @throws NamingException
	 * @throws CreateException
	 * @throws HinemosUnknown
	 * @throws NotifyNotFound
	 * @throws MonitorNotFound
	 * @throws TriggerSchedulerException
	 * @throws JMSException
	 * 
	 * @see com.clustercontrol.monitor.run.ejb.entity.MonitorInfoBean
	 * @see #deleteCheckInfo()
	 * @see com.clustercontrol.monitor.run.factory.ModifySchedule#deleteSchedule(String, String)
	 */
	private boolean deleteMonitorInfo() throws RemoveException, NamingException, CreateException, NotifyNotFound, HinemosUnknown, MonitorNotFound, TriggerSchedulerException{

		try
		{
			// 監視情報を取得
			m_monitor = MonitorInfoUtil.getLocalHome().findByPrimaryKey(m_monitorId);

			// 監視グループ情報を削除
			NotifyControllerLocal nc = NotifyControllerUtil.getLocalHome().create();
			nc.deleteNotifyRelation(m_monitor.getNotifyGroupId());

			// この監視設定において収集された情報を削除
			if(deleteCascadeFlg){
				m_log.info("Delete Performance Data. monitorId = " + m_monitorId);
				try{
					CalculatedDataDAO calDao = new CalculatedDataDAO();
					calDao.delete(m_monitorId);
				}catch(EJBException e){
					// この監視設定に関連する収集情報が存在しない場合は何もしない
				}
			}else{
				m_log.info("Not Delete Performance Data. monitorId = " + m_monitorId);
			}

			// チェック条件情報を削除
			if(deleteCheckInfo()){
				// Quartzから削除
				deleteSchedule();

				// 監視情報を削除
				m_monitor.remove();

				// この監視設定の監視結果状態を削除する
				Collection<MonitorStatusLocal> statusList =
					MonitorStatusUtil.getLocalHome().findByPluginIdAndMonitorId(m_monitorTypeId, m_monitorId);
				for(MonitorStatusLocal status : statusList){
					MonitorStatusPK pk = (MonitorStatusPK)status.getPrimaryKey();
					status.remove();
					MonitorResultStatusUpdater.removeCache(pk);
				}

				// この監視設定の結果として通知された通知履歴を削除する
				Collection<NotifyHistoryLocal> historyList =
					NotifyHistoryUtil.getLocalHome().findByPluginIdAndMonitorId(m_monitorTypeId, m_monitorId);
				for(NotifyHistoryLocal history : historyList){
					history.remove();
				}

				return true;
			}


		} catch (FinderException e) {
			outputLog(e, "deleteMonitorInfo()");
			throw new MonitorNotFound(e.getMessage(),e);
		} catch (RemoveException e) {
			outputLog(e, "deleteMonitorInfo()");
			throw e;
		} catch (TriggerSchedulerException e) {
			outputLog(e, "deleteMonitorInfo()");
			throw e;
		} catch (NamingException e) {
			outputLog(e, "deleteMonitorInfo()");
			throw e;
		} catch (CreateException e) {
			outputLog(e, "deleteMonitorInfo()");
			throw e;
		}
		return false;
	}

	/**
	 * アプリケーションログにログを出力します。
	 * 
	 * @param e 例外
	 * @param method メソッド名
	 */
	private void outputLog(Exception e, String method) {
		AplLogger apllog = new AplLogger("MON", "mon");
		String[] args = {m_monitorTypeId, m_monitorId };
		apllog.put("SYS", "009", args);
		m_log.debug(method + ":" + e.getMessage());
	}
}
