/*

Copyright (C) 2006 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.

 */

package com.clustercontrol.monitor.run.ejb.session;

import java.rmi.RemoteException;

import javax.ejb.CreateException;
import javax.ejb.EJBException;
import javax.ejb.FinderException;
import javax.ejb.RemoveException;
import javax.ejb.SessionBean;
import javax.ejb.SessionContext;
import javax.naming.NamingException;
import javax.transaction.HeuristicMixedException;
import javax.transaction.HeuristicRollbackException;
import javax.transaction.InvalidTransactionException;
import javax.transaction.NotSupportedException;
import javax.transaction.RollbackException;
import javax.transaction.SystemException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.fault.HinemosUnknown;
import com.clustercontrol.bean.HinemosModuleConstant;
import com.clustercontrol.hinemosagent.factory.RunMonitorAgent;
import com.clustercontrol.http.factory.RunMonitorHttp;
import com.clustercontrol.http.factory.RunMonitorHttpString;
import com.clustercontrol.monitor.run.bean.MonitorTypeConstant;
import com.clustercontrol.monitor.run.factory.RunMonitor;
import com.clustercontrol.performance.monitor.factory.RunMonitorPerformance;
import com.clustercontrol.ping.factory.RunMonitorPing;
import com.clustercontrol.port.factory.RunMonitorPort;
import com.clustercontrol.process.factory.RunMonitorProcess;
import com.clustercontrol.snmp.factory.RunMonitorSnmp;
import com.clustercontrol.snmp.factory.RunMonitorSnmpString;
import com.clustercontrol.sql.factory.RunMonitorSql;
import com.clustercontrol.sql.factory.RunMonitorSqlString;
import com.clustercontrol.winservice.factory.RunMonitorWinService;

/**
 *  * Quartzから呼びだされて監視を実行するSession Bean<BR>
 * 
 * @ejb.bean name="MonitorRunManagement"
 *           jndi-name="MonitorRunManagement"
 *           type="Stateless"
 *           transaction-type="Container"
 *           view-type="local"
 * 
 * @ejb.transaction type="Required"
 * 
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=MonitorInfoLocal"
 * 
 * @ejb.permission
 *     unchecked="true"
 * 
 */
public abstract class MonitorRunManagementBean implements SessionBean {

	// Logger
	private static Log m_log = LogFactory.getLog( MonitorRunManagementBean.class );

	// SessionContext
	@SuppressWarnings("unused")
	private SessionContext m_context;

	/**
	 * SessionContextの設定
	 */
	@Override
	public void setSessionContext(SessionContext ctx) throws EJBException, RemoteException {
		m_context = ctx;
	}

	/**
	 * Quartzからのコールバックメソッド
	 * 
	 * @ejb.interface-method
	 * 
	 * トランザクション開始はユーザが制御する。
	 * また、追加実装により、トランザクションの入れ子が予期せず生じることを避けるため、Neverを採用する。
	 * @ejb.transaction type="Never"
	 * 
	 * @param monitorTypeId 監視監視対象ID
	 * @param monitorId 監視項目ID
	 * @param monitorType 監視判定タイプ
	 * @throws CreateException
	 * @throws FinderException
	 * @throws RemoveException
	 * @throws NamingException
	 * @throws NotSupportedException
	 * @throws HeuristicMixedException
	 * @throws HeuristicRollbackException
	 * @throws RollbackException
	 * @throws InvalidTransactionException
	 * @throws SystemException
	 * @version 4.0.0
	 * @throws HinemosUnknown
	 * @since 4.0.0
	 */
	public void run(String monitorTypeId, String monitorId, Integer monitorType) throws CreateException, FinderException, RemoveException, NamingException, NotSupportedException, HeuristicMixedException, HeuristicRollbackException, RollbackException, InvalidTransactionException, SystemException, HinemosUnknown {
		m_log.debug("run() monitorTypeId = " + monitorTypeId + ", monitorId = " + monitorId + ", monitorType = " + monitorType);

		// null チェック
		if(monitorId == null || "".equals(monitorId)){
			throw new HinemosUnknown("monitorId is null or empty.");
		}
		if(monitorTypeId == null || "".equals(monitorTypeId)){
			throw new HinemosUnknown("monitorTypeId is null or empty.");
		}

		RunMonitor runMonitor = null;
		try {
			// Hinemos エージェント監視
			if(HinemosModuleConstant.MONITOR_AGENT.equals(monitorTypeId)){
				runMonitor = new RunMonitorAgent();
			}
			// HTTP 監視
			else if(HinemosModuleConstant.MONITOR_HTTP.equals(monitorTypeId)){
				// 数値監視
				if(MonitorTypeConstant.TYPE_NUMERIC == monitorType){
					runMonitor = new RunMonitorHttp();
				}
				// 文字列
				else if(MonitorTypeConstant.TYPE_STRING == monitorType){
					runMonitor = new RunMonitorHttpString();
				}
				// それ以外
				else{
					throw new HinemosUnknown("This monitorTypeId = " + monitorTypeId + ", but unknown monitorType = " + monitorType);
				}
			}
			// ログファイル 監視
			else if(HinemosModuleConstant.MONITOR_LOGFILE.equals(monitorTypeId)){
				return;
			}
			// リソース 監視
			else if(HinemosModuleConstant.MONITOR_PERFORMANCE.equals(monitorTypeId)){
				runMonitor = new RunMonitorPerformance();
			}
			// ping監視
			else if(HinemosModuleConstant.MONITOR_PING.equals(monitorTypeId)){
				runMonitor = new RunMonitorPing();
			}
			// ポート監視
			else if(HinemosModuleConstant.MONITOR_PORT.equals(monitorTypeId)){
				runMonitor = new RunMonitorPort();
			}
			// プロセス監視
			else if(HinemosModuleConstant.MONITOR_PROCESS.equals(monitorTypeId)){
				runMonitor = new RunMonitorProcess();
			}
			// SNMPTRAP監視
			else if(HinemosModuleConstant.MONITOR_SNMPTRAP.equals(monitorTypeId)){
				return;
			}
			// SNMP監視
			else if(HinemosModuleConstant.MONITOR_SNMP.equals(monitorTypeId)){
				// 数値監視
				if(MonitorTypeConstant.TYPE_NUMERIC == monitorType){
					runMonitor = new RunMonitorSnmp();
				}
				// 文字列
				else if(MonitorTypeConstant.TYPE_STRING == monitorType){
					runMonitor = new RunMonitorSnmpString();
				}
				// それ以外
				else{
					throw new HinemosUnknown("This monitorTypeId = " + monitorTypeId + ", but unknown monitorType = " + monitorType);
				}
			}
			// システムログ監視
			else if(HinemosModuleConstant.MONITOR_SYSTEMLOG.equals(monitorTypeId)){
				return;
			}
			// SQL監視
			else if(HinemosModuleConstant.MONITOR_SQL.equals(monitorTypeId)){
				// 数値監視
				if(MonitorTypeConstant.TYPE_NUMERIC == monitorType){
					runMonitor = new RunMonitorSql();
				}
				// 文字列
				else if(MonitorTypeConstant.TYPE_STRING == monitorType){
					runMonitor = new RunMonitorSqlString();
				}
				// それ以外
				else{
					throw new HinemosUnknown("This monitorTypeId = " + monitorTypeId + ", but unknown monitorType = " + monitorType);
				}
			}
			// コマンド監視
			else if(HinemosModuleConstant.MONITOR_CUSTOM.equals(monitorTypeId)){
				return;
			}
			// Windowsサービス監視
			else if(HinemosModuleConstant.MONITOR_WINSERVICE.equals(monitorTypeId)){
				runMonitor = new RunMonitorWinService();
			}
			// Other
			else{
				throw new HinemosUnknown("Unknown monitorTypeId = " + HinemosModuleConstant.MONITOR_HTTP);
			}
		} catch (NamingException e) {
			m_log.warn("run() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw e;
		} catch (CreateException e) {
			m_log.warn("run() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw e;
		}

		if(runMonitor != null){
			try{
				runMonitor.run(monitorTypeId, monitorId);
			}catch(CreateException e){
				m_log.warn("run() : " + e.getClass().getSimpleName() +
						", " + e.getMessage());
				m_context.setRollbackOnly();
				throw e;
			}catch(RemoveException e){
				m_log.warn("run() : " + e.getClass().getSimpleName() +
						", " + e.getMessage());
				m_context.setRollbackOnly();
				throw e;
			}catch(NamingException e){
				m_log.warn("run() : " + e.getClass().getSimpleName() +
						", " + e.getMessage());
				m_context.setRollbackOnly();
				throw e;
			}catch(NotSupportedException e){
				m_log.warn("run() : " + e.getClass().getSimpleName() +
						", " + e.getMessage());
				m_context.setRollbackOnly();
				throw e;
			}catch(HeuristicMixedException e){
				m_log.warn("run() : " + e.getClass().getSimpleName() +
						", " + e.getMessage());
				m_context.setRollbackOnly();
				throw e;
			}catch(HeuristicRollbackException e){
				m_log.warn("run() : " + e.getClass().getSimpleName() +
						", " + e.getMessage());
				m_context.setRollbackOnly();
				throw e;
			}catch(RollbackException e){
				m_log.warn("run() : " + e.getClass().getSimpleName() +
						", " + e.getMessage());
				m_context.setRollbackOnly();
				throw e;
			}catch(InvalidTransactionException e){
				m_log.warn("run() : " + e.getClass().getSimpleName() +
						", " + e.getMessage());
				m_context.setRollbackOnly();
				throw e;
			}catch(SystemException e){
				m_log.warn("run() : " + e.getClass().getSimpleName() +
						", " + e.getMessage());
				m_context.setRollbackOnly();
				throw e;
			} catch (IllegalStateException e) {
				m_log.warn("run() : " + e.getClass().getSimpleName() +
						", " + e.getMessage());
				m_context.setRollbackOnly();
				throw e;
			} catch (FinderException e) {
				m_log.warn("run() : " + e.getClass().getSimpleName() +
						", " + e.getMessage());
				m_context.setRollbackOnly();
				throw e;
			} catch (HinemosUnknown e) {
				m_log.warn("run() : " + e.getClass().getSimpleName() +
						", " + e.getMessage());
				m_context.setRollbackOnly();
				throw e;
			}
		}
	}
}
