/*

 Copyright (C) 2006 NTT DATA Corporation

 This program is free software; you can redistribute it and/or
 Modify it under the terms of the GNU General Public License
 as published by the Free Software Foundation, version 2.

 This program is distributed in the hope that it will be
 useful, but WITHOUT ANY WARRANTY; without even the implied
 warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
 PURPOSE.  See the GNU General Public License for more details.

 */

package com.clustercontrol.jobmanagement.dao;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.Collection;

import javax.ejb.CreateException;
import javax.ejb.DuplicateKeyException;
import javax.ejb.EJBException;
import javax.ejb.FinderException;
import javax.ejb.RemoveException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.commons.util.ConnectionManager;
import com.clustercontrol.jobmanagement.ejb.entity.JobNoticeMasterBean;
import com.clustercontrol.jobmanagement.ejb.entity.JobNoticeMasterPK;

/**
 * ジョブ通知マスタのDAOインターフェースを実装するクラス<BR>
 * 
 * @version 3.0.0
 * @since 2.1.0
 * 
 * @see com.clustercontrol.jobmanagement.ejb.entity.JobNoticeMasterBean
 * @see com.clustercontrol.jobmanagement.dao.JobNoticeMasterDAO
 */
public class JobNoticeMasterDAOImpl implements JobNoticeMasterDAO {

	/** ログ出力のインスタンス。 */
	private static Log m_log = LogFactory.getLog(JobNoticeMasterDAOImpl.class);

	@Override
	public void init() {
	}

	/**
	 * SELECT文を発行します。
	 * 引数で指定されたプライマリキーで検索し、取得したデータベースの内容をEntity Beanに反映します。
	 * 
	 * @see com.clustercontrol.jobmanagement.dao.JobNoticeMasterDAO#load(com.clustercontrol.jobmanagement.ejb.entity.JobNoticeMasterPK, com.clustercontrol.jobmanagement.ejb.entity.JobNoticeMasterBean)
	 */
	@Override
	public void load(JobNoticeMasterPK pk, JobNoticeMasterBean ejb) throws EJBException {
		m_log.debug("load() start : " + pk.toString());

		Connection conn = null;
		PreparedStatement stmt = null;
		ResultSet res = null;

		try {
			conn = ConnectionManager.getConnectionManager().getConnection();

			//SQL文の定義
			String sql = "SELECT * FROM cc_job_notice_mst WHERE job_id = ? AND notice_type = ? AND jobunit_id = ?";
			//SQL文のセット
			stmt = conn.prepareStatement(sql);

			stmt.setString(1, pk.getJob_id());
			stmt.setInt(2, pk.getNotice_type());
			stmt.setString(3, pk.getJobunit_id());

			res = stmt.executeQuery();

			if (res.next()) {
				//取得した値をBeanにセット
				ejb.setJob_id(res.getString("job_id"));
				ejb.setNotice_type(res.getInt("notice_type"));
				ejb.setNotifyGroupId(res.getString("notify_group_id"));
				ejb.setPriority(res.getInt("priority"));
				ejb.setJobunit_id(res.getString("jobunit_id"));
			} else {
				String msg = "JobNoticeMaster data is not found.";
				m_log.error("load() error : " + pk.toString() + " SQLException " + msg);
				throw new EJBException(msg);
			}
		} catch (SQLException e) {
			m_log.error("load() error : " + pk.toString() + " SQLException");
			throw new EJBException(e.getMessage());
		} finally {
			//コネクション、結果セット、プリペアドステートメントのクローズ
			try {
				if (stmt != null) {
					stmt.close();
				}
				if (res != null) {
					res.close();
				}
				if (conn != null) {
					conn.close();
				}
			} catch (SQLException e1) {
				m_log.error("load() error : " + pk.toString() + " SQLException " + e1.getMessage());
				throw new EJBException(e1.getMessage());
			}
		}
		m_log.debug("load() end : " + pk.toString());
	}

	/**
	 * UPDATE文を発行します。
	 * 引数で指定されたEntity Beanの内容でデータベースを更新します。
	 * 
	 * @see com.clustercontrol.jobmanagement.dao.JobNoticeMasterDAO#store(com.clustercontrol.jobmanagement.ejb.entity.JobNoticeMasterBean)
	 */
	@Override
	public void store(JobNoticeMasterBean ejb) throws EJBException {
		m_log.debug("store() start : jobunitId=" + ejb.getJobunit_id() + ", jobId=" + ejb.getJob_id() + ", noticeType=" + ejb.getNotice_type());

		Connection conn = null;
		PreparedStatement stmt = null;

		try {
			conn = ConnectionManager.getConnectionManager().getConnection();
			//SQL文の定義
			StringBuilder sql = new StringBuilder();
			sql.append("UPDATE cc_job_notice_mst SET ");
			sql.append("priority = ?, ");
			sql.append("notify_group_id = ? ");
			sql.append("WHERE job_id = ? AND ");
			sql.append("notice_type = ? AND ");
			sql.append("jobunit_id = ? ");

			//SQL文のセット
			stmt = conn.prepareStatement(sql.toString());

			stmt.setInt(1, ejb.getPriority());
			stmt.setString(2, ejb.getNotifyGroupId());
			stmt.setString(3, ejb.getJob_id());
			stmt.setInt(4, ejb.getNotice_type());
			stmt.setString(5, ejb.getJobunit_id());

			int row = stmt.executeUpdate();

			if (row != 1) {
				String msg = "result row is not 1";
				m_log.error("store() error : jobunitId=" + ejb.getJobunit_id()
						+ ", jobId=" + ejb.getJob_id() + ", noticeType="
						+ ejb.getNotice_type() + " SQLException" + msg);
				throw new EJBException(msg);
			}
		} catch (SQLException e) {
			m_log.error("store() error : jobunitId=" + ejb.getJobunit_id()
					+ ", jobId=" + ejb.getJob_id() + ", noticeType="
					+ ejb.getNotice_type() + " SQLException");
			throw new EJBException(e.getMessage());
		} finally {
			// コネクション、結果セット、プリペアドステートメントのクローズ
			try {
				if (stmt != null) {
					stmt.close();
				}
				if (conn != null) {
					conn.close();
				}
			} catch (SQLException e1) {
				m_log.error("store() error : jobunitId=" + ejb.getJobunit_id()
						+ ", jobId=" + ejb.getJob_id() + ", noticeType="
						+ ejb.getNotice_type() + " SQLException");
				throw new EJBException(e1.getMessage());
			}
		}
		m_log.debug("store() end : jobunitId=" + ejb.getJobunit_id()
				+ ", jobId=" + ejb.getJob_id() + ", noticeType="
				+ ejb.getNotice_type());
	}

	/**
	 * DELETE文を発行します。
	 * 引数で指定されたプライマリキーでデータベースから削除します。
	 * 
	 * @see com.clustercontrol.jobmanagement.dao.JobNoticeMasterDAO#remove(com.clustercontrol.jobmanagement.ejb.entity.JobNoticeMasterPK)
	 */
	@Override
	public void remove(JobNoticeMasterPK pk) throws RemoveException, EJBException {
		m_log.debug("remove() start : " + pk.toString());

		Connection conn = null;
		PreparedStatement stmt = null;
		try {
			conn = ConnectionManager.getConnectionManager().getConnection();

			//SQL文の定義
			String sql = "DELETE FROM cc_job_notice_mst WHERE job_id = ? AND notice_type = ? AND jobunit_id = ?";
			//SQL文のセット
			stmt = conn.prepareStatement(sql);

			stmt.setString(1, pk.getJob_id());
			stmt.setInt(2, pk.getNotice_type());
			stmt.setString(3, pk.getJobunit_id());

			int row = stmt.executeUpdate();

			if (row != 1 ) {
				String msg = "result row is not 1";
				m_log.error("remove() error : " + pk.toString() + " SQLException" + msg);
				throw new EJBException(msg);
			}
		} catch (SQLException e) {
			m_log.error("remove() error : " + pk.toString() + " SQLException");
			throw new EJBException(e.getMessage());

		} finally {
			try {
				if(stmt != null){
					stmt.close();
				}
				if(conn != null){
					conn.close();
				}
			} catch (SQLException e1) {
				m_log.error("remove() error : " + pk.toString() + " SQLException");
				throw new EJBException(e1.getMessage());
			}
		}
		m_log.debug("remove() end : " + pk.toString());
	}

	/**
	 * INSERT文を発行します。
	 * 引数で指定されたEntity Beanの内容をデータベースに挿入し、プライマリキーを返します。
	 * 
	 * @see com.clustercontrol.jobmanagement.dao.JobNoticeMasterDAO#create(com.clustercontrol.jobmanagement.ejb.entity.JobNoticeMasterBean)
	 */
	@Override
	public JobNoticeMasterPK create(JobNoticeMasterBean ejb) throws CreateException, EJBException {
		m_log.debug("create() start : jobunitId=" + ejb.getJobunit_id()
				+ ", jobId=" + ejb.getJob_id() + ", noticeType"
				+ ejb.getNotice_type());

		Connection conn = null;
		PreparedStatement stmt = null;
		JobNoticeMasterPK pk;
		try {
			conn = ConnectionManager.getConnectionManager().getConnection();
			//SQL文の定義
			StringBuilder sql = new StringBuilder();
			sql.append("INSERT INTO cc_job_notice_mst (");
			sql.append("job_id, ");
			sql.append("notice_type, ");
			sql.append("priority, ");
			sql.append("notify_group_id, ");
			sql.append("jobunit_id) ");
			sql.append("VALUES (?,?,?,?,?)");

			//SQL文のセット
			stmt = conn.prepareStatement(sql.toString());

			stmt.setString(1, ejb.getJob_id());
			stmt.setInt(2, ejb.getNotice_type());
			stmt.setInt(3, ejb.getPriority());
			stmt.setString(4, ejb.getNotifyGroupId());
			stmt.setString(5, ejb.getJobunit_id());

			int row =	stmt.executeUpdate();
			pk = new JobNoticeMasterPK(
					ejb.getJobunit_id(),
					ejb.getJob_id(),
					ejb.getNotice_type());

			if (row != 1) {
				String msg = "result row is not 1";
				m_log.error("create() error : jobunitId=" + ejb.getJobunit_id()
						+ ", jobId=" + ejb.getJob_id() + ", noticeType"
						+ ejb.getNotice_type() + " SQLException");
				throw new EJBException(msg);
			}
		} catch (SQLException e) {
			if(e.getSQLState().equals("23505")){
				m_log.error("create() error :: jobunitId="
						+ ejb.getJobunit_id() + ", jobId=" + ejb.getJob_id()
						+ ", noticeType" + ejb.getNotice_type()
						+ " SQLException" + "DuplicateKeyException ");
				throw new DuplicateKeyException(e.getMessage());
			}
			throw new CreateException(e.getMessage());
		}finally{
			try {
				if(stmt != null){
					stmt.close();
				}
				if(conn != null){
					conn.close();
				}
			} catch (SQLException e1) {
				m_log.error("create() error : jobunitId=" + ejb.getJobunit_id()
						+ ", jobId=" + ejb.getJob_id() + ", noticeType"
						+ ejb.getNotice_type() + " SQLException");
				throw new EJBException(e1.getMessage());
			}
		}
		m_log.debug("create() end : jobunitId=" + ejb.getJobunit_id()
				+ ", jobId=" + ejb.getJob_id() + ", noticeType"
				+ ejb.getNotice_type());
		return pk;
	}

	/**
	 * 全件取得するSELECT文を発行します。
	 * 取得したデータのプライマリキーをコレクションに格納し返します。
	 * 
	 * @see com.clustercontrol.jobmanagement.ejb.entity.JobNoticeMasterBean#ejbFindAll()
	 * @see com.clustercontrol.jobmanagement.dao.JobNoticeMasterDAO#findAll()
	 */
	@Override
	public Collection findAll() throws FinderException {
		m_log.debug("findAll() start : ");

		ArrayList<JobNoticeMasterPK> ret = new ArrayList<JobNoticeMasterPK>();

		Connection conn = null;
		PreparedStatement stmt = null;
		ResultSet res = null;
		try {
			conn = ConnectionManager.getConnectionManager().getConnection();
			//SQL文の定義
			String sql = "SELECT jobunit_id, job_id, notice_type FROM cc_job_notice_mst";
			//SQL文のセット
			stmt = conn.prepareStatement(sql);
			res = stmt.executeQuery();

			while(res.next()) {
				JobNoticeMasterPK pk= new JobNoticeMasterPK(
						res.getString("jobunit_id"),
						res.getString("job_id"),
						res.getInt("notice_type")
				);
				ret.add(pk);
			}

		} catch (SQLException e) {
			m_log.error("findAll() error :  SQLException");
			throw new EJBException(e.getMessage());
		}  finally{
			try {
				if(stmt != null){
					stmt.close();}
				if(res != null){
					res.close();}
				if(conn != null){
					conn.close();
				}
			} catch (SQLException e1) {
				m_log.error("findAll() error :  SQLException");
				throw new EJBException(e1.getMessage());
			}
		}
		m_log.debug("findAll() end : ");
		return ret;
	}

	/**
	 * 1件取得するSELECT文を発行します。
	 * 引数で指定されたプライマリキーで、データベースを検索します。
	 * 
	 * @see com.clustercontrol.jobmanagement.ejb.entity.JobNoticeMasterBean#ejbFindByPrimaryKey(com.clustercontrol.jobmanagement.ejb.entity.JobNoticeMasterPK)
	 * @see com.clustercontrol.jobmanagement.dao.JobNoticeMasterDAO#findByPrimaryKey(com.clustercontrol.jobmanagement.ejb.entity.JobNoticeMasterPK)
	 */
	@Override
	public JobNoticeMasterPK findByPrimaryKey(JobNoticeMasterPK pk) throws FinderException {
		m_log.debug("findByPrimaryKey() start : " + pk.toString());

		Connection conn = null;
		PreparedStatement stmt = null;
		ResultSet res = null;
		try {
			conn = ConnectionManager.getConnectionManager().getConnection();

			//SQL文の定義
			String sql = "SELECT jobunit_id, job_id, notice_type FROM cc_job_notice_mst WHERE job_id = ? AND notice_type = ? AND jobunit_id = ?";
			//SQL文のセット
			stmt = conn.prepareStatement(sql);

			stmt.setString(1, pk.getJob_id());
			stmt.setInt(2, pk.getNotice_type());
			stmt.setString(3, pk.getJobunit_id());

			res = stmt.executeQuery();

			if (res.next()) {
				return pk;
			} else {
				String msg = "id " + pk.toString() + " not found.";
				throw new FinderException(msg);
			}
		} catch (SQLException e) {
			m_log.error("find() error : " + pk.toString() + " SQLException");
			throw new EJBException(e.getMessage());
		}  finally{
			try {
				if(stmt != null){
					stmt.close();}
				if(res != null){
					res.close();}
				if(conn != null){
					conn.close();
				}
			} catch (SQLException e1) {
				m_log.error("find() error : " + pk.toString() + " SQLException");
				throw new EJBException(e1.getMessage());
			}
			m_log.debug("findByPrimaryKey() end : " + pk.toString());
		}
	}

	/**
	 * 複数件取得するSELECT文を発行します。
	 * 引数で指定されたジョブIDで、データベースを検索します。
	 * 
	 */
	@Override
	public Collection findByJobunitJob(String jobunit_id, String job_id) throws FinderException {
		m_log.debug("findByJobId() start : " + job_id);

		ArrayList<JobNoticeMasterPK> ret = new ArrayList<JobNoticeMasterPK>();

		Connection conn = null;
		PreparedStatement stmt = null;
		ResultSet res = null;
		try {
			conn = ConnectionManager.getConnectionManager().getConnection();
			//SQL文の定義
			String sql = "SELECT jobunit_id, job_id, notice_type FROM cc_job_notice_mst" +
			" WHERE job_id = ? AND jobunit_id = ?";
			//SQL文のセット
			stmt = conn.prepareStatement(sql);

			stmt.setString(1, job_id);
			stmt.setString(2, jobunit_id);

			res = stmt.executeQuery();

			while(res.next()) {
				JobNoticeMasterPK pk= new JobNoticeMasterPK(
						res.getString("jobunit_id"),
						res.getString("job_id"),
						res.getInt("notice_type")
				);
				ret.add(pk);
			}

		} catch (SQLException e) {
			m_log.error("findByJobId() error : " + job_id + " SQLException");
			throw new EJBException(e.getMessage());
		}  finally{
			try {
				if(stmt != null){
					stmt.close();}
				if(res != null){
					res.close();}
				if(conn != null){
					conn.close();
				}
			} catch (SQLException e1) {
				m_log.error("findByJobId() error : " + job_id + " SQLException");
				throw new EJBException(e1.getMessage());
			}
		}
		m_log.debug("findByJobId() end : " + job_id);
		return ret;
	}
}
