/*
Copyright (C) 2011 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.
 */

package com.clustercontrol.hinemosagent.util;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.DatagramPacket;
import java.net.DatagramSocket;
import java.net.InetAddress;
import java.net.Socket;
import java.net.SocketException;
import java.net.UnknownHostException;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.concurrent.ConcurrentHashMap;

import javax.ejb.CreateException;
import javax.ejb.FinderException;
import javax.jms.JMSException;
import javax.naming.NamingException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.fault.FacilityNotFound;
import com.clustercontrol.fault.HinemosUnknown;
import com.clustercontrol.hinemosagent.bean.AgentInfo;
import com.clustercontrol.hinemosagent.bean.TopicInfo;
import com.clustercontrol.bean.PriorityConstant;
import com.clustercontrol.commons.util.HinemosProperties;
import com.clustercontrol.commons.util.SendQueue;
import com.clustercontrol.monitor.run.ejb.entity.MonitorInfoLocal;
import com.clustercontrol.monitor.run.ejb.entity.MonitorInfoUtil;
import com.clustercontrol.notify.bean.OutputBasicInfo;
import com.clustercontrol.notify.bean.QueueConstant;
import com.clustercontrol.repository.bean.NodeInfo;
import com.clustercontrol.repository.ejb.session.RepositoryControllerLocal;
import com.clustercontrol.repository.ejb.session.RepositoryControllerUtil;
import com.clustercontrol.repository.factory.FacilitySelector;
import com.clustercontrol.repository.factory.NodeProperty;
import com.clustercontrol.util.Messages;


public class AgentConnectUtil {

	private static Log m_log = LogFactory.getLog( AgentConnectUtil.class );

	// synchronized用のオブジェクト
	private static Object topicObject = new Object();

	private static SendQueue m_queueInternal;

	// 接続しているエージェントのリスト
	private static ConcurrentHashMap<String, AgentInfo> agentMap =
		new ConcurrentHashMap<String, AgentInfo>();

	// 接続しているエージェントのライブラリのMD5
	// ConcurrentHashMap<facilityId, HashMap<filename, md5>>
	private static ConcurrentHashMap<String, HashMap<String, String>> agentLibMd5 =
		new ConcurrentHashMap<String, HashMap<String, String>>();

	// エージェントへのTOPIC送信先ポート番号
	private static int AWAKE_PORT = 24005;
	
	// 初期化
	static {
		try{
			AWAKE_PORT = Integer.parseInt(HinemosProperties.getProperty("common.ws.awake.port", "24005"));
		} catch (Exception e) {
			m_log.warn("common.ws.awake.port is invalid. " + e.getMessage()); 
		}
		m_log.info("common.ws.awake.port(udp)=" + AWAKE_PORT); 
	}
	
	/**
	 * TODO
	 * このHashMapはejbを利用してテーブルに置き、session beanでアクセスする事。
	 * (src_agentに移動すること。)
	 * そうしないと、HashMapに入っていてエージェントに送る前にシャットダウンすると、
	 * データが損失してしまう。
	 */
	private static ConcurrentHashMap<String, ArrayList<TopicInfo>> topicMap =
		new ConcurrentHashMap<String, ArrayList<TopicInfo>> ();

	/**
	 * [Topic] エージェントにTopicを送る場合はこのメソッドを利用する。
	 * リポジトリ機能やログファイル監視機能やコマンド監視機能などから呼ばれる。
	 */
	public static void setTopic (String facilityId, TopicInfo info) {
		synchronized(topicObject) {
			// 全ノードにTopicを送信
			// この場合、引数のfacilityIdはnullになる。
			if (facilityId == null) {
				for (String fid : getValidAgent()) {
					subSetTopic(fid, info);
					awakeAgent(fid);
				}
				return;
			}

			// 特定のノードにTopicを送信
			// 存在しているノードか否か確認
			boolean flag = false;
			for (String fid : getValidAgent()) {
				if (fid.equals(facilityId)) {
					flag = true;
				}
			}
			if (flag) {
				m_log.info("setTopic : " + facilityId + ", " + info);
				subSetTopic(facilityId, info);
				awakeAgent(facilityId);
			} else {
				m_log.info("setTopic : " + facilityId + " is not valid");
			}
		}
	}

	/**
	 * [Topic]
	 * @param facilityId
	 * @param info
	 */
	private static void subSetTopic(String facilityId, TopicInfo info) {
		info.setFacilityId(facilityId);
		ArrayList<TopicInfo> infoList = topicMap.get(facilityId);
		if (infoList == null) {
			infoList = new ArrayList<TopicInfo>();
			topicMap.put(facilityId, infoList);
		}
		// Topicのリストに同一のものがある場合は、追加しない。
		if (infoList.contains(info)) {
			m_log.info("subSetTopic(): same topic. Maybe, agent timeout error occured.");
		} else {
			infoList.add(info);
			if (infoList.size() > 10) {
				m_log.warn("subSetTopic(): topicList is too large : size=" +
						infoList.size());
			}
		}
		printRunInstructionInfo(facilityId, infoList);
	}


	/**
	 * [Topic]
	 * ファシリティIDをキーにして、トピックのリストを取得
	 * @param facilityId
	 * @return
	 */
	public static ArrayList<TopicInfo> getTopic (String facilityId) {
		ArrayList<TopicInfo> ret = null;
		m_log.debug("getJobOrder : " + facilityId);
		synchronized(topicObject) {
			ArrayList<TopicInfo> tmpInfoList = topicMap.get(facilityId);

			if (tmpInfoList == null) {
				return null;
			}
			ret = new ArrayList<TopicInfo> ();
			for (TopicInfo topicInfo : tmpInfoList) {
				if (topicInfo.isValid()) {
					ret.add(topicInfo);
				} else {
					m_log.warn("topic is too old : " + topicInfo.toString());
				}
			}
			if (tmpInfoList != null) {
				tmpInfoList.clear();
			}
		}
		return ret;
	}

	/**
	 * [Topic] getTopicを実行するという命令を、UDPパケットで送信する。
	 * @param facilityId
	 */
	private static void awakeAgent(String facilityId) {
		String ipAddress = "";
		m_log.debug("awakeAgent facilityId=" + facilityId);
		try {
			RepositoryControllerLocal local;
			local = RepositoryControllerUtil.getLocalHome().create();
			ipAddress = local.getNode(facilityId).getAvailableIpAddress();
		} catch (CreateException e) {
			m_log.warn("awakeAgent " + e.getClass().getSimpleName() + ", " + e.getMessage());
			return;
		} catch (NamingException e) {
			m_log.warn("awakeAgent " + e.getClass().getSimpleName() + ", " + e.getMessage());
			return;
		} catch (FacilityNotFound e) {
			m_log.warn("awakeAgent " + e.getClass().getSimpleName() + ", " + e.getMessage());
			return;
		}
		m_log.debug("awakeAgent ipaddress=" + ipAddress);

		final int BUFSIZE = 1;
		byte[] buf = new byte[BUFSIZE];
		buf[0] = 1;
		InetAddress sAddr;
		try {
			sAddr = InetAddress.getByName(ipAddress);
		} catch (UnknownHostException e) {
			m_log.warn("awakeAgent " + e.getClass().getSimpleName() + ", " + e.getMessage());
			return;
		}
		DatagramPacket sendPacket = new DatagramPacket(buf, BUFSIZE, sAddr, AWAKE_PORT);
		DatagramSocket soc = null;
		try {
			soc = new DatagramSocket();
			soc.send(sendPacket);
		} catch (SocketException e) {
			m_log.warn("awakeAgent " + e.getClass().getSimpleName() + ", " + e.getMessage());
		} catch (IOException e) {
			m_log.warn("awakeAgent " + e.getClass().getSimpleName() + ", " + e.getMessage());
		} finally {
			if (soc != null) {
				soc.close();
			}
		}
	}

	private static void printRunInstructionInfo (String facilityId, ArrayList<TopicInfo> topicInfoList) {
		String str = "";
		if (topicInfoList == null) {
			m_log.debug("printRunInstructionInfo JobOrder : " + facilityId + ", infoList is null");
			return;
		}
		for (TopicInfo topicInfo : topicInfoList) {
			if (topicInfo.getRunInstructionInfo() != null) {
				str += topicInfo.getRunInstructionInfo().getJobId() + ", "
				+ topicInfo.getRunInstructionInfo().getSessionId() + ", ";
			}
		}
		m_log.debug("printRunInstructionInfo JobOrder : " + facilityId + ", " + str);
	}


	/**
	 * エージェント一覧に追加。
	 * このメソッドを利用する場合は、agentInfoのfacilityIdに値をつめること。
	 */
	public static void putAgentMap (AgentInfo agentInfo) {
		String facilityId = agentInfo.getFacilityId();
		if (facilityId == null) {
			m_log.warn("putAgentMap facilityId=null");
			return;
		}
		agentInfo.refreshLastLogin();
		AgentInfo cacheInfo = agentMap.get(facilityId);
		// マップに存在しない場合は、単純に追加。
		if(cacheInfo == null) {
			m_log.info("new agent appeared : " + agentInfo);
			if (!facilityId.equals(agentInfo.getFacilityId())){
				m_log.error("facilityId=" + facilityId + ", f2=" + agentInfo.getFacilityId());
			}
			agentMap.put(facilityId, agentInfo.clone());
			return;
		}

		// マップに過去の自分が存在している場合は、lastLoginを更新。
		if (cacheInfo.getStartupTime() == agentInfo.getStartupTime()) {
			m_log.debug("refresh " + agentInfo);
			cacheInfo.refreshLastLogin();
		} else {
			/*
			 * TODO
			 * 「同一ファシリティIDのエージェントが存在します。」
			 * というメッセージを出力させる。
			 */
			m_log.info("agents are duplicate : " + facilityId + "[" +
					cacheInfo.toString() + "->" + agentInfo.toString() + "]");
			OutputBasicInfo outputBasicInfo = new OutputBasicInfo();
			outputBasicInfo.setApplication("agent");
			outputBasicInfo.setFacilityId(facilityId);
			outputBasicInfo.setGenerationDate(new Date().getTime());
			String[] args = {facilityId};
			outputBasicInfo.setMessage(Messages.getString("message.agent.3", args));
			outputBasicInfo.setMessageId("001");
			outputBasicInfo.setMessageOrg(Messages.getString("message.agent.3", args));
			outputBasicInfo.setMonitorId("SYS");
			outputBasicInfo.setPluginId("001");
			outputBasicInfo.setPriority(PriorityConstant.TYPE_CRITICAL);
			outputBasicInfo.setScopeText(facilityId);
			try {
				ArrayList<String> facilityIdList = new ArrayList<String>();
				facilityIdList.add(facilityId);
				sendMessageLocal(outputBasicInfo, facilityIdList);
			} catch (HinemosUnknown e) {
				m_log.error("pubAgentMap " + e.getClass().getSimpleName() + e.getMessage());
			} catch (FacilityNotFound e) {
				m_log.error("pubAgentMap " + e.getClass().getSimpleName() + e.getMessage());
			}
			if (!facilityId.equals(agentInfo.getFacilityId())){
				// ここは通らないはず。
				m_log.error("facilityId=" + facilityId + ", f2=" + agentInfo.getFacilityId());
			}
			agentMap.put(facilityId, agentInfo.clone());
		}
	}

	/**
	 * 有効なエージェント一覧を取得する。
	 */
	public static ArrayList<String> getValidAgent() {
		ArrayList<String> list = new ArrayList<String>();
		for (String fid : agentMap.keySet()) {
			if (agentMap.get(fid).isValid()) {
				list.add(fid);
			} else {
				AgentInfo agentInfo = agentMap.remove(fid);
				m_log.info("remove " + agentInfo);
			}
		}
		return list;
	}

	/**
	 * エージェントが有効かチェック
	 */
	public static boolean isValidAgent(String facilityId){
		AgentInfo agentInfo = agentMap.get(facilityId);
		if (agentInfo == null) {
			return false;
		}
		boolean flag = agentInfo.isValid();
		if (!flag) {
			agentMap.remove(facilityId);
		}
		return flag;
	}

	/**
	 * AgentInfo#toStringを取得する。
	 */
	public static String getAgentString(String facilityId) {
		AgentInfo agentInfo = agentMap.get(facilityId);
		if (agentInfo == null) {
			return null;
		}
		return agentInfo.toString();
	}

	public static ArrayList<AgentInfo> getAgentList() {
		ArrayList<AgentInfo> ret = new ArrayList<AgentInfo>();
		for (String facilityId : agentMap.keySet()) {
			ret.add(agentMap.get(facilityId));
		}
		return ret;
	}

	public static void deleteAgent(String facilityId) {
		agentMap.remove(facilityId);
		agentLibMd5.remove(facilityId);
	}



	//////////////////////////////////
	// エージェント　リモートアップデート機能
	//////////////////////////////////
	public static HashMap<String, String> getAgentLibMd5(String facilityId) {
		HashMap<String, String> map = agentLibMd5.get(facilityId);
		return map;
	}

	/**
	 * AgentLibMd5にファシリティIDとMD5の組がセットされているかチェックする。
	 * セットされていない場合は、トピックを発行して、情報取得を依頼する。
	 * (その直後にsetAgentLibMd5がやってくるはず。)
	 * @param facilityId
	 */
	public static void checkAgentLibMd5(String facilityId) {
		HashMap<String, String> map = agentLibMd5.get(facilityId);
		if (map != null) {
			return;
		}
		TopicInfo topicInfo = new TopicInfo();
		topicInfo.setFacilityId(facilityId);
		topicInfo.setNewFacilityFlag(true);
		AgentConnectUtil.setTopic(facilityId, topicInfo);
	}

	public static void setAngetLibMd5(String facilityId, HashMap<String, String> map) {
		agentLibMd5.put(facilityId, map);
	}

	//////////////////////////////////
	// インターナルメッセージ
	//////////////////////////////////
	public static void sendMessageLocal(OutputBasicInfo outputBasicInfo,
			ArrayList<String> facilityIdList)
	throws HinemosUnknown, FacilityNotFound {

		if (m_queueInternal == null) {
			try {
				m_queueInternal = new SendQueue(QueueConstant.QUEUE_NAME_EVENT);
			} catch (NamingException e) {
				m_log.error(e,e);
				throw new HinemosUnknown(e.getMessage(),e);
			} catch (JMSException e) {
				m_log.error(e,e);
				throw new HinemosUnknown(e.getMessage(),e);
			}
		}
		// 監視情報を取得
		ArrayList<String> facilityList = null;
		try {
			MonitorInfoLocal localMonitor = MonitorInfoUtil.getLocalHome().findByPrimaryKey(outputBasicInfo.getMonitorId());
			String monitorFacilityId = localMonitor.getFacilityId();
			facilityList = FacilitySelector.getFacilityIdList(monitorFacilityId, 0, false);
		} catch (FinderException e) {
			m_log.warn(e.getMessage() + " (" + outputBasicInfo.getMonitorId() + ")",e);
		} catch (NamingException e) {
			m_log.error(e.getMessage(),e);
		}
		for (String facilityId : facilityIdList) {
			if (facilityList != null && !facilityList.contains(facilityId)) {
				m_log.debug("not match facilityId(" + facilityId + ")");
				continue;
			}
			try {
				RepositoryControllerLocal local;
				try {
					local = RepositoryControllerUtil.getLocalHome().create();
				} catch (CreateException e) {
					m_log.error(e,e);
					throw new HinemosUnknown(e.getMessage(),e);
				} catch (NamingException e) {
					m_log.error(e,e);
					throw new HinemosUnknown(e.getMessage(),e);
				}
				if (local == null) {
					m_log.error("RepositoryControllerLocal is null");
					return;
				}
				String scopeText = "";
				try {
					NodeInfo nodeInfo = local.getNode(facilityId);
					scopeText = nodeInfo.getFacilityName();
				} catch (FacilityNotFound e) {
					throw e;
				}
				outputBasicInfo.setFacilityId(facilityId);
				outputBasicInfo.setScopeText(scopeText);
				m_queueInternal.put(outputBasicInfo);
			} catch (JMSException e) {
				m_log.error(e,e);
				throw new HinemosUnknown(e.getMessage(),e);
			}
		}
	}
	
	/**
	 * エージェントにマネージャのIPを送信する。
	 * エージェントとTCPセッションが確立できた場合、trueを返す。
	 * @param facilityId
	 * @return
	 * @throws UnknownHostException
	 * @throws IOException 
	 */
	public static boolean sendManagerIp(String facilityId) throws UnknownHostException, IOException{
		int servPort = 24005;
		String managerIpAddr = "";
		boolean successFlag = true;
		String agentIpAddr = "";
		
		try{
			managerIpAddr = HinemosProperties.getProperty("agent.connection.ipaddres", InetAddress.getLocalHost().getHostAddress());
		} catch (UnknownHostException e){
			throw e;
		}
		
		Socket socket = null;
		InputStream is = null;
		try {
			String sendData = "managerIp=" + managerIpAddr + ",agentFacilityId=" + facilityId;
			byte[] data = sendData.getBytes();
			byte[] msg = new byte[data.length];
			agentIpAddr = NodeProperty.getProperty(facilityId).getAvailableIpAddress();
			socket = new Socket(agentIpAddr, servPort);
			
			m_log.info("established the connection to the hinemos agent server at " + agentIpAddr);
			
			is = socket.getInputStream();
			OutputStream out = socket.getOutputStream();
			out.write(data);
			
			m_log.info("sent the message： " + new String(data));
			
			// エージェントからの返信を受信
			int totalBytesRcvd = 0;
			int bytesRcvd;
			
			while (totalBytesRcvd < data.length) {
				if ((bytesRcvd = is.read(msg, totalBytesRcvd, data.length - totalBytesRcvd)) == -1) {
					continue;
				}
				totalBytesRcvd += bytesRcvd;
			}
			m_log.info("received the message: " + new String(msg));
		} catch (Exception e) {
			successFlag = false;
			m_log.warn("facilityId: " + facilityId + ", " + e.getMessage());
		}
		
		try {
			if(is != null){
				is.close();
			}
			if(socket != null){
				socket.close();
			}
		} catch (IOException e) {
			throw e;
		}
		return successFlag;
	}
	
	/**
	 * 該当のファシリティIDに対応するエージェントがマネージャと接続されているかを確認する
	 * @param facilityId
	 * @return
	 */
	public static boolean checkAgentConnection(String facilityId){
		boolean ret = false;
		for(AgentInfo agentInfo : AgentConnectUtil.getAgentList()){
			if(facilityId.equals(agentInfo.getFacilityId())){
				// 該当ファシリティIDがエージェントリストに存在する場合
				ret = true;
			}
		}
		return ret;
	}
}