/*

Copyright (C) 2006 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.

 */

package com.clustercontrol.repository.dialog;

import java.util.ArrayList;
import java.util.Locale;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.TableTree;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Shell;

import com.clustercontrol.action.FacilityTree;
import com.clustercontrol.bean.Property;
import com.clustercontrol.bean.PropertyConstant;
import com.clustercontrol.bean.RequiredFieldColorConstant;
import com.clustercontrol.composite.PropertySheet;
import com.clustercontrol.dialog.CommonDialog;
import com.clustercontrol.dialog.ValidateResult;
import com.clustercontrol.repository.bean.ScopeConstant;
import com.clustercontrol.repository.util.RepositoryEndpointWrapper;
import com.clustercontrol.repository.util.ScopePropertyUtil;
import com.clustercontrol.util.Messages;
import com.clustercontrol.util.PropertyUtil;
import com.clustercontrol.ws.repository.FacilityDuplicate_Exception;
import com.clustercontrol.ws.repository.InvalidRole_Exception;
import com.clustercontrol.ws.repository.ScopeInfo;

/**
 * スコープの作成・変更ダイアログクラス<BR>
 * 
 * @version 4.0.0
 * @since 1.0.0
 */
public class ScopeCreateDialog extends CommonDialog {

	// ログ
	private static Log m_log = LogFactory.getLog( ScopeCreateDialog.class );

	// ----- instance フィールド ----- //

	/** 初期表示ノードのファシリティID */
	private String facilityId = "";

	/** 親ファシリティID */
	private String parentFacilityId = "";

	/** 変更用ダイアログ判別フラグ */
	private boolean isModifyDialog = false;

	/** ノード属性プロパティシート */
	private PropertySheet propertySheet = null;

	/** ノード属性テーブル **/
	private TableTree table = null;

	// ----- コンストラクタ ----- //

	/**
	 * 指定した形式のダイアログのインスタンスを返します。
	 * 
	 * @param parent
	 *            親のシェルオブジェクト
	 * @param facilityId
	 *            初期表示するノードのファシリティID
	 * @param isModifyDialog
	 *            変更用ダイアログとして利用する場合は、true
	 */
	public ScopeCreateDialog(Shell parent, String facilityId,
			boolean isModifyDialog) {
		super(parent);

		this.facilityId = facilityId;
		this.isModifyDialog = isModifyDialog;
	}

	// ----- instance メソッド ----- //

	/**
	 * ダイアログの初期サイズを返します。
	 * 
	 * @return 初期サイズ
	 */
	@Override
	protected Point getInitialSize() {
		return new Point(450, 400);
	}

	/**
	 * ダイアログエリアを生成します。
	 * 
	 * @param parent
	 *            親のインスタンス
	 */
	@Override
	protected void customizeDialog(Composite parent) {
		Shell shell = this.getShell();

		// タイトル
		shell.setText(Messages
				.getString("dialog.repository.scope.create.modify"));

		// レイアウト
		GridLayout layout = new GridLayout(1, true);
		layout.marginWidth = 10;
		layout.marginHeight = 10;
		parent.setLayout(layout);

		/*
		 * 属性プロパティシート
		 */

		// ラベル
		Label label = new Label(parent, SWT.LEFT);
		GridData gridData = new GridData();
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		gridData.horizontalSpan = 1;
		label.setLayoutData(gridData);
		label.setText(Messages.getString("attribute") + " : ");

		// プロパティシート
		table = new TableTree(parent, SWT.H_SCROLL | SWT.V_SCROLL
				| SWT.FULL_SELECTION | SWT.MULTI | SWT.BORDER);
		gridData = new GridData();
		gridData.horizontalAlignment = GridData.FILL;
		gridData.verticalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		gridData.grabExcessVerticalSpace = true;
		gridData.horizontalSpan = 1;
		table.setLayoutData(gridData);
		table.addSelectionListener(new SelectionAdapter(){
			@Override
			public void widgetSelected(SelectionEvent e) {
				update();
			}
		});


		this.propertySheet = new PropertySheet(table);

		// プロパティ取得及び設定
		Property property = null;
		if (this.isModifyDialog) {
			property = getScopeProperty(this.facilityId, PropertyConstant.MODE_MODIFY);
		} else {
			property = getScopeProperty(null, PropertyConstant.MODE_ADD);
		}
		this.propertySheet.setInput(property);

		// ラインを引く
		Label line = new Label(parent, SWT.SEPARATOR | SWT.HORIZONTAL);
		gridData = new GridData();
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		gridData.horizontalSpan = 1;
		line.setLayoutData(gridData);

		// 画面中央に
		Display display = shell.getDisplay();
		shell.setLocation((display.getBounds().width - shell.getSize().x) / 2,
				(display.getBounds().height - shell.getSize().y) / 2);

		// 必須入力項目を可視化
		this.update();
	}

	/**
	 * 更新処理
	 * 
	 */
	public void update(){
		// 必須項目を明示

		// ファシリティIDのインデックス：0
		if("".equals(table.getItem(0).getText(1))){
			table.getItem(0).setBackground(RequiredFieldColorConstant.COLOR_REQUIRED);
		}else{
			table.getItem(0).setBackground(RequiredFieldColorConstant.COLOR_UNREQUIRED);
		}

		// ファシリティ名のインデックス：1
		if("".equals(table.getItem(1).getText(1))){
			table.getItem(1).setBackground(RequiredFieldColorConstant.COLOR_REQUIRED);
		}else{
			table.getItem(1).setBackground(RequiredFieldColorConstant.COLOR_UNREQUIRED);
		}
	}

	/**
	 * 入力値チェックをします。
	 * 
	 * @return 検証結果
	 */
	@Override
	protected ValidateResult validate() {
		return null;
	}

	/**
	 * 入力値をマネージャに登録します。
	 * 
	 * @return true：正常、false：異常
	 * 
	 * @see com.clustercontrol.dialog.CommonDialog#action()
	 */
	@Override
	protected boolean action() {
		boolean result = false;

		Property property = this.getInputData();
		if(property != null){
			String errMessage = "";
			Property copy = PropertyUtil.copy(property);
			PropertyUtil.deletePropertyDefine(copy);
			if(!this.isModifyDialog()){
				// 作成の場合
				try {
					ScopeInfo scopeinfo = ScopePropertyUtil.property2scope(copy);
					RepositoryEndpointWrapper.addScope(parentFacilityId, scopeinfo);

					// リポジトリキャッシュの更新
					FacilityTree.reflesh();

					MessageDialog.openInformation(
							null,
							Messages.getString("successful"),
							Messages.getString("message.repository.14"));

					result = true;

				} catch (FacilityDuplicate_Exception e) {
					//ファシリティID取得
					ArrayList values = PropertyUtil.getPropertyValue(copy, ScopeConstant.FACILITY_ID);
					String args[] = { (String)values.get(0) };

					// ファシリティIDが重複している場合、エラーダイアログを表示する
					MessageDialog.openInformation(
							null,
							Messages.getString("message"),
							Messages.getString("message.repository.26", args));

				} catch (Exception e) {
					if (e instanceof InvalidRole_Exception) {
						// アクセス権なしの場合、エラーダイアログを表示する
						MessageDialog.openInformation(null, Messages.getString("message"),
								Messages.getString("message.accesscontrol.16"));
					} else {
						errMessage = ", " + e.getMessage();
					}
					MessageDialog.openError(
							null,
							Messages.getString("failed"),
							Messages.getString("message.repository.15") + errMessage);
				}
			} else {
				// 変更の場合
				try {
					ScopeInfo scopeInfo = ScopePropertyUtil.property2scope(copy);
					RepositoryEndpointWrapper.modifyScope(scopeInfo);

					// リポジトリキャッシュの更新
					FacilityTree.reflesh();

					MessageDialog.openInformation(
							null,
							Messages.getString("successful"),
							Messages.getString("message.repository.18"));

					result = true;

				} catch (Exception e) {
					if (e instanceof InvalidRole_Exception) {
						// アクセス権なしの場合、エラーダイアログを表示する
						MessageDialog.openInformation(null, Messages.getString("message"),
								Messages.getString("message.accesscontrol.16"));
					} else {
						errMessage = ", " + e.getMessage();
					}

					MessageDialog.openError(
							null,
							Messages.getString("failed"),
							Messages.getString("message.repository.19") + errMessage);
				}
			}
		}

		return result;
	}

	/**
	 * 変更用ダイアログなのかを返します。
	 * 
	 * @return 変更用ダイアログの場合、true
	 */
	public boolean isModifyDialog() {
		return this.isModifyDialog;
	}

	/**
	 * 入力値を保持したデータモデルを生成します。
	 * 
	 * @return データモデル
	 */
	public Property getInputData() {
		return (Property) this.propertySheet.getInput();
	}

	/**
	 * 親ファシリティIDを返します。
	 * 
	 * @return 親ファシリティID
	 */
	public String getParentFacilityId() {
		return parentFacilityId;
	}

	/**
	 * 親ファシリティIDを設定します。
	 * 
	 * @param parentFacilityId 親ファシリティID
	 */
	public void setParentFacilityId(String parentFacilityId) {
		this.parentFacilityId = parentFacilityId;
	}

	/**
	 * ＯＫボタンのテキストを返します。
	 * 
	 * @return ＯＫボタンのテキスト
	 */
	@Override
	protected String getOkButtonText() {
		return Messages.getString("register");
	}

	/**
	 * キャンセルボタンのテキストを返します。
	 * 
	 * @return キャンセルボタンのテキスト
	 */
	@Override
	protected String getCancelButtonText() {
		return Messages.getString("cancel");
	}

	/**
	 * スコープ属性情報を取得します。<BR>
	 * 
	 * @param facilityId
	 * @param property
	 * @return スコープ属性情報
	 */
	private static Property getScopeProperty(String facilityId, int mode) {

		Property property = null;
		try {
			ScopeInfo scopeInfo = RepositoryEndpointWrapper.getScope(facilityId);
			property = ScopePropertyUtil.scope2property(scopeInfo, mode, Locale.getDefault());
		} catch (InvalidRole_Exception e) {
			// アクセス権なしの場合、エラーダイアログを表示する
			MessageDialog.openInformation(null, Messages.getString("message"),
					Messages.getString("message.accesscontrol.16"));
		} catch (Exception e) {
			m_log.warn("getScopeProperty(), " + e.getMessage(), e);
			MessageDialog.openError(
					null,
					Messages.getString("failed"),
					Messages.getString("message.hinemos.failure.unexpected") + ", " + e.getMessage());
		}
		return property;
	}
}