/*

Copyright (C) 2006 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.

 */

package com.clustercontrol.notify.dialog;

import java.util.ArrayList;
import java.util.List;

import org.eclipse.jface.dialogs.IDialogConstants;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.ModifyEvent;
import org.eclipse.swt.events.ModifyListener;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.graphics.Color;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Combo;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Group;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.Text;

import com.clustercontrol.ClusterControlPlugin;
import com.clustercontrol.bean.PriorityColorConstant;
import com.clustercontrol.bean.PriorityConstant;
import com.clustercontrol.bean.RequiredFieldColorConstant;
import com.clustercontrol.bean.ValidConstant;
import com.clustercontrol.dialog.ScopeTreeDialog;
import com.clustercontrol.dialog.ValidateResult;
import com.clustercontrol.jobmanagement.common.composite.JobIdSelectionListener;
import com.clustercontrol.notify.action.AddNotify;
import com.clustercontrol.notify.action.GetNotify;
import com.clustercontrol.notify.action.ModifyNotify;
import com.clustercontrol.notify.bean.ExecFacilityConstant;
import com.clustercontrol.repository.FacilityPath;
import com.clustercontrol.repository.bean.FacilityConstant;
import com.clustercontrol.util.Messages;
import com.clustercontrol.ws.notify.NotifyInfo;
import com.clustercontrol.ws.notify.NotifyJobInfo;
import com.clustercontrol.ws.repository.FacilityTreeItem;

/**
 * 通知（ジョブ実行）作成・変更ダイアログクラス<BR>
 * 
 * @version 4.0.0
 * @since 3.0.0
 */
public class NotifyJobCreateDialog extends NotifyBasicCreateDialog {

	/** カラム数（重要度）。 */
	private static final int WIDTH_PRIORITY 		= 1;

	/** カラム数（ジョブ実行）。 */
	private static final int WIDTH_JOB_RUN	 		= 1;

	/** カラム数（ジョブユニットID）。 */
	private static final int WIDTH_JOBUNIT_ID	 		= 4;

	/** カラム数（ジョブID）。 */
	private static final int WIDTH_JOB_ID	 		= 4;

	/** カラム数（参照ボタン）。 */
	private static final int WIDTH_REF_BTN 			= 2;

	/** カラム数（通知抑制と連動）。 */
	private static final int WIDTH_INHIBITION_FLG 	= 0;

	/** カラム数（呼出失敗時）。 */
	private static final int WIDTH_FAILURE_PRIORITY = 2;

	/** カラム数（空欄）。 */
	private static final int WIDTH_BLANK 			= 1;

	// ----- instance フィールド ----- //

	/** 通知タイプ
	 * @see com.clustercontrol.bean.NotifyTypeConstant
	 */
	private final int TYPE_JOB = 3;

	/** 入力値を保持するオブジェクト。 */
	private NotifyInfo inputData = null;

	/** 入力値の正当性を保持するオブジェクト。 */
	private ValidateResult validateResult = null;

	/** スコープ用テキスト */
	private Text m_textScope = null;

	/** ファシリティID */
	private String m_facilityId = null;

	/** スコープ */
	private String m_facilityPath = null;

	/** ジョブ変数用ラジオボタン */
	private Button m_radioGenerationNodeValue = null;

	/** 固定値用ラジオボタン */
	private Button m_radioFixedValue = null;

	/** スコープ参照用ボタン */
	private Button m_scopeSelect = null;

	/** 実行（通知） チェックボックス。 */
	private Button m_checkJobRunInfo = null;
	/** 実行（警告） チェックボックス。 */
	private Button m_checkJobRunWarning = null;
	/** 実行（異常） チェックボックス。 */
	private Button m_checkJobRunCritical = null;
	/** 実行（不明） チェックボックス。 */
	private Button m_checkJobRunUnknown = null;

	/** ジョブユニットID（通知） */
	private Text m_textJobunitIdInfo = null;
	/** ジョブユニットID（警告） */
	private Text m_textJobunitIdWarning = null;
	/** ジョブユニットID（危険） */
	private Text m_textJobunitIdCritical = null;
	/** ジョブユニットID（不明） */
	private Text m_textJobunitIdUnknown = null;

	/** ジョブID（通知） テキストボックス。 */
	private Text m_textJobIdInfo = null;
	/** ジョブID（警告） テキストボックス。 */
	private Text m_textJobIdWarning = null;
	/** ジョブID（異常） テキストボックス。 */
	private Text m_textJobIdCritical = null;
	/** ジョブID（不明） テキストボックス。 */
	private Text m_textJobIdUnknown = null;

	/** 参照（通知） ボタン。 */
	private Button m_buttonReferInfo = null;
	/** 参照（警告） ボタン。 */
	private Button m_buttonReferWarning = null;
	/** 参照（異常） ボタン。 */
	private Button m_buttonReferCritical = null;
	/** 参照（不明） ボタン。 */
	private Button m_buttonReferUnknown = null;

	/** 呼出失敗時の重要度（通知） コンボボックス。 */
	private Combo m_comboFailurePriorityInfo = null;
	/** 呼出失敗時の重要度（警告） コンボボックス。 */
	private Combo m_comboFailurePriorityWarning = null;
	/** 呼出失敗時の重要度（異常） コンボボックス。 */
	private Combo m_comboFailurePriorityCritical = null;
	/** 呼出失敗時の重要度（不明） コンボボックス。 */
	private Combo m_comboFailurePriorityUnknown = null;

	// ----- コンストラクタ ----- //

	/**
	 * 作成用ダイアログのインスタンスを返します。
	 * 
	 * @param parent 親のシェルオブジェクト
	 */
	public NotifyJobCreateDialog(Shell parent) {
		super(parent);
	}

	public NotifyJobCreateDialog(Shell parent, int notifyIdType) {
		super(parent, notifyIdType);
	}

	/**
	 * 変更用ダイアログのインスタンスを返します。
	 * 
	 * @param parent 親のシェルオブジェクト
	 * @param notifyId 変更する通知情報の通知ID
	 * @param referenceFlg 参照フラグ（参照のみで変更不可の場合、<code> true </code>）
	 */
	public NotifyJobCreateDialog(Shell parent, String notifyId, boolean referenceFlg) {
		super(parent, notifyId, referenceFlg);
	}

	// ----- instance メソッド ----- //

	/**
	 * ダイアログエリアを生成します。
	 * 
	 * @param parent 親のコンポジット
	 * 
	 * @see com.clustercontrol.notify.dialog.NotifyBasicCreateDialog#customizeDialog(Composite)
	 * @see com.clustercontrol.notify.action.GetNotify#getNotify(String)
	 * @see #setInputData(NotifyInfo)
	 */
	@Override
	protected void customizeDialog(Composite parent) {

		super.customizeDialog(parent);

		// 通知IDが指定されている場合、その情報を初期表示する。
		NotifyInfo info = null;
		if(this.notifyId != null){
			info = new GetNotify().getNotify(this.notifyId);
		}
		else{
			info = new NotifyInfo();
		}
		this.setInputData(info);

	}

	/**
	 * 親のクラスから呼ばれ、各通知用のダイアログエリアを生成します。
	 * 
	 * @param parent 親のコンポジット
	 * 
	 * @see com.clustercontrol.notify.dialog.NotifyBasicCreateDialog#customizeDialog(Composite)
	 */
	@Override
	protected void customizeSettingDialog(Composite parent) {
		final Shell shell = this.getShell();

		// タイトル
		shell.setText(Messages.getString("dialog.notify.job.create.modify"));

		// 変数として利用されるラベル
		Label label = null;
		// 変数として利用されるグリッドデータ
		GridData gridData = null;

		// レイアウト
		GridLayout layout = new GridLayout(1, true);
		layout.marginWidth = 10;
		layout.marginHeight = 10;
		layout.numColumns = 15;
		parent.setLayout(layout);

		/*
		 * ジョブ
		 */
		// ジョブグループ
		Group groupJob = new Group(parent, SWT.NONE);
		layout = new GridLayout(1, true);
		layout.marginWidth = 5;
		layout.marginHeight = 5;
		layout.numColumns = 15;
		groupJob.setLayout(layout);
		gridData = new GridData();
		gridData.horizontalSpan = 15;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		groupJob.setLayoutData(gridData);
		groupJob.setText(Messages.getString("notifies.job"));

		/*
		 * スコープグループ
		 */
		Group groupScope = new Group(groupJob, SWT.NONE);
		groupScope.setText(Messages.getString("notify.job.scope"));
		layout = new GridLayout(1, true);
		layout.marginWidth = 5;
		layout.marginHeight = 5;
		layout.numColumns = 15;
		groupScope.setLayout(layout);
		gridData = new GridData();
		gridData.horizontalSpan = 15;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		groupScope.setLayoutData(gridData);

		// ジョブ変数 ラジオボタン
		this.m_radioGenerationNodeValue = new Button(groupScope, SWT.RADIO);
		gridData = new GridData();
		gridData.horizontalSpan = 15;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		this.m_radioGenerationNodeValue.setLayoutData(gridData);
		this.m_radioGenerationNodeValue.setText(Messages.getString("notify.node.generation") + " : ");
		this.m_radioGenerationNodeValue.addSelectionListener(new SelectionListener() {
			@Override
			public void widgetSelected(SelectionEvent e) {
				Button check = (Button) e.getSource();
				if (check.getSelection()) {
					m_radioFixedValue.setSelection(false);
					m_scopeSelect.setEnabled(false);
				}
			}

			@Override
			public void widgetDefaultSelected(SelectionEvent e) {

			}
		});

		// 固定値 ラジオボタン
		this.m_radioFixedValue = new Button(groupScope, SWT.RADIO);
		gridData = new GridData();
		gridData.horizontalSpan = 3;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		this.m_radioFixedValue.setLayoutData(gridData);
		this.m_radioFixedValue.setText(Messages.getString("notify.node.fix") + " : ");
		this.m_radioFixedValue.addSelectionListener(new SelectionListener() {
			@Override
			public void widgetSelected(SelectionEvent e) {
				Button check = (Button) e.getSource();
				if (check.getSelection()) {
					m_radioGenerationNodeValue.setSelection(false);
					m_scopeSelect.setEnabled(true);
				}
				update();
			}

			@Override
			public void widgetDefaultSelected(SelectionEvent e) {

			}
		});

		this.m_textScope = new Text(groupScope, SWT.BORDER | SWT.READ_ONLY);
		gridData = new GridData();
		gridData.horizontalSpan = 5;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		this.m_textScope.setLayoutData(gridData);
		this.m_textScope.setText("");

		this.m_scopeSelect = new Button(groupScope, SWT.NONE);
		gridData = new GridData();
		gridData.horizontalSpan = 2;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		this.m_scopeSelect.setLayoutData(gridData);
		this.m_scopeSelect.setText(Messages.getString("refer"));
		this.m_scopeSelect.addSelectionListener(new SelectionAdapter() {
			@Override
			public void widgetSelected(SelectionEvent e) {
				ScopeTreeDialog dialog = new ScopeTreeDialog(shell);
				if (dialog.open() == IDialogConstants.OK_ID) {
					FacilityTreeItem selectItem = dialog.getSelectItem();
					if (selectItem.getData().getFacilityType() == FacilityConstant.TYPE_COMPOSITE) {
						m_facilityPath = "";
						m_facilityId = "";
					} else {
						FacilityPath path = new FacilityPath(
								ClusterControlPlugin.getDefault()
								.getSeparator());
						m_facilityPath = path.getPath(selectItem);
						m_facilityId = selectItem.getData().getFacilityId();
					}
					m_textScope.setText(m_facilityPath);
					update();
				}
			}
		});

		label = new Label(groupScope, SWT.NONE);
		gridData = new GridData();
		gridData.horizontalSpan = 5;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		label.setLayoutData(gridData);



		// 空行
		label = new Label(groupJob, SWT.NONE);
		gridData = new GridData();
		gridData.horizontalSpan = 15;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		label.setLayoutData(gridData);

		/*
		 * 重要度 ごとの設定
		 */
		// ラベル（重要度）
		label = new Label(groupJob, SWT.NONE);
		gridData = new GridData();
		gridData.horizontalSpan = WIDTH_PRIORITY;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		label.setLayoutData(gridData);
		label.setText(Messages.getString("priority"));

		// ラベル（実行する）
		label = new Label(groupJob, SWT.NONE);
		gridData = new GridData();
		gridData.horizontalSpan = WIDTH_JOB_RUN;
		gridData.horizontalAlignment = GridData.CENTER;
		gridData.grabExcessHorizontalSpace = true;
		label.setLayoutData(gridData);
		label.setText(Messages.getString("notify.attribute"));

		// ラベル（ジョブユニットID）
		label = new Label(groupJob, SWT.NONE);
		gridData = new GridData();
		gridData.horizontalSpan = WIDTH_JOBUNIT_ID;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		label.setLayoutData(gridData);
		label.setText(Messages.getString("jobunit.id"));

		// ラベル（ジョブID）
		label = new Label(groupJob, SWT.NONE);
		gridData = new GridData();
		gridData.horizontalSpan = WIDTH_JOB_ID;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		label.setLayoutData(gridData);
		label.setText(Messages.getString("job.id"));

		// ラベル（参照）
		label = new Label(groupJob, SWT.NONE);
		gridData = new GridData();
		gridData.horizontalSpan = WIDTH_REF_BTN;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		label.setLayoutData(gridData);

		// ラベル（呼出失敗時）
		label = new Label(groupJob, SWT.NONE);
		gridData = new GridData();
		gridData.horizontalSpan = WIDTH_FAILURE_PRIORITY;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		label.setLayoutData(gridData);
		label.setText(Messages.getString("failure.call.value"));

		// 空欄
		label = new Label(groupJob, SWT.NONE);
		gridData = new GridData();
		gridData.horizontalSpan = WIDTH_BLANK + WIDTH_INHIBITION_FLG;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		label.setLayoutData(gridData);

		//　ジョブ　重要度：情報
		label = this.getLabelPriority(groupJob, Messages.getString("info"),PriorityColorConstant.COLOR_INFO);
		this.m_checkJobRunInfo = this.getCheckJobRun(groupJob);
		this.m_textJobunitIdInfo = this.getTextJobunitId(groupJob);
		this.m_textJobIdInfo = this.getTextJobId(groupJob);
		this.m_buttonReferInfo = this.getButtonJobRefer(groupJob);
		this.m_comboFailurePriorityInfo = this.getComboPriority(groupJob);
		this.m_buttonReferInfo.addSelectionListener(new JobIdSelectionListener(m_textJobunitIdInfo, m_textJobIdInfo));

		// 空欄
		label = new Label(groupJob, SWT.NONE);
		gridData = new GridData();
		gridData.horizontalSpan = WIDTH_BLANK + WIDTH_INHIBITION_FLG;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		label.setLayoutData(gridData);

		//　ジョブ　重要度：警告
		label = this.getLabelPriority(groupJob, Messages.getString("warning"),PriorityColorConstant.COLOR_WARNING);
		this.m_checkJobRunWarning = this.getCheckJobRun(groupJob);
		this.m_textJobunitIdWarning = this.getTextJobunitId(groupJob);
		this.m_textJobIdWarning = this.getTextJobId(groupJob);
		this.m_buttonReferWarning = this.getButtonJobRefer(groupJob);
		this.m_comboFailurePriorityWarning = this.getComboPriority(groupJob);
		this.m_buttonReferWarning.addSelectionListener(new JobIdSelectionListener(m_textJobunitIdWarning, m_textJobIdWarning));

		// 空欄
		label = new Label(groupJob, SWT.NONE);
		gridData = new GridData();
		gridData.horizontalSpan = WIDTH_BLANK + WIDTH_INHIBITION_FLG;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		label.setLayoutData(gridData);

		//　ジョブ　重要度：危険
		label = this.getLabelPriority(groupJob, Messages.getString("critical"),PriorityColorConstant.COLOR_CRITICAL);
		this.m_checkJobRunCritical = this.getCheckJobRun(groupJob);
		this.m_textJobunitIdCritical = this.getTextJobunitId(groupJob);
		this.m_textJobIdCritical = this.getTextJobId(groupJob);
		this.m_buttonReferCritical = this.getButtonJobRefer(groupJob);
		this.m_comboFailurePriorityCritical = this.getComboPriority(groupJob);
		this.m_buttonReferCritical.addSelectionListener(new JobIdSelectionListener(m_textJobunitIdCritical, m_textJobIdCritical));

		// 空欄
		label = new Label(groupJob, SWT.NONE);
		gridData = new GridData();
		gridData.horizontalSpan = WIDTH_BLANK + WIDTH_INHIBITION_FLG;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		label.setLayoutData(gridData);

		//　ジョブ　重要度：不明
		label = this.getLabelPriority(groupJob, Messages.getString("unknown"),PriorityColorConstant.COLOR_UNKNOWN);
		this.m_checkJobRunUnknown = this.getCheckJobRun(groupJob);
		this.m_textJobunitIdUnknown = this.getTextJobunitId(groupJob);
		this.m_textJobIdUnknown = this.getTextJobId(groupJob);
		this.m_buttonReferUnknown = this.getButtonJobRefer(groupJob);
		this.m_comboFailurePriorityUnknown = this.getComboPriority(groupJob);
		this.m_buttonReferUnknown.addSelectionListener(new JobIdSelectionListener(m_textJobunitIdUnknown, m_textJobIdUnknown));

		// 空欄
		label = new Label(groupJob, SWT.NONE);
		gridData = new GridData();
		gridData.horizontalSpan = WIDTH_BLANK + WIDTH_INHIBITION_FLG;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		label.setLayoutData(gridData);

	}

	/**
	 * 更新処理
	 * 
	 */
	public void update(){
		// 必須項目を明示

		// ジョブ実行スコープ
		if(this.m_radioFixedValue.getSelection() && "".equals(this.m_textScope.getText())){
			this.m_textScope.setBackground(RequiredFieldColorConstant.COLOR_REQUIRED);
		}else{
			this.m_textScope.setBackground(RequiredFieldColorConstant.COLOR_UNREQUIRED);
		}

		// 情報
		if(this.m_checkJobRunInfo.getSelection() && "".equals(this.m_textJobunitIdInfo.getText())){
			this.m_textJobunitIdInfo.setBackground(RequiredFieldColorConstant.COLOR_REQUIRED);
			this.m_textJobIdInfo.setBackground(RequiredFieldColorConstant.COLOR_REQUIRED);
		}else{
			this.m_textJobunitIdInfo.setBackground(RequiredFieldColorConstant.COLOR_UNREQUIRED);
			this.m_textJobIdInfo.setBackground(RequiredFieldColorConstant.COLOR_UNREQUIRED);
		}
		// 警告
		if(this.m_checkJobRunWarning.getSelection() && "".equals(this.m_textJobunitIdWarning.getText())){
			this.m_textJobunitIdWarning.setBackground(RequiredFieldColorConstant.COLOR_REQUIRED);
			this.m_textJobIdWarning.setBackground(RequiredFieldColorConstant.COLOR_REQUIRED);
		}else{
			this.m_textJobunitIdWarning.setBackground(RequiredFieldColorConstant.COLOR_UNREQUIRED);
			this.m_textJobIdWarning.setBackground(RequiredFieldColorConstant.COLOR_UNREQUIRED);
		}
		// 危険
		if(this.m_checkJobRunCritical.getSelection() && "".equals(this.m_textJobunitIdCritical.getText())){
			this.m_textJobunitIdCritical.setBackground(RequiredFieldColorConstant.COLOR_REQUIRED);
			this.m_textJobIdCritical.setBackground(RequiredFieldColorConstant.COLOR_REQUIRED);
		}else{
			this.m_textJobunitIdCritical.setBackground(RequiredFieldColorConstant.COLOR_UNREQUIRED);
			this.m_textJobIdCritical.setBackground(RequiredFieldColorConstant.COLOR_UNREQUIRED);
		}
		// 不明
		if(this.m_checkJobRunUnknown.getSelection() && "".equals(this.m_textJobunitIdUnknown.getText())){
			this.m_textJobunitIdUnknown.setBackground(RequiredFieldColorConstant.COLOR_REQUIRED);
			this.m_textJobIdUnknown.setBackground(RequiredFieldColorConstant.COLOR_REQUIRED);
		}else{
			this.m_textJobunitIdUnknown.setBackground(RequiredFieldColorConstant.COLOR_UNREQUIRED);
			this.m_textJobIdUnknown.setBackground(RequiredFieldColorConstant.COLOR_UNREQUIRED);
		}
	}
	/**
	 * 入力値を保持した通知情報を返します。
	 * 
	 * @return 通知情報
	 */
	@Override
	public NotifyInfo getInputData() {
		return this.inputData;
	}

	/**
	 * 引数で指定された通知情報の値を、各項目に設定します。
	 * 
	 * @param notify 設定値として用いる通知情報
	 */
	@Override
	protected void setInputData(NotifyInfo notify) {

		this.inputData = notify;

		// 通知基本情報
		super.m_notifyBasic.setInputData(notify);

		// 初回通知情報
		super.m_notifyInitial.setInputData(notify);

		// 抑制情報
		super.m_notifyInhibition.setInputData(notify);

		// ジョブ実行情報
		List<NotifyJobInfo> list = notify.getNotifyJobInfo();
		if (list != null && list.size() > 0) {
			for (int index=0; index<list.size(); index++){
				NotifyJobInfo job = list.get(index);
				if(job != null){
					// 通知の場合
					if(PriorityConstant.TYPE_INFO == job.getPriority()){
						this.setInputDataForJob(job, m_checkJobRunInfo, m_textJobunitIdInfo, m_textJobIdInfo, m_comboFailurePriorityInfo);
						// 通知のみジョブ通知の共通部分の設定を行う
						if (job.getJobExecFacility() != null) {
							this.m_facilityId = job.getJobExecFacility();
							this.m_textScope.setText(job.getJobExecScope());
						}
						if (job.getJobExecFacilityFlg() != null && job.getJobExecFacilityFlg() == ExecFacilityConstant.TYPE_GENERATION) {
							this.m_radioGenerationNodeValue.setSelection(true);
							this.m_scopeSelect.setEnabled(false);
						}
						else {
							this.m_radioFixedValue.setSelection(true);
							this.m_scopeSelect.setEnabled(true);
						}
					}
					// 警告の場合
					else if(PriorityConstant.TYPE_WARNING == job.getPriority()){
						this.setInputDataForJob(job, m_checkJobRunWarning, m_textJobunitIdWarning, m_textJobIdWarning, m_comboFailurePriorityWarning);
					}
					// 危険の場合
					else if(PriorityConstant.TYPE_CRITICAL == job.getPriority()){
						this.setInputDataForJob(job, m_checkJobRunCritical, m_textJobunitIdCritical, m_textJobIdCritical, m_comboFailurePriorityCritical);
					}
					// 不明の場合
					else if(PriorityConstant.TYPE_UNKNOWN == job.getPriority()){
						this.setInputDataForJob(job, m_checkJobRunUnknown, m_textJobunitIdUnknown, m_textJobIdUnknown, m_comboFailurePriorityUnknown);
					}
				}
			}
		} else {
			// 新規追加の場合
			this.m_radioGenerationNodeValue.setSelection(true);
			this.m_scopeSelect.setEnabled(false);
		}

		// 有効／無効
		if (notify.getValidFlg() != null && notify.getValidFlg() == ValidConstant.TYPE_INVALID) {
			this.m_confirmValid.setSelection(false);
		}
		else {
			this.m_confirmValid.setSelection(true);
		}

		// 入力制御
		if(this.referenceFlg){
			this.setEnabled(false);
		}
		else{
			this.setEnabled(true);
		}

	}

	/**
	 * 引数で指定された通知ジョブ情報の値を、各項目に設定します。
	 * 
	 * @param info 設定値として用いる通知ジョブ情報
	 * @param checkJobRun 通知チェックボックス
	 * @param textJobId ジョブIDテキストボックス
	 * @param checkInhibition 抑制チェックボックス
	 * @param comboFailurePriority 呼出失敗時の重要度
	 */
	private void setInputDataForJob(NotifyJobInfo info,
			Button checkJobRun,
			Text textJobunitId,
			Text textJobId,
			Combo comboFailurePriority
	) {
		// ジョブ実行
		if (info.getValidFlg() != null && info.getValidFlg().intValue() == ValidConstant.TYPE_VALID) {
			checkJobRun.setSelection(true);
		}

		//ジョブユニットID
		if (info.getJobunitId() != null){
			textJobunitId.setText(info.getJobunitId());
		}

		// ジョブID
		if (info.getJobId() != null) {
			textJobId.setText(info.getJobId());
		}
		// ジョブ失敗時の重要度
		if (info.getJobFailurePriority() != null) {
			comboFailurePriority.setText(PriorityConstant.typeToString(info.getJobFailurePriority()));
		}
	}

	/**
	 * 入力値を設定した通知情報を返します。<BR>
	 * 入力値チェックを行い、不正な場合は<code>null</code>を返します。
	 * 
	 * @return 通知情報
	 * 
	 * @see #createInputDataForJob(ArrayList, int, Button, Text, Button, Combo)
	 */
	@Override
	protected NotifyInfo createInputData() {
		NotifyInfo info = new NotifyInfo();

		// 通知タイプの設定
		info.setNotifyType(TYPE_JOB);

		// 通知基本情報
		validateResult = super.m_notifyBasic.createInputData(info, this.notifyId);
		if(validateResult != null){
			return null;
		}

		// 初回通知情報
		validateResult = super.m_notifyInitial.createInputData(info);
		if(validateResult != null){
			return null;
		}

		// 抑制情報
		validateResult = super.m_notifyInhibition.createInputData(info);
		if(validateResult != null){
			return null;
		}

		//　ジョブ情報

		ArrayList<NotifyJobInfo> list = new ArrayList<NotifyJobInfo>();
		list = this.createInputDataForJob(list, PriorityConstant.TYPE_INFO,			this.m_checkJobRunInfo,			this.m_textJobunitIdInfo,			this.m_textJobIdInfo,			this.m_comboFailurePriorityInfo);
		list = this.createInputDataForJob(list, PriorityConstant.TYPE_WARNING,		this.m_checkJobRunWarning,		this.m_textJobunitIdWarning,		this.m_textJobIdWarning,		this.m_comboFailurePriorityWarning);
		list = this.createInputDataForJob(list, PriorityConstant.TYPE_CRITICAL,		this.m_checkJobRunCritical,		this.m_textJobunitIdCritical,		this.m_textJobIdCritical,		this.m_comboFailurePriorityCritical);
		list = this.createInputDataForJob(list, PriorityConstant.TYPE_UNKNOWN,		this.m_checkJobRunUnknown,		this.m_textJobunitIdUnknown,		this.m_textJobIdUnknown,		this.m_comboFailurePriorityUnknown);

		if(list == null){
			return null;
		}

		List<NotifyJobInfo> notifyJobInfo = info.getNotifyJobInfo();
		notifyJobInfo.clear();
		notifyJobInfo.addAll(list);

		// 有効/無効
		if (this.m_confirmValid.getSelection()) {
			info.setValidFlg(ValidConstant.TYPE_VALID);
		} else {
			info.setValidFlg(ValidConstant.TYPE_INVALID);
		}

		return info;
	}

	/**
	 * 入力値を設定した通知ジョブ情報を返します。<BR>
	 * 入力値チェックを行い、不正な場合は<code>null</code>を返します。
	 * 
	 * @return 通知ジョブ情報
	 * 
	 */
	protected ArrayList<NotifyJobInfo> createInputDataForJob(ArrayList<NotifyJobInfo> list,
			int priority,
			Button checkJobRun,
			Text textJobunitId,
			Text textJobId,
			Combo comboFailurePriority
	) {

		if(list == null)
			return null;

		NotifyJobInfo job = new NotifyJobInfo();

		// 重要度
		job.setPriority(priority);

		//　実行
		if (checkJobRun.getSelection()) {
			job.setValidFlg(new Integer(ValidConstant.TYPE_VALID));
		} else {
			job.setValidFlg(new Integer(ValidConstant.TYPE_INVALID));
		}
		// jobunitId
		if (textJobunitId.getText() != null
				&& !"".equals(textJobunitId.getText().trim())) {
			job.setJobunitId(textJobunitId.getText());
		}

		// jobId
		if (textJobId.getText() != null
				&& !"".equals(textJobId.getText().trim())) {
			job.setJobId(textJobId.getText());
		}

		// 呼出失敗時
		if (comboFailurePriority.getText() != null
				&& !"".equals(comboFailurePriority.getText())) {
			job.setJobFailurePriority(PriorityConstant.stringToType(comboFailurePriority.getText()));
		}

		// 共通部分登録
		// 実行ファシリティID
		if (this.m_textScope.getText() != null
				&& !"".equals(this.m_textScope.getText())) {
			job.setJobExecFacility(this.m_facilityId);
			job.setJobExecScope(this.m_textScope.getText());
		}
		// 実行ファシリティ
		if (this.m_radioGenerationNodeValue.getSelection()) {
			job.setJobExecFacilityFlg(ExecFacilityConstant.TYPE_GENERATION);
		}
		else if (this.m_radioFixedValue.getSelection()){
			job.setJobExecFacilityFlg(ExecFacilityConstant.TYPE_FIX);
		}

		list.add(job);
		return list;
	}

	/**
	 * 入力値チェックをします。
	 * 
	 * @return 検証結果
	 */
	@Override
	protected ValidateResult validate() {
		// 入力値生成
		this.inputData = this.createInputData();

		if (this.inputData != null) {
			return super.validate();
		} else {
			return validateResult;
		}
	}

	/**
	 * 入力値をマネージャに登録します。
	 * 
	 * @return true：正常、false：異常
	 * 
	 * @see com.clustercontrol.dialog.CommonDialog#action()
	 */
	@Override
	protected boolean action() {
		boolean result = false;

		NotifyInfo info = this.getInputData();
		if(info != null){
			if(this.notifyId == null){
				// 作成の場合
				result = new AddNotify().add(info);
			}
			else{
				// 変更の場合
				result = new ModifyNotify().modify(info);
			}
		}

		return result;
	}

	/**
	 * ＯＫボタンのテキストを返します。
	 * 
	 * @return ＯＫボタンのテキスト
	 */
	@Override
	protected String getOkButtonText() {
		return Messages.getString("ok");
	}

	/**
	 * キャンセルボタンのテキストを返します。
	 * 
	 * @return キャンセルボタンのテキスト
	 */
	@Override
	protected String getCancelButtonText() {
		return Messages.getString("cancel");
	}

	/**
	 * 無効な入力値の情報を設定します。
	 * 
	 * @param id ID
	 * @param message メッセージ
	 */
	@Override
	protected void setValidateResult(String id, String message) {

		this.validateResult = new ValidateResult();
		this.validateResult.setValid(false);
		this.validateResult.setID(id);
		this.validateResult.setMessage(message);
	}

	/**
	 * ボタンを生成します。<BR>
	 * 参照フラグが<code> true </code>の場合は閉じるボタンを生成し、<code> false </code>の場合は、デフォルトのボタンを生成します。
	 * 
	 * @param parent ボタンバーコンポジット
	 * 
	 * @see #createButtonsForButtonBar(Composite)
	 */
	@Override
	protected void createButtonsForButtonBar(Composite parent) {

		if(!this.referenceFlg){
			super.createButtonsForButtonBar(parent);
		}
		else{
			// 閉じるボタン
			this.createButton(parent, IDialogConstants.CANCEL_ID, IDialogConstants.CLOSE_LABEL, false);
		}
	}

	/**
	 * コンポジットの選択可/不可を設定します。
	 * 
	 * @param enable 選択可の場合、<code> true </code>
	 */
	@Override
	protected void setEnabled(boolean enable) {
		super.m_notifyBasic.setEnabled(enable);
		super.m_notifyInhibition.setEnabled(enable);
	}

	/**
	 * 重要度のラベルを返します。
	 * 
	 * @param parent 親のコンポジット
	 * @param text ラベルに表示するテキスト
	 * @param background ラベルの背景色
	 * @return 生成されたラベル
	 */
	protected Label getLabelPriority(Composite parent,
			String text,
			Color background
	) {

		// ラベル（重要度）
		Label label = new Label(parent, SWT.NONE);
		GridData gridData = new GridData();
		gridData.horizontalSpan = WIDTH_PRIORITY;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		label.setLayoutData(gridData);
		label.setText(text + " : ");
		label.setBackground(background);

		return label;
	}

	/**
	 * ジョブの実行チェックボックスを返します。
	 * 
	 * @param parent 親のコンポジット
	 * @return 生成されたチェックボックス
	 */
	protected Button getCheckJobRun(Composite parent) {

		// チェックボックス（実行）
		Button button = new Button(parent, SWT.CHECK);
		GridData gridData = new GridData();
		gridData.horizontalSpan = WIDTH_JOB_RUN;
		gridData.horizontalAlignment = GridData.CENTER;
		gridData.grabExcessHorizontalSpace = true;
		button.setLayoutData(gridData);
		button.addSelectionListener(new SelectionListener() {
			@Override
			public void widgetSelected(SelectionEvent e) {
				update();
			}

			@Override
			public void widgetDefaultSelected(SelectionEvent e) {

			}
		});

		return button;
	}

	/**
	 * ジョブユニットIDテキストボックスを返します。
	 * 
	 * @param parent 親のコンポジット
	 * @return 生成されたチェックボックス
	 */
	protected Text getTextJobunitId(Composite parent) {

		// テキストボックス（ジョブユニットID）
		Text text = new Text(parent, SWT.BORDER | SWT.READ_ONLY);
		GridData gridData = new GridData();
		gridData.horizontalSpan = WIDTH_JOBUNIT_ID;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		text.setLayoutData(gridData);
		text.addModifyListener(new ModifyListener(){
			@Override
			public void modifyText(ModifyEvent arg0) {
				update();
			}
		});

		return text;
	}

	/**
	 * ジョブジョブIDテキストボックスを返します。
	 * 
	 * @param parent 親のコンポジット
	 * @return 生成されたチェックボックス
	 */
	protected Text getTextJobId(Composite parent) {

		// テキストボックス（ジョブID）
		Text text = new Text(parent, SWT.BORDER | SWT.READ_ONLY);
		GridData gridData = new GridData();
		gridData.horizontalSpan = WIDTH_JOB_ID;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		text.setLayoutData(gridData);

		return text;
	}

	/**
	 * ジョブの参照ボタンを返します。
	 * 
	 * @param parent 親のコンポジット
	 * @return 生成されたチェックボックス
	 */
	protected Button getButtonJobRefer(Composite parent) {

		// チェックボックス（参照）
		Button button = new Button(parent, SWT.NONE);
		GridData gridData = new GridData();
		gridData.horizontalSpan = WIDTH_REF_BTN;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		button.setLayoutData(gridData);
		button.setText(Messages.getString("refer"));

		return button;
	}

	/**
	 * ジョブの重要度のコンボボックスを返します。
	 * 
	 * @param parent 親のコンポジット
	 * @param horizontalSpan コンボボックスのカラム数
	 * @return コンボボックス
	 */
	protected Combo getComboPriority(Composite parent) {

		Combo combo = new Combo(parent, SWT.DROP_DOWN | SWT.READ_ONLY);
		GridData gridData = new GridData();
		gridData.horizontalSpan = WIDTH_FAILURE_PRIORITY;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		combo.setLayoutData(gridData);
		combo.add(PriorityConstant.STRING_CRITICAL);
		combo.add(PriorityConstant.STRING_WARNING);
		combo.add(PriorityConstant.STRING_INFO);
		combo.add(PriorityConstant.STRING_UNKNOWN);
		combo.setText(PriorityConstant.STRING_UNKNOWN);

		return combo;
	}

}