/*

Copyright (C) 2006 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.

 */

package com.clustercontrol.jobmanagement.composite.action;

import org.eclipse.jface.viewers.ISelectionChangedListener;
import org.eclipse.jface.viewers.SelectionChangedEvent;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.ui.IViewPart;
import org.eclipse.ui.IWorkbenchPage;
import org.eclipse.ui.PlatformUI;

import com.clustercontrol.jobmanagement.bean.JobConstant;
import com.clustercontrol.jobmanagement.composite.JobListComposite;
import com.clustercontrol.jobmanagement.composite.JobTreeComposite;
import com.clustercontrol.jobmanagement.util.JobUtil;
import com.clustercontrol.jobmanagement.view.JobListView;
import com.clustercontrol.util.LoginManager;
import com.clustercontrol.ws.jobmanagement.JobTreeItem;

/**
 * ジョブ[一覧]ビューのツリービューア用のSelectionChangedListenerです。
 * 
 * @version 1.0.0
 * @since 1.0.0
 */
public class JobTreeSelectionChangedListener implements ISelectionChangedListener {
	/** ジョブ[一覧]ビューのツリービューア用のコンポジット */
	protected JobTreeComposite m_tree;
	/** ジョブ[一覧]ビューのテーブルビューア用のコンポジット */
	protected JobListComposite m_list;

	/**
	 * コンストラクタ
	 * 
	 * @param tree ジョブ[一覧]ビューのツリービューア用のコンポジット
	 * @param list ジョブ[一覧]ビューのテーブルビューア用のコンポジット
	 */
	public JobTreeSelectionChangedListener(JobTreeComposite tree,
			JobListComposite list) {
		m_tree = tree;
		m_list = list;
	}

	/**
	 * 選択変更時に呼び出されます。<BR>
	 * ジョブ[一覧]ビューのツリービューアを選択した際に、<BR>
	 * 選択したアイテムの内容でジョブ[一覧]ビューのテーブルビューアを更新し、<BR>
	 * 選択したアイテムの内容でジョブ[一覧]ビューのアクションの有効・無効を設定します。
	 * <P>
	 * <ol>
	 * <li>選択変更イベントからジョブツリーアイテムを取得します。</li>
	 * <li>ジョブ[一覧]ビューのツリービューア用のコンポジットに、ジョブツリーアイテムを設定します。</li>
	 * <li>ジョブ[一覧]ビューのアクションの有効・無効を設定します。</li>
	 * <li>ジョブ[一覧]ビューのテーブルビューア用のコンポジットを更新します。</li>
	 * </ol>
	 * 
	 * @param event 選択変更イベント
	 * 
	 * @see org.eclipse.jface.viewers.ISelectionChangedListener#selectionChanged(org.eclipse.jface.viewers.SelectionChangedEvent)
	 */
	@Override
	public void selectionChanged(SelectionChangedEvent event) {
		JobTreeItem selectItem = null;


		//ジョブ[登録]ビューのインスタンスを取得
		IWorkbenchPage page = PlatformUI.getWorkbench()
		.getActiveWorkbenchWindow().getActivePage();
		IViewPart viewPart = page.findView(JobListView.ID);

		//選択アイテムを取得
		StructuredSelection selection = (StructuredSelection) event.getSelection();

		if (viewPart != null && selection != null) {
			selectItem = (JobTreeItem) selection.getFirstElement();
			JobListView view = (JobListView) viewPart.getAdapter(JobListView.class);

			if (selectItem instanceof JobTreeItem) {
				//選択ツリーアイテムを設定
				m_tree.setSelectItem(selectItem);

				// ログインユーザで参照可能なジョブユニットかどうかチェックする
				if (selectItem.getData().getType() == JobConstant.TYPE_JOBUNIT) {

					// 参照できない場合、各種ボタン操作を無効化する
					if (JobUtil.isReferableJobunit(selectItem, LoginManager.getContextManager().getUserId())) {
						view.setEnabledActionAll(true);
						view.setEnabledAction(selectItem.getData().getType(), selection);
					} else {
						view.setEnabledActionAll(false);
					}

				} else {
					//ビューのアクションの有効/無効を設定
					view.setEnabledAction(selectItem.getData().getType(), selection);
				}

			} else {
				//選択ツリーアイテムを設定
				m_tree.setSelectItem(null);

				//ビューのアクションを全て無効に設定
				view.setEnabledAction(-9, selection);
			}

			//ジョブ一覧を更新
			m_list.update(selectItem);
		}

	}
}