/*

Copyright (C) 2006 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.

 */

package com.clustercontrol.accesscontrol.view.action;

import java.util.List;

import org.eclipse.jface.action.IAction;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.ui.IViewActionDelegate;
import org.eclipse.ui.IViewPart;

import com.clustercontrol.accesscontrol.action.GetUserListTableDefine;
import com.clustercontrol.accesscontrol.util.AccessEndpointWrapper;
import com.clustercontrol.accesscontrol.view.UserListView;
import com.clustercontrol.util.Messages;
import com.clustercontrol.ws.access.InvalidRole_Exception;

/**
 * アクセス[ユーザ]ビューの「削除」のアクションクラス<BR>
 * 
 * @version 2.0.0
 * @since 2.0.0
 */
public class UserDeleteAction implements IViewActionDelegate {
	/** アクションID */
	public static final String ID = "com.clustercontrol.accesscontrol.view.action.UserDeleteAction";
	/** ビュー */
	private IViewPart viewPart;

	/**
	 * ビューを保持します。ワークベンチにロードされた際に呼ばれます。
	 * 
	 * @param view ビューのインスタンス
	 * 
	 * @see org.eclipse.ui.IViewActionDelegate#init(org.eclipse.ui.IViewPart)
	 */
	@Override
	public void init(IViewPart viewPart) {
		this.viewPart = viewPart;
	}

	/**
	 * アクセス[ユーザ]ビューの「削除」が押された場合に、<BR>
	 * ユーザを削除します。
	 * <p>
	 * <ol>
	 * <li>アクセス[ユーザ]ビューから選択されているユーザのユーザIDを取得します。</li>
	 * <li>ユーザIDが一致するユーザを削除します。</li>
	 * <li>アクセス[ユーザ]ビューを更新します。</li>
	 * </ol>
	 * 
	 * @see org.eclipse.ui.IActionDelegate#run(org.eclipse.jface.action.IAction)
	 * @see com.clustercontrol.accesscontrol.view.UserListView
	 * @see com.clustercontrol.accesscontrol.action.DeleteUserProperty#delete(String)
	 */
	@Override
	public void run(IAction action) {
		// ユーザ一覧より、選択されているユーザのUIDを取得

		UserListView view = (UserListView) this.viewPart
		.getAdapter(UserListView.class);

		StructuredSelection selection = (StructuredSelection) view
		.getComposite().getTableViewer().getSelection();

		List list = (List) selection.getFirstElement();
		String uid = null;
		if (list != null) {
			uid = (String) list.get(GetUserListTableDefine.UID);
		}

		if (uid != null) {
			// 確認ダイアログにて変更が選択された場合、削除処理を行う。
			String[] args = { uid };
			if (MessageDialog.openConfirm(
					null,
					Messages.getString("confirmed"),
					Messages.getString("message.accesscontrol.13", args))) {

				try {
					// 削除対象のユーザが、ジョブユニットの管理ユーザに設定されているかをチェックする
					List<String> jobunitIdList = AccessEndpointWrapper.checkJobManagementUser(uid);

					if (jobunitIdList.size() != 0) {
						// 管理ユーザに設定されている場合
						MessageDialog.openWarning(null, Messages.getString("message"),
								Messages.getString("message.accesscontrol.24",
										new String[]{ uid, jobunitIdList.toString() }));

					} else {
						try {
							// 削除処理
							AccessEndpointWrapper.deleteUserInfo(uid);
							// 完了メッセージ
							MessageDialog.openInformation(
									null,
									Messages.getString("successful"),
									Messages.getString("message.accesscontrol.11"));

						} catch (Exception e) {
							throw e;
						}
					}
				} catch (Exception e) {
					String errMessage = "";
					if (e instanceof InvalidRole_Exception) {
						// 権限なし
						MessageDialog.openInformation(null, Messages.getString("message"),
								Messages.getString("message.accesscontrol.16"));
					} else {
						errMessage = ", " + e.getMessage();
					}

					// 上記以外の例外
					MessageDialog.openError(
							null,
							Messages.getString("failed"),
							Messages.getString("message.accesscontrol.12") + errMessage);
				}
				// ビューを更新
				view.update();
			}
		}
	}

	/* (non-Javadoc)
	 * @see org.eclipse.ui.IActionDelegate#selectionChanged(org.eclipse.jface.action.IAction, org.eclipse.jface.viewers.ISelection)
	 */
	@Override
	public void selectionChanged(IAction action, ISelection selection) {
	}
}