/*

Copyright (C) 2012 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.

 */

package com.clustercontrol.snmptrap.util;

import java.net.InetAddress;
import java.net.SocketException;
import java.net.UnknownHostException;
import java.util.ArrayList;
import java.util.List;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.opennms.protocols.snmp.SnmpCounter32;
import org.opennms.protocols.snmp.SnmpCounter64;
import org.opennms.protocols.snmp.SnmpGauge32;
import org.opennms.protocols.snmp.SnmpIPAddress;
import org.opennms.protocols.snmp.SnmpInt32;
import org.opennms.protocols.snmp.SnmpObjectId;
import org.opennms.protocols.snmp.SnmpOctetString;
import org.opennms.protocols.snmp.SnmpOpaque;
import org.opennms.protocols.snmp.SnmpPduPacket;
import org.opennms.protocols.snmp.SnmpPduRequest;
import org.opennms.protocols.snmp.SnmpPduTrap;
import org.opennms.protocols.snmp.SnmpSyntax;
import org.opennms.protocols.snmp.SnmpTimeTicks;
import org.opennms.protocols.snmp.SnmpTrapSession;
import org.opennms.protocols.snmp.SnmpUInt32;
import org.opennms.protocols.snmp.SnmpVarBind;

import com.clustercontrol.snmptrap.bean.SnmpTrapV1;
import com.clustercontrol.snmptrap.bean.SnmpVarbind;
import com.clustercontrol.snmptrap.bean.SnmpVarbind.SyntaxType;

/**
 * snmptrapを受信するクラス<br/>
 * 受信したsnmptrapはSnmpTrapV1インスタンスとしてSnmpTrapHandlerの具象クラスに引き渡すことが可能である。<br/>
 * @author takahatat
 */
public class SnmpTrapReceiver implements org.opennms.protocols.snmp.SnmpTrapHandler {

	private static final Log log = LogFactory.getLog(SnmpTrapReceiver.class);

	// 待ち受けアドレスと待ち受けポート
	public final String listenAddress;
	public final int listenPort;

	// joesnmpのsessionインスタンス
	private SnmpTrapSession session;

	// SnmpTrapV1を引き渡すSnmpTrapHandlerインスタンス
	private final SnmpTrapHandler _handler;

	// 各種OID
	public static final String snmpSysuptimeOid = ".1.3.6.1.2.1.1.3.0";
	public static final String snmpSysuptimePrefixOid = ".1.3.6.1.2.1.1.3";

	public static final String snmpTrapOid = ".1.3.6.1.6.3.1.1.4.1.0";
	public static final String snmpTrapEnterpriseOid = ".1.3.6.1.6.3.1.1.4.3.0";
	public static final String snmpTrapsOid = ".1.3.6.1.6.3.1.1.5";

	public static final List<SnmpObjectId> genericTraps;

	public static final String genericTrapColdStartOid = "1.3.6.1.6.3.1.1.5.1";
	public static final String genericTrapWarnStartOid = "1.3.6.1.6.3.1.1.5.2";
	public static final String genericTrapLinkDownOid = "1.3.6.1.6.3.1.1.5.3";
	public static final String genericTrapLinkUpOid = "1.3.6.1.6.3.1.1.5.4";
	public static final String genericTrapAuthFailure = "1.3.6.1.6.3.1.1.5.5";
	public static final String genericTrapEgpNeighborLoss = "1.3.6.1.6.3.1.1.5.6";

	static {
		// initialize generic traps
		genericTraps = new ArrayList<SnmpObjectId>();
		genericTraps.add(new SnmpObjectId(genericTrapColdStartOid));
		genericTraps.add(new SnmpObjectId(genericTrapWarnStartOid));
		genericTraps.add(new SnmpObjectId(genericTrapLinkDownOid));
		genericTraps.add(new SnmpObjectId(genericTrapLinkUpOid));
		genericTraps.add(new SnmpObjectId(genericTrapAuthFailure));
		genericTraps.add(new SnmpObjectId(genericTrapEgpNeighborLoss));
	}

	public SnmpTrapReceiver(String listenAddress, int listenPort, SnmpTrapHandler handler)  {
		this.listenAddress = listenAddress;
		this.listenPort = listenPort;

		this._handler = handler;
	}

	public synchronized void start() throws SocketException, UnknownHostException {
		log.info(String.format("starting SnmpTrapReceiver. [address %s, port = %s, handler = %s",
				listenAddress, listenPort, _handler.getClass().getName()));

		_handler.start();
		session = new SnmpTrapSession(this, listenPort, InetAddress.getByName(listenAddress));
	}

	public synchronized void shutdown() {
		log.info(String.format("stopping SnmpTrapReceiver. [address %s, port = %s, handler = %s",
				listenAddress, listenPort, _handler.getClass().getName()));

		session.close();
		_handler.shutdown();
	}


	/**
	 * SNMP v2を受信した際のハンドラ実装
	 */
	@Override
	public void snmpReceivedTrap(SnmpTrapSession session, InetAddress addr,
			int port, SnmpOctetString community, SnmpPduPacket pdu) {

		long receivedTime = System.currentTimeMillis();

		if (log.isDebugEnabled()) {
			log.debug("SnmpTrap v2 received from agent " + addr + " on port " + port);
			log.debug("SnmpTrap v2 : community = " + community);
			log.debug("SnmpTrap v2 PDU : command = " + pdu.getCommand());
			log.debug("SnmpTrap v2 PDU : ID = " + pdu.getRequestId());
			log.debug("SnmpTrap v2 PDU : Length = " + pdu.getLength());

			if (pdu instanceof SnmpPduRequest) {
				log.debug("SnmpTrap v2 PDU : Error Status = " + ((SnmpPduRequest)pdu).getErrorStatus());
				log.debug("SnmpTrap v2 PDU : Error Index = " + ((SnmpPduRequest)pdu).getErrorIndex());
			}
		}

		SnmpTrapV1 snmptrap = convertV2toV1(receivedTime, community.toString(), addr, pdu);
		if (snmptrap == null) {
			log.info("dicarding unexpected SnmpTrap v2 received from host " + addr);
		} else {
			if (log.isDebugEnabled()) {
				log.debug("start filtering : " + snmptrap);
			}
			_handler.snmptrapReceived(snmptrap);
		}

	}



	/**
	 * SNMP v1を受信した際のハンドラ実装
	 */
	@Override
	public void snmpReceivedTrap(SnmpTrapSession session, InetAddress addr,
			int port, SnmpOctetString community, SnmpPduTrap pdu) {

		long receivedTime = System.currentTimeMillis();

		if (log.isDebugEnabled()) {
			log.debug("SnmpTrap v1 received from agent " + addr + " on port " + port);
			log.debug("SnmpTrap v1 : community = " + community);
			log.debug("SnmpTrap v1 PDU : IP ADDRESS = " + pdu.getAgentAddress());
			log.debug("SnmpTrap v1 PDU : Enterprise Id = " + pdu.getEnterprise());
			log.debug("SnmpTrap v1 PDU : Generic = " + pdu.getGeneric());
			log.debug("SnmpTrap v1 PDU : Specific = " + pdu.getSpecific());
			log.debug("SnmpTrap v1 PDU : Timestamp = " + pdu.getTimeStamp());
			log.debug("SnmpTrap v1 PDU : Length = " + pdu.getLength());
		}

		List<SnmpVarbind> varbinds = new ArrayList<SnmpVarbind>();
		for (int i = 0; i < pdu.getLength(); i++) {
			try {
				SnmpVarBind varbind = pdu.getVarBindAt(i);
				SnmpSyntax value = varbind.getValue();

				if (value instanceof SnmpCounter32) {
					varbinds.add(new SnmpVarbind(SyntaxType.Counter32, value.toString().getBytes()));
				} else if (value instanceof SnmpCounter64) {
					varbinds.add(new SnmpVarbind(SyntaxType.Counter64, value.toString().getBytes()));
				} else if (value instanceof SnmpGauge32) {
					varbinds.add(new SnmpVarbind(SyntaxType.Gauge32, value.toString().getBytes()));
				} else if (value instanceof SnmpInt32) {
					varbinds.add(new SnmpVarbind(SyntaxType.Int32, value.toString().getBytes()));
				} else if (value instanceof SnmpIPAddress) {
					varbinds.add(new SnmpVarbind(SyntaxType.IPAddress, ((SnmpIPAddress) value).getString()));
				} else if (value instanceof SnmpObjectId) {
					varbinds.add(new SnmpVarbind(SyntaxType.ObjectId, value.toString().getBytes()));
				} else if (value instanceof SnmpOctetString) {
					varbinds.add(new SnmpVarbind(SyntaxType.OctetString, ((SnmpOctetString) value).getString()));
				} else if (value instanceof SnmpOpaque) {
					varbinds.add(new SnmpVarbind(SyntaxType.Opaque, ((SnmpOpaque) value).getString()));
				} else if (value instanceof SnmpTimeTicks) {
					varbinds.add(new SnmpVarbind(SyntaxType.TimeTicks, value.toString().getBytes()));
				} else if (value instanceof SnmpUInt32) {
					varbinds.add(new SnmpVarbind(SyntaxType.UnsignedInt32, value.toString().getBytes()));
				} else {
					varbinds.add(new SnmpVarbind(SyntaxType.Null, "".getBytes()));
				}

				if (log.isDebugEnabled()) {
					log.debug("SnmpTrap v2 PDU (varbind[" + i +  "]) : name = " + varbind.getName().toString() + ", value = " + value.toString());
				}
			} catch (Exception e) {
				log.warn("SnmpTrap v2 PDU (varbind " + i + " access failure) : Length = " + pdu.getLength()
						+ " [" + e.getMessage() + "]", e);
			}
		}

		SnmpTrapV1 snmptrap = new SnmpTrapV1(receivedTime, community.toString(), addr.getHostAddress(),
				pdu.getEnterprise().toString(), pdu.getGeneric(), pdu.getSpecific(),
				pdu.getTimeStamp(), varbinds);
		if (log.isDebugEnabled()) {
			log.debug("start filtering : " + snmptrap);
		}
		_handler.snmptrapReceived(snmptrap);

	}

	/**
	 * 受信時に異常が発生した際のハンドラ実装
	 */
	@Override
	public void snmpTrapSessionError(SnmpTrapSession session, int error, Object ref) {
		log.warn("An error occurred in snmptrap session. (error code = " + error + ")");
		if (ref != null) {
			log.warn("snmptrap session error : reference = " + ref.toString());
		}

	}

	/**
	 * SNMPTRAP v2をv1にマッピングさせる。
	 * 
	 * <p>
	 * 抜粋：RFC2089 ('Mapping SNMPv2 onto SNMPv1'), section 3.3 ('Processing an outgoing SNMPv2 TRAP')
	 * </p>
	 * 
	 * <p>
	 * <strong>2b </strong>
	 * <p>
	 * If the snmpTrapOID.0 value is one of the standard traps the specific-trap
	 * field is set to zero and the generic trap field is set according to this
	 * mapping:
	 * <p>
	 * 
	 * <pre>
	 * 
	 *       value of snmpTrapOID.0                generic-trap
	 *       ===============================       ============
	 *       1.3.6.1.6.3.1.1.5.1 (coldStart)                  0
	 *       1.3.6.1.6.3.1.1.5.2 (warmStart)                  1
	 *       1.3.6.1.6.3.1.1.5.3 (linkDown)                   2
	 *       1.3.6.1.6.3.1.1.5.4 (linkUp)                     3
	 *       1.3.6.1.6.3.1.1.5.5 (authenticationFailure)      4
	 *       1.3.6.1.6.3.1.1.5.6 (egpNeighborLoss)            5
	 * 
	 * </pre>
	 * 
	 * <p>
	 * The enterprise field is set to the value of snmpTrapEnterprise.0 if this
	 * varBind is present, otherwise it is set to the value snmpTraps as defined
	 * in RFC1907 [4].
	 * </p>
	 * 
	 * <p>
	 * <strong>2c. </strong>
	 * </p>
	 * <p>
	 * If the snmpTrapOID.0 value is not one of the standard traps, then the
	 * generic-trap field is set to 6 and the specific-trap field is set to the
	 * last subid of the snmpTrapOID.0 value.
	 * </p>
	 * 
	 * <p>
	 * If the next to last subid of snmpTrapOID.0 is zero, then the enterprise
	 * field is set to snmpTrapOID.0 value and the last 2 subids are truncated
	 * from that value. If the next to last subid of snmpTrapOID.0 is not zero,
	 * then the enterprise field is set to snmpTrapOID.0 value and the last 1
	 * subid is truncated from that value.
	 * </p>
	 * 
	 * <p>
	 * In any event, the snmpTrapEnterprise.0 varBind (if present) is ignored in this case.
	 * </p>
	 * 
	 * @param agent The address of the remote sender.
	 * @param pdu The decoded V2 trap pdu.
	 * @return トラップOID情報
	 */
	private SnmpTrapV1 convertV2toV1(long receivedTime, String community, InetAddress agentAddr, SnmpPduPacket pdu) {

		if (pdu.typeId() != (byte)SnmpPduPacket.V2TRAP) {
			// pduがV2でない場合はnullを返す
			log.info("SnmpTrap v2 PDU (command must be " + SnmpPduPacket.V2TRAP + ") : Command = " + pdu.getCommand());
			return null;
		}

		if (pdu.getLength() < 2) {
			// pduのvarbind数が2より少ない場合はnullを返す
			log.info("SnmpTrap v2 PDU (sysUpTime.0 and snmpTrapOID.0 must be contained) : Length = " + pdu.getLength());
			return null;
		}

		// 1番目はsysuptime
		SnmpVarBind varBind0 = pdu.getVarBindAt(0);

		String varBindName0 = varBind0.getName().toString();
		if (varBindName0 != null && varBindName0.startsWith(snmpSysuptimePrefixOid)) {
			// 一部の機器(extremeなどで末尾に0が付与されていない場合がある)への対策
			varBindName0 = snmpSysuptimeOid;
		} else {
			log.info("SnmpTrap v2 PDU (first varbind must be sysUpTime.0) : varBindName0 = " + varBindName0);
			return null;
		}

		long timestamp = receivedTime;
		SnmpSyntax varBindValue0 = varBind0.getValue();
		if (varBindValue0 instanceof SnmpTimeTicks) {
			timestamp = ((SnmpTimeTicks)varBindValue0).getValue();
		} else {
			log.info("SnmpTrap v2 PDU (first varbind must be timeticks) : varBindValue0 = " + varBindValue0.toString());
			return null;
		}

		// 2番目はSNMPトラップOID
		SnmpVarBind varBind1 = pdu.getVarBindAt(1);

		String varBindName1 = varBind1.getName().toString();
		if (! snmpTrapOid.equals(varBindName1)) {
			log.info("SnmpTrap v2 PDU (first varbind must be snmpTrapOID.0) : varBindName1 = " + varBindName1);
			return null;
		}

		SnmpObjectId oidObj = null;
		String oid = "";
		SnmpSyntax varBindValue1 = varBind1.getValue();
		if (varBindValue1 instanceof SnmpObjectId) {
			oidObj = (SnmpObjectId)varBindValue1;
			oid = oidObj.toString();
			if (! oid.startsWith(".")) {
				oid = "." + oid;
			}
		} else {
			log.info("SnmpTrap v2 PDU (second varbind must be object id) : varBindValue1 = " + varBindValue1.toString());
			return null;
		}

		int dotIdx = oid.lastIndexOf(".");
		int index = -1;
		try {
			index = Integer.parseInt(oid.substring(dotIdx + 1));
			if (log.isDebugEnabled()) {
				log.debug("SnmpTrap v2 PDU (oid) : index = " + index);
			}
		} catch (Exception e) {
			log.warn("SnmpTrap v2 PDU (oid) : snmpTrapOID.0 = " + oid, e);
		}

		String enterpriseId = null;
		int genericId = -1;
		int specificId = -1;
		if (genericTraps.contains(oidObj)) {
			if (log.isDebugEnabled()) {
				log.debug("SnmpTrap v2 PDU (this is Generic Trap) : oid = " + oid);
			}

			// スタンダードトラップの場合
			genericId = index - 1;
			specificId = 0;

			for (int i = 0; i < pdu.getLength(); i++) {
				SnmpVarBind varBind = pdu.getVarBindAt(i);
				String name = varBind.getName().toString();

				if (snmpTrapEnterpriseOid.equals(name)) {
					SnmpSyntax value = varBind.getValue();
					if (value instanceof SnmpObjectId) {
						if (log.isDebugEnabled()) {
							log.debug("SnmpTrap v2 PDU (SnmpObjectId found) : enterpriseId = " + value.toString());
						}

						// SNMPTRAP_ENTERPRISE_OIDの1番目を採用する(複数定義されている可能性もある)
						enterpriseId = value.toString();
						break;
					}
				}

				if (enterpriseId == null) {
					// varbindの値にSNMPトラップ値をセット（RFC 1907）
					enterpriseId = snmpTrapsOid + "." + oid.charAt(oid.length() - 1);
				}
			}

			if (log.isDebugEnabled()) {
				log.debug("SnmpTrap v2 PDU : enterpriseId = " + enterpriseId + ", genericId = " + genericId + ", specificId = " + specificId);
			}
		} else {
			if (log.isDebugEnabled()) {
				log.debug("SnmpTrap v2 PDU (this is not Generic Trap) : oid = " + oid);
			}

			// スタンダードでないトラップの場合
			genericId = 6;
			specificId = index;

			int nextDotIdx = oid.lastIndexOf(".", dotIdx - 1);
			String nextIndex = oid.substring(nextDotIdx + 1, dotIdx);

			if ("0".equals(nextIndex)) {
				enterpriseId = oid.substring(0, nextDotIdx);
			} else {
				enterpriseId = oid.substring(0, dotIdx);
			}

			if (log.isDebugEnabled()) {
				log.debug("SnmpTrap v2 PDU : enterpriseId = " + enterpriseId + ", genericId = " + genericId + ", specificId = " + specificId);
			}
		}

		List<SnmpVarbind> varbinds = new ArrayList<SnmpVarbind>();
		// varbind(v2c)の1番目はsysuptime
		// varbind(v2c)の2番目はSNMPトラップOID
		// varbind(v2c)の3番目以降をvarbinds(v1)に入れる。
		for (int i = 2; i < pdu.getLength(); i++) {
			try {
				SnmpVarBind varbind = pdu.getVarBindAt(i);
				SnmpSyntax value = varbind.getValue();

				if (value instanceof SnmpCounter32) {
					varbinds.add(new SnmpVarbind(SyntaxType.Counter32, value.toString().getBytes()));
				} else if (value instanceof SnmpCounter64) {
					varbinds.add(new SnmpVarbind(SyntaxType.Counter64, value.toString().getBytes()));
				} else if (value instanceof SnmpGauge32) {
					varbinds.add(new SnmpVarbind(SyntaxType.Gauge32, value.toString().getBytes()));
				} else if (value instanceof SnmpInt32) {
					varbinds.add(new SnmpVarbind(SyntaxType.Int32, value.toString().getBytes()));
				} else if (value instanceof SnmpOpaque) {
					varbinds.add(new SnmpVarbind(SyntaxType.Opaque, value.toString().getBytes()));
				} else if (value instanceof SnmpIPAddress) {
					varbinds.add(new SnmpVarbind(SyntaxType.IPAddress, value.toString().getBytes()));
				} else if (value instanceof SnmpObjectId) {
					varbinds.add(new SnmpVarbind(SyntaxType.ObjectId, value.toString().getBytes()));
				} else if (value instanceof SnmpTimeTicks) {
					varbinds.add(new SnmpVarbind(SyntaxType.TimeTicks, value.toString().getBytes()));
				} else if (value instanceof SnmpOctetString) {
					varbinds.add(new SnmpVarbind(SyntaxType.OctetString, ((SnmpOctetString) value).getString()));
				} else if (value instanceof SnmpUInt32) {
					varbinds.add(new SnmpVarbind(SyntaxType.UnsignedInt32, value.toString().getBytes()));
				} else {
					varbinds.add(new SnmpVarbind(SyntaxType.Null, "".getBytes()));
				}

				if (log.isDebugEnabled()) {
					log.debug("SnmpTrap v2 PDU (varbind[" + i +  "]) : name = " + varbind.getName().toString() + ", value = " + value.toString());
				}
			} catch (Exception e) {
				log.warn("SnmpTrap v2 PDU (varbind " + i + " access failure) : Length = " + pdu.getLength()
						+ " [" + e.getMessage() + "]", e);
			}
		}

		return new SnmpTrapV1(receivedTime, community, agentAddr.getHostAddress(), enterpriseId, genericId, specificId, timestamp, varbinds);
	}

	public static byte[] getByteFromVarbind(SnmpSyntax varbind) {
		if (varbind == null) {
			if (log.isDebugEnabled()) {
				log.debug("varbind value is null, use as 0-length string.");
			}
			return "".getBytes();
		}

		if (varbind instanceof SnmpOctetString) {
			return ((SnmpOctetString)varbind).getString();
		} else {
			if (log.isDebugEnabled()) {
				log.debug("not octet string, skip decoding : " + varbind.toString());
			}
			return varbind.toString().getBytes();
		}
	}

}
