/*

Copyright (C) 2010 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.

 */

package com.clustercontrol.selfcheck.monitor;

import java.util.ArrayList;
import java.util.Date;
import java.util.List;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.poller.bean.PollerProtocolConstant;
import com.clustercontrol.poller.impl.SnmpPollerImpl;
import com.clustercontrol.sharedtable.DataTable;
import com.clustercontrol.sharedtable.TableEntry;
import com.clustercontrol.util.apllog.AplLogger;

/**
 * メモリのスワップアウト発生を確認する処理の実装クラス
 */
public class RAMSwapOutMonitor extends SelfCheckMonitorBase {

	private static Log m_log = LogFactory.getLog( RAMSwapOutMonitor.class );

	public final String monitorId = "SYS_SWAPOUT";
	public final String subKey = "";
	public final String application = "SELFCHECK (swap-out)";

	public final long intervalMSec;

	public final int snmpPort;
	public final String snmpVersion;
	public final String snmpCommunity;
	public final int snmpRetries;
	public final int snmpTimeout;

	public final String SNMP_POLLING_IPADDRESS = "127.0.0.1";
	public final String POLLING_TARGET_OID  = ".1.3.6.1.4.1.2021.11.63";
	public final String RAW_SWAP_OUT_OUT_OID = ".1.3.6.1.4.1.2021.11.63.0";

	private static volatile TableEntry previousMibValue = null;

	/**
	 * コンストラクタ
	 * @param snmpPort SNMPポート番号
	 * @param snmpVersion SNMPバージョン
	 * @param snmpCommunity SNMPコミュニティ名
	 * @param snmpRetries SNMPリトライ回数
	 * @param snmpTimeout SNMPタイムアウト[msec]
	 */
	public RAMSwapOutMonitor(int interval, int snmpPort, String snmpVersion, String snmpCommunity, int snmpRetries, int snmpTimeout) {
		/** メイン処理 */
		this.intervalMSec = interval * 1000;

		this.snmpPort = snmpPort;
		this.snmpVersion = snmpVersion;
		this.snmpCommunity = snmpCommunity;
		this.snmpRetries = snmpRetries;
		this.snmpTimeout = snmpTimeout;
	}

	/**
	 * セルフチェック処理名
	 */
	@Override
	public String toString() {
		return "monitoring ram's swap-out.";
	}

	/**
	 * 監視項目ID
	 */
	@Override
	public String getMonitorId() {
		return monitorId;
	}

	/**
	 * 監視項目subKey
	 */
	@Override
	public String getSubKey() {
		return subKey;
	}

	/**
	 * スワップアウトの発生を確認する処理
	 * @return 通知情報（アプリケーション名は未格納）
	 */
	@Override
	public void execute() {
		/** ローカル変数 */
		long swapOutSize = 0;
		long lastUpdateTime = 0;
		boolean warn = true;

		/** メイン処理 */
		if (m_log.isDebugEnabled()) m_log.debug("monitoring swap-out.");

		// 最終収集日時を取得する
		if (previousMibValue != null) {
			lastUpdateTime = previousMibValue.getDate();
		}

		// 利用可能なヒープ容量をMByte単位で取得する
		swapOutSize = getSwapOut();

		if (swapOutSize < 0) {
			m_log.info("skipped monitoring swap-out.");
			return;
		} else if (swapOutSize == 0) {
			m_log.debug("swap-out does not occurred.");
			warn = false;
		}

		if (warn) {
			m_log.info("swap-out occurred. (swapOutSize=" + swapOutSize + ")");
		}
		if (!isNotify(warn)) {
			return;
		}
		String[] msgAttr1 = { new Long(swapOutSize).toString() };
		AplLogger aplLogger = new AplLogger(PLUGIN_ID, APL_ID);
		aplLogger.put(MESSAGE_ID, "005", msgAttr1,
				"ram swap-out(" +
						swapOutSize +
						" [blocks]) occurred since " +
						String.format("%1$tY-%1$tm-%1$td %1$tH:%1$tM:%1$tS", new Date(lastUpdateTime)) +
				".");

		return;
	}

	/**
	 * Swap Out のカウント値の前回ポーリング時との差分を求めます。
	 * ポーリングに失敗した場合、前回収集から収集間隔の2倍以上経過している場合は、
	 * -1を返します。
	 */
	private long getSwapOut(){
		/** ローカル変数 */
		int version = 0;

		SnmpPollerImpl poller = null;
		List<String> oidList = null;
		DataTable dataTable = null;
		TableEntry entry = null;

		long now = 0;
		long lastUpdateTime = 0;

		long previousSwapOut = 0;
		long currentSwapOut = 0;

		long swapOut = -1;

		/** メイン処理 */

		// SNMPのバージョン変換
		if("1".equals(snmpVersion)){
			version = 0;
		} else if("2c".equals(snmpVersion)){
			version = 1;
		} else {
			// それ以外の場合はv2cに設定
			version = 1;
		}

		try {
			// 収集対象のOID
			oidList = new ArrayList<String>();
			oidList.add(POLLING_TARGET_OID);

			// ポーラを生成してポーリングを実行
			poller = new SnmpPollerImpl();
			dataTable = poller.polling(
					SNMP_POLLING_IPADDRESS,
					snmpPort,
					version,
					snmpCommunity,
					snmpRetries,
					snmpTimeout,
					oidList,
					false);

			entry = dataTable.getValue(getEntryKey(RAW_SWAP_OUT_OUT_OID));

			// 前回収集値と比較する
			if (previousMibValue != null) {
				now = System.currentTimeMillis();

				// 前回収集からの経過時間を算出
				lastUpdateTime = previousMibValue.getDate();

				// 収集間隔の2倍以上古い場合は判定しない
				if ((now - lastUpdateTime) < intervalMSec * 2) {
					previousSwapOut = (Long)previousMibValue.getValue();
					currentSwapOut = (Long)entry.getValue();

					// 単純に差分を求める（カウンタ値の桁溢れは考慮しない）
					swapOut = currentSwapOut - previousSwapOut;
				}
			}

			// 次回のために、現在値を前回収集値として格納する
			previousMibValue = entry;

		} catch (Exception e) {
			m_log.warn("failed to snmp polling.", e);
		}

		// 取得できなかった場合、もしくは判定できなかった場合
		return swapOut;
	}

	/**
	 * OIDをTableEntryのキーに変換する
	 */
	private String getEntryKey(String oidString){
		return PollerProtocolConstant.PROTOCOL_SNMP + "." + oidString;
	}

	/**
	 * 単体試験用
	 */
	public static void main(String[] args) {
		long swapOut = new RAMSwapOutMonitor(60, 161, "2c", "public", 1, 3000).getSwapOut();

		try {
			Thread.sleep(1500);
		} catch (InterruptedException e) {
			e.printStackTrace();
		}

		swapOut = new RAMSwapOutMonitor(60, 161, "2c", "public", 1, 3000).getSwapOut();

		System.out.println("Usage : " + swapOut);
	}

}
