/*

Copyright (C) 2012 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.

 */

package com.clustercontrol.plugin.impl;

import java.io.File;
import java.io.FileInputStream;
import java.io.Serializable;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.Properties;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.commons.util.JpaTransactionManager;
import com.clustercontrol.commons.util.MonitoredThreadPoolExecutor;
import com.clustercontrol.fault.HinemosUnknown;
import com.clustercontrol.plugin.api.AsyncTaskFactory;
import com.clustercontrol.plugin.api.HinemosPlugin;
import com.clustercontrol.plugin.util.TaskExecutionAfterCommitCallback;

/**
 * 通知処理などの非同期処理の実行制御や永続化制御を管理するプラグインサービス<br/>
 * @author takahatat
 */
public class AsyncWorkerPlugin implements HinemosPlugin {

	public static final Log log = LogFactory.getLog(AsyncWorkerPlugin.class);

	// 非同期処理の定義ファイル
	public static final String _configFilePath;
	public static final String _configFilePathDefault = System.getProperty("hinemos.manager.etc.dir") + File.separator + "async-worker.properties";

	// 非同期処理の処理スレッド数（デフォルト値）
	public static final int _threadSizeDefault = 1;
	// 非同期処理の最大待ち処理数（デフォルト値）
	public static final int _queueSizeDefault = 20000;
	// 停止時に残留している処理の最大処理時間（デフォルト値）
	public static final long _shutdownTimeoutDefault = 10000;

	// 定義ファイルのキー定義
	public static final String _keyPrefix = "worker.";
	public static final String _keyWorkerList = _keyPrefix + "list";
	public static final String _keyPostfixFactoryClassName = ".factoryclass";
	public static final String _keyPostfixThreadSize = ".thread.size";
	public static final String _keyPostfixQueueSize = ".queue.size";
	public static final String _keyPostfixShutdownTimeout = ".shutdown.timeout";

	// 定義ファイルに対応するプロパティクラス
	private static final Properties _properties = new Properties();

	// 非同期処理のWorker一覧
	private static String[] workers = null;
	// 各Workerに対応するExecutor一覧
	private static final Map<String, ThreadPoolExecutor> _executorMap = new HashMap<String, ThreadPoolExecutor>();
	// 各Workerに対応するRunnable生成クラス一覧
	private static final Map<String, AsyncTaskFactory> _factoryMap = new HashMap<String, AsyncTaskFactory>();
	// 各Workerに対応する停止時最大処理時間の一覧
	private static final Map<String, Long> _shutdownTimeoutMap = new HashMap<String, Long>();
	// 各Workerに対応する払い出し処理IDの一覧
	private static final Map<String, Long> _nextTaskIdMap = new HashMap<String, Long>();

	// 排他制御用のLockオブジェクト
	private static final Map<String, Object> _executorLock = new HashMap<String, Object>();
	private static final Map<String, Object> _counterLock = new HashMap<String, Object>();

	static {
		// 定義ファイルのファイルパスを取得
		_configFilePath = System.getProperty("hinemos.asynctask.file", _configFilePathDefault);
	}

	@Override
	public Set<String> getDependency() {
		Set<String> dependency = new HashSet<String>();
		dependency.add(Log4jReloadPlugin.class.getName());
		return dependency;
	}

	@Override
	public void create() {
		// 定義ファイルをPropertiesに読み込む
		FileInputStream fin = null;
		try {
			fin = new FileInputStream(_configFilePath);
			_properties.load(fin);
		} catch (Exception e) {
			log.warn("config file loading failure. (" + _configFilePath + ")", e);
		} finally {
			if (fin != null) {
				try {
					fin.close();
				} catch (Exception e) {
					log.warn("file closing failure.", e);
				}
			}
		}

	}

	@Override
	public void activate() {
		if (!_properties.containsKey(_keyWorkerList)) {
			return;
		}
		String workerList = _properties.getProperty(_keyWorkerList);
		workers = workerList.split(",");

		for (String worker : workers) {
			_executorLock.put(worker, new Object());
			_counterLock.put(worker, new Object());

			synchronized (_executorLock.get(worker)) {
				String className = _properties.getProperty(_keyPrefix + worker + _keyPostfixFactoryClassName);
				if (className == null || "".equals(className)) {
					log.warn("class not defined. (" + _keyPrefix + worker + _keyPostfixFactoryClassName + ")");
				}
				try {
					@SuppressWarnings("rawtypes")
					Class clazz = Class.forName(className);

					if (clazz.newInstance() instanceof AsyncTaskFactory) {
						AsyncTaskFactory taskFactory = (AsyncTaskFactory)clazz.newInstance();
						_factoryMap.put(worker, taskFactory);
						_nextTaskIdMap.put(worker, 0L);
					} else {
						log.warn("class is not sub class of AsyncTaskFactory. (" + className + ")");
						continue;
					}
				} catch (ClassNotFoundException e) {
					log.warn("class not found. (" + className + ")", e);
					continue;
				} catch (Exception e) {
					log.warn("instantiation failure. (" + className + ")", e);
					continue;
				}

				String threadSizeStr = _properties.getProperty(_keyPrefix + worker + _keyPostfixThreadSize);
				int threadSize = _threadSizeDefault;
				try {
					threadSize = Integer.parseInt(threadSizeStr);
				} catch (NumberFormatException e) {
					log.warn("parameter is not integer. (" + _keyPrefix + worker + _keyPostfixThreadSize + " = " + threadSizeStr + ")", e);
				}

				String queueSizeStr = _properties.getProperty(_keyPrefix + worker + _keyPostfixQueueSize);
				int queueSize = _queueSizeDefault;
				try {
					queueSize = Integer.parseInt(queueSizeStr);
				} catch (NumberFormatException e) {
					log.warn("parameter is not integer. (" + _keyPrefix + worker + _keyPostfixQueueSize + " = " + queueSizeStr + ")", e);
				}

				log.info("activating asynchronous worker. (worker = " + worker + ", class = " + className +
						", threadSize = " + threadSize + ", queueSize = " + queueSize + ")");

				ThreadPoolExecutor executor = new MonitoredThreadPoolExecutor(threadSize, threadSize,
						0L, TimeUnit.MILLISECONDS,
						new LinkedBlockingQueue<Runnable>(queueSize),
						new AsyncThreadFactory(worker), new TaskRejectionHandler(worker));

				_executorMap.put(worker, executor);

				String shutdownTimeoutStr = _properties.getProperty(_keyPrefix + worker + _keyPostfixShutdownTimeout);
				long shutdownTimeout = _shutdownTimeoutDefault;
				try {
					shutdownTimeout = Long.parseLong(shutdownTimeoutStr);
				} catch (NumberFormatException e) {
					log.warn("parameter is not long. (" + _keyPrefix + worker + _keyPostfixShutdownTimeout + " = " + shutdownTimeoutStr + ")", e);
				}

				_shutdownTimeoutMap.put(worker, shutdownTimeout);

				runPersistedTask(worker);
			}
		}
	}

	@Override
	public void deactivate() {
		if (workers != null) {
			for (String worker : workers) {
				log.info("stopping asynchronous worker. (worker = " + worker + ")");
				_executorMap.get(worker).shutdown();
				try {
					if (! _executorMap.get(worker).awaitTermination(_shutdownTimeoutMap.get(worker), TimeUnit.MILLISECONDS)) {
						List<Runnable> remained = _executorMap.get(worker).shutdownNow();
						if (remained != null) {
							log.info("shutdown timeout. runnable remained. (worker" + worker + ", size = " + remained.size() + ")");
						}
					}
				} catch (InterruptedException e) {
					_executorMap.get(worker).shutdownNow();
				}
			}
		}
	}

	@Override
	public void destroy() {

	}

	private static void runPersistedTask(String worker) {
		List<Serializable> params = AsyncTask.getRemainedParams(worker);
		log.info("running remained task : num = " + params.size());

		for (Serializable param : params) {
			try {
				if (log.isDebugEnabled()) {
					log.debug("running remained task. (worker = " + worker + ", param = " + param + ")");
				}
				addTask(worker, param, true);
			} catch (HinemosUnknown e) {
				log.warn(e.getMessage());
			}
		}
	}

	private static long getNextTaskId(String worker) {
		synchronized (_counterLock.get(worker)) {
			long taskId = _nextTaskIdMap.get(worker);
			_nextTaskIdMap.put(worker, taskId == Long.MAX_VALUE ? 0L : taskId + 1L);
			if (taskId % 100 == 0) {
				log.info("asynchronomous worker statistics (worker = " + worker + ", count = " + taskId + ")");
			}
			return taskId;
		}
	}

	/**
	 * Hoge hoge = new Hoge();
	 * for (int i = 0; i < 10; i ++) {
	 *     hoge.set(i);
	 *     AsyncWorkerPlugin.addTask(A, hoge, b);
	 * }
	 * とするとバグります（元インスタンス（hoge）の変更の影響を受ける場合があります）。
	 * 下記のように、newしなおして別インスタンスとする必要があります。
	 * for (int i = 0; i < 10; i ++) {
	 *     Hoge hoge = new Hoge();
	 *     hoge.set(i);
	 *     AsyncWorkerPlugin.addTask(A, hoge, b);
	 * }
	 * 
	 * @param worker
	 * @param param ここに入力するオブジェクトには要注意。
	 * @param persist
	 * @throws HinemosUnknown
	 */
	public static void addTask(String worker, Serializable param, boolean persist) throws HinemosUnknown {
		AsyncTaskFactory factory = _factoryMap.get(worker);

		if (factory == null) {
			throw new HinemosUnknown("worker not found. (worker = " + worker + ")");
		}

		Runnable r = factory.createTask(param);

		long taskId = getNextTaskId(worker);
		if (persist && log.isDebugEnabled()) {
			log.debug("task will be persisted. (worker = " + worker + ", taskId = " + taskId + ", param = " + param + ")");
		}
		AsyncTask task = new AsyncTask(r, worker, param, taskId, persist);

		// commit成功後にworkerスレッドにタスクが割り当てる
		JpaTransactionManager tm = null;
		try {
			tm = new JpaTransactionManager();
			tm.begin();

			tm.addCallback(new TaskExecutionAfterCommitCallback(task));

			tm.commit();
		} catch (Exception e) {
			log.warn("task addition failure. (worker = " + worker + ", taskId = " + taskId + ", param = " + param + ")", e);
			tm.rollback();
		} finally {
			tm.close();
		}
	}

	public static void commitTaskExecution(AsyncTask task) {
		ThreadPoolExecutor executor = _executorMap.get(task._worker);

		synchronized (_executorLock.get(task._worker)) {
			executor.execute(task);
		}
		
		if (log.isDebugEnabled()) {
			log.debug("committed, task will be executed. (worker = " + task._worker + ", taskId = " + task._taskId + ", param = " + task._param + ")");
		}
	}

	public static String[] getWorkerList() {
		return workers;
	}

	public static int getTaskCount(String worker) throws HinemosUnknown {
		ThreadPoolExecutor executor = _executorMap.get(worker);

		if (executor == null) {
			throw new HinemosUnknown("worker thread is not initialized. (worker = " + worker + ")");
		}

		synchronized (_executorLock) {
			return executor.getQueue().size();
		}
	}

	private class AsyncThreadFactory implements ThreadFactory {

		private final String _worker;
		private volatile int _count = 0;

		public AsyncThreadFactory(String worker) {
			this._worker = worker;
		}

		@Override
		public Thread newThread(Runnable r) {
			String threadName = "AsyncTask-" + _count++ + " [" + _worker + "]";
			return new Thread(r, threadName);
		}
	}

	private class TaskRejectionHandler extends ThreadPoolExecutor.DiscardPolicy {

		private final String _worker;

		public TaskRejectionHandler(String worker) {
			this._worker = worker;
		}

		@Override
		public void rejectedExecution(Runnable r, ThreadPoolExecutor e) {
			log.warn("too many tasks are assigned to " + _worker + ". rejecting new task. : " + r + ".");
		}
	}

}
