/*

Copyright (C) 2012 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.

 */

/**
 * ジョブの多重度をキャッシュするクラス
 */
package com.clustercontrol.jobmanagement.util;

import java.util.ArrayList;
import java.util.List;
import java.util.Queue;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentLinkedQueue;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.accesscontrol.bean.PrivilegeConstant.ObjectPrivilegeMode;
import com.clustercontrol.bean.StatusConstant;
import com.clustercontrol.commons.util.HinemosEntityManager;
import com.clustercontrol.commons.util.JpaTransactionManager;
import com.clustercontrol.fault.FacilityNotFound;
import com.clustercontrol.fault.HinemosUnknown;
import com.clustercontrol.fault.InvalidRole;
import com.clustercontrol.fault.JobInfoNotFound;
import com.clustercontrol.jobmanagement.bean.JobConstant;
import com.clustercontrol.jobmanagement.factory.JobSessionNodeImpl;
import com.clustercontrol.jobmanagement.model.JobSessionJobEntity;
import com.clustercontrol.jobmanagement.model.JobSessionJobEntityPK;
import com.clustercontrol.jobmanagement.model.JobSessionNodeEntity;
import com.clustercontrol.jobmanagement.model.JobSessionNodeEntityPK;
import com.clustercontrol.repository.bean.NodeInfo;
import com.clustercontrol.repository.session.RepositoryControllerBean;

public class JobMultiplicityCache {
	private static Log m_log = LogFactory.getLog( JobMultiplicityCache.class );

	// map操作やqueue操作の際にはこのロックを利用する。
	private static Object mapLock = new Object();

	private static ConcurrentHashMap<String, Queue<JobSessionNodeEntityPK>> waitMap =
			new ConcurrentHashMap<String, Queue<JobSessionNodeEntityPK>>();
	private static ConcurrentHashMap<String, Queue<JobSessionNodeEntityPK>> runningMap =
			new ConcurrentHashMap<String, Queue<JobSessionNodeEntityPK>>();

	private static boolean initFlag = true;

	/**
	 * ノードごとの現在の多重度を返すメソッド。
	 * リポジトリ[エージェント]ビューから呼ばれる。
	 * @param facilityId
	 * @return
	 */
	public static Integer getRunningMultiplicity(String facilityId) {
		refresh();
		Queue<JobSessionNodeEntityPK> runningQueue = runningMap.get(facilityId);
		if (runningQueue == null) {
			return 0;
		}
		return runningQueue.size();
	}

	/**
	 * ノードごとの現在の多重度を返すメソッド。
	 * リポジトリ[エージェント]ビューから呼ばれる。
	 * @param facilityId
	 * @return
	 */
	public static Integer getWaitMultiplicity(String facilityId) {
		refresh();
		Queue<JobSessionNodeEntityPK> waitQueue = waitMap.get(facilityId);
		if (waitQueue == null) {
			return 0;
		}
		return waitQueue.size();
	}

	/**
	 * ジョブ多重度を確認して、ジョブが実行できるか確認するメソッド
	 * @param facilityId
	 * @return
	 */
	public static boolean isRunNow(String facilityId) {
		refresh();
		int multiplicity = 0;
		int queueSize = 0;
		synchronized(mapLock) {
			Queue<JobSessionNodeEntityPK> runningQueue = runningMap.get(facilityId);
			if (runningQueue == null) {
				runningQueue = new ConcurrentLinkedQueue<JobSessionNodeEntityPK>();
				runningMap.put(facilityId, runningQueue);
			}
			queueSize = runningQueue.size();

			if(m_log.isDebugEnabled()){
				for(JobSessionNodeEntityPK q : runningQueue){
					m_log.debug("isRunNow runningQueue : " + q);
				}
			}
}
		try {
			NodeInfo nodeInfo = new RepositoryControllerBean().getNode(facilityId);
			multiplicity = nodeInfo.getJobMultiplicity();
		} catch (FacilityNotFound e) {
			m_log.warn("kick " + e.getMessage());
		} catch (HinemosUnknown e) {
			m_log.warn("kick " + e.getMessage(),e);
		}
		// ジョブ多重度が0以下に設定されていた場合は、多重度のロジックは利用しない。
		if (multiplicity == 0) {
			return true;
		}
		if (queueSize < multiplicity) {
			return true;
		}
		return false;
	}

	/**
	 * statusが100(StatusConstant.TYPE_RUNNING)に遷移したい場合は、このメソッドを呼ぶこと。
	 * 
	 * このメソッドはwaitQueueに該当のジョブを追加する。
	 * また、waitQueueに追加したジョブを実行するために、kickを呼び出す。
	 * （多重度が多い場合はwaitQueueのまま待機。
	 *   多重度が少ない場合はwaitQueueから削除され、runningQueueに追加。） 
	 * @param facilityId
	 */
	public static boolean toRunning(JobSessionNodeEntityPK pk) {
		m_log.info("toRunning " + pk);

		refresh();
		String facilityId = pk.getFacilityId();
		synchronized(mapLock) {
			Queue<JobSessionNodeEntityPK> runningQueue = runningMap.get(facilityId);
			Queue<JobSessionNodeEntityPK> waitQueue = waitMap.get(facilityId);
			if (waitQueue == null) {
				waitQueue = new ConcurrentLinkedQueue<JobSessionNodeEntityPK>();
				waitMap.put(facilityId, waitQueue);
			}
			if ((runningQueue == null || !runningQueue.contains(pk)) &&
					!waitQueue.contains(pk)) {
				m_log.debug("toRunning add waitQueue : " + pk);
				waitQueue.offer(pk); //// waitQueueに追加
			}
			
			if(m_log.isDebugEnabled()){
				for(JobSessionNodeEntityPK q : runningQueue){
					m_log.debug("toRunning runningQueue : " + q);
				}
				for(JobSessionNodeEntityPK q : waitQueue){
					m_log.debug("toRunning waitQueue : " + q);
				}
			}
		}
		
		// 数行上でwaitQueueに追加されたジョブを実行。
		kick(facilityId);
		return true;
	}

	/**
	 * statusが100(StatusConstant.TYPE_RUNNING)から別の状態に遷移したら、このメソッドを呼ぶこと。
	 * 
	 * このメソッドはrunningQueueから該当のジョブを削除して、多重度を下げる。
	 * また、多重度を下げた後に、待っているジョブを実行させる。（kick）
	 * @param pk
	 */
	public static boolean fromRunning(JobSessionNodeEntityPK pk) {
		m_log.info("fromRunning " + pk);

		refresh();
		String facilityId = pk.getFacilityId();
		Queue<JobSessionNodeEntityPK> runningQueue = runningMap.get(facilityId);
		if (runningQueue == null) {
			m_log.warn("fromRunning " + pk);
			return false;
		}
		synchronized(mapLock) {
			if (!runningQueue.remove(pk)) { //// runningQueueから削除
				// 普通は実行中から停止に遷移するが、
				// 実行中以外(待機など)から、停止に遷移することがある。
				m_log.info("fromRunning(). from not-running to stop : " + pk);
			}
		}
		if(m_log.isDebugEnabled()){
			for(JobSessionNodeEntityPK q : runningQueue){
				m_log.debug("fromRunning runningQueue : " + q);
			}
		}
		
		// 同一facilityIDのノードで待機中のジョブを実行状態に遷移。
		kick(facilityId);
		return true;
	}

	/**
	 * waitQueueから削除する。
	 * 
	 * ジョブ詳細で実行中で、多重度が多く待機になっている場合は、
	 * waitQueueからジョブを削除する必要がある。
	 * 
	 * ジョブ詳細が待機で、ノード詳細も待機の場合はwaitQueueから削除する必要はない。
	 * （しかし、念のためこのメソッドを呼ぶこと。）
	 * 
	 * @param pk
	 */
	public static void removeWait(JobSessionNodeEntityPK pk) {
		Queue<JobSessionNodeEntityPK> waitQueue = waitMap.get(pk.getFacilityId());
		if (waitQueue == null) {
			return;
		}
		synchronized(mapLock) {
			if (waitQueue.remove(pk)) {
				m_log.info("removeWait " + pk);
			}
		}
	}
	
	/**
	 * ノード詳細のジョブを実行する。
	 * ただし、ジョブ多重度が上限に達している場合は、実行されない。
	 * 
	 * @param facilityId
	 * @return
	 */
	public static void kick(String facilityId) {
		m_log.debug("kick " + facilityId);

		refresh();
		boolean kickFlag = false;
		// ジョブノードの実行
		synchronized(mapLock) {
			Queue<JobSessionNodeEntityPK> waitQueue = waitMap.get(facilityId);
			Queue<JobSessionNodeEntityPK> runningQueue = runningMap.get(facilityId);
			if (waitQueue == null || waitQueue.size() == 0) {
				return;
			}
			if (runningQueue == null) {
				runningQueue = new ConcurrentLinkedQueue<JobSessionNodeEntityPK>();
				runningMap.put(facilityId, runningQueue);
			}
			if (isRunNow(facilityId)) {
				JpaTransactionManager jtm = new JpaTransactionManager();
				try {
					jtm.begin();
					JobSessionNodeEntityPK pk = waitQueue.poll(); //// waitQueueから削除
					m_log.debug("kick remove waitQueue : " + pk);
					if (new JobSessionNodeImpl().wait2running(pk)) {
						m_log.debug("kick add runningQueue : " + pk);
						runningQueue.offer(pk); //// runningQueueに追加
						kickFlag = true;
					}
					jtm.commit();
				} catch (Exception e) {
					m_log.warn("kick : " + e.getClass().getSimpleName() + ", " + e.getMessage(), e);
					jtm.rollback();
				} finally {
					jtm.close();
				}
			}
			
			if(m_log.isDebugEnabled()){
				for(JobSessionNodeEntityPK q : runningQueue){
					m_log.debug("kick runningQueue : " + q);
				}
				for(JobSessionNodeEntityPK q : waitQueue){
					m_log.debug("kick waitQueue : " + q);
				}
			}
		}
		if (kickFlag) {
			kick(facilityId);
		}
	}

	/**
	 * 「マネージャ起動時」と「ジョブ履歴削除」から呼ばれる。
	 * 
	 * 処理概要は下記の通り。
	 * 
	 * 処理1.
	 * ノード詳細からrunningを検索して、runningQueueを構築する。
	 * 
	 * 処理2.
	 * ジョブ詳細からrunnningを検索して、ジョブを実行する。JobSessionNodeImpl().startNode()
	 * 正確に言うと、startNodeメソッドの内部で待機中のジョブに対して、toRunningさせる。
	 * (toRunningによりwaitQueueに追加され、多重度が低ければ、kickメソッド内部で実行中に遷移する。)
	 */
	public static void refresh() {
		List<JobSessionJobEntityPK> execJobList = new ArrayList<JobSessionJobEntityPK>();
		synchronized(mapLock) {
			if (!initFlag) {
				return;
			}
			m_log.info("cache refresh");
			JpaTransactionManager jtm = new JpaTransactionManager();
			if (!jtm.isNestedEm()) {
				m_log.warn("refresh() : transaction has not been begined.");
				jtm.close();
				return;
			}

			HinemosEntityManager em = jtm.getEntityManager();
			// runningQueueの再構築
			{
				// オブジェクト権限チェックなし
				List<JobSessionNodeEntity> nodeList = em.createNamedQuery("JobSessionNodeEntity.findByStatus", JobSessionNodeEntity.class, ObjectPrivilegeMode.NONE)
						.setParameter("status", StatusConstant.TYPE_RUNNING).getResultList();
				for (JobSessionNodeEntity node : nodeList) {
					String facilityId = node.getId().getFacilityId();
					Queue<JobSessionNodeEntityPK> runningQueue = runningMap.get(facilityId);
					if (runningQueue == null) {
						runningQueue = new ConcurrentLinkedQueue<JobSessionNodeEntityPK>();
						runningMap.put(facilityId, runningQueue);
					}
					m_log.debug("refresh add runningQueue : " + node.getId());
					runningQueue.offer(node.getId());
				}
			}

			// 履歴削除により多重度が下がったノードを実行させるlistを作成
			{
				// オブジェクト権限チェックなし
				List<JobSessionJobEntity> jobList = em.createNamedQuery("JobSessionJobEntity.findByStatus", JobSessionJobEntity.class, ObjectPrivilegeMode.NONE)
						.setParameter("status", StatusConstant.TYPE_RUNNING).getResultList();
				for (JobSessionJobEntity job : jobList) {
					if (job.getJobInfoEntity() == null || job.getJobInfoEntity().getJobType() == null) {
						m_log.info("wait job is deleted"); // 待機中のジョブが履歴削除により消された場合にこのルートを通る。
						continue;
					}
					if (job.getJobInfoEntity().getJobType() != JobConstant.TYPE_JOB) {
						continue;
					}
					execJobList.add(job.getId());
				}
			}
			initFlag = false;
		}

		// execJobListで実行
		for (JobSessionJobEntityPK id : execJobList) {
			try {
				m_log.info("refresh() startNode=" + id);
				new JobSessionNodeImpl().startNode(id.getSessionId(), id.getJobunitId(), id.getJobId());
			} catch (InvalidRole e) {
				m_log.warn("refresh " + e.getMessage());
			} catch (JobInfoNotFound e) {
				m_log.warn("refresh " + e.getMessage());
			}
		}
		print();
	}

	private static void print() {
		for (String facilityId : waitMap.keySet()) {
			m_log.info("print facilityId=" + facilityId +
					", waitQueue=" + waitMap.get(facilityId).size());
		}
		for (String facilityId : runningMap.keySet()) {
			m_log.info("print facilityId=" + facilityId +
					", runningQueue=" + runningMap.get(facilityId).size());
		}
	}
	
	public static String getJobQueueStr() {
		if (initFlag) {
			return "cache has not been created";
		}
		String message = "";
		synchronized(mapLock) {
			message += "Running:\n";
			for (String facilityId : runningMap.keySet()) {
				Queue<JobSessionNodeEntityPK> queue = runningMap.get(facilityId);
				String str = "";
				for (JobSessionNodeEntityPK pk : queue) {
					str += "[" + pk.getSessionId() + "," + pk.getJobunitId() + "," + pk.getJobId() + "]\n";
				}
				message += facilityId + "(" + queue.size() + ")=\n" + str;
			}
			message += "Wait:\n";
			for (String facilityId : waitMap.keySet()) {
				Queue<JobSessionNodeEntityPK> queue = waitMap.get(facilityId);
				String str = "";
				for (JobSessionNodeEntityPK pk : queue) {
					str += "[" + pk.getSessionId() + "," + pk.getJobunitId() + "," + pk.getJobId() + "]\n";
				}
				message += facilityId + "(" + queue.size() + ")=\n" + str;
			}
		}
		return message;
	}
}
