/*

Copyright (C) 2006 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.

 */

package com.clustercontrol.collectiverun.factory;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Locale;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.fault.CollectiveRunNotFound;
import com.clustercontrol.bean.StatusConstant;
import com.clustercontrol.commons.bean.ViewListInfo;
import com.clustercontrol.collectiverun.bean.DetailInfo;
import com.clustercontrol.collectiverun.bean.HistoryFilterInfo;
import com.clustercontrol.collectiverun.bean.HistoryInfo;
import com.clustercontrol.collectiverun.model.CrunCmdMstEntity;
import com.clustercontrol.collectiverun.model.CrunSessionDetailEntity;
import com.clustercontrol.collectiverun.model.CrunSessionEntity;
import com.clustercontrol.collectiverun.util.QueryUtil;
import com.clustercontrol.util.Messages;

/**
 * 履歴検索クラス
 *
 * @version 1.0.0
 * @since 1.0.0
 */
public class SelectSession {
	/** ログ出力のインスタンス */
	private static Log m_log = LogFactory.getLog( SelectSession.class );

	private final static int MAX_DISPLAY_NUMBER = 500;

	/**
	 * 履歴一覧取得
	 * 
	 * @param histories
	 * @return
	 */
	public ViewListInfo getHistoryList(int histories) {
		ViewListInfo list = new ViewListInfo();
		ArrayList<HistoryInfo> history = new ArrayList<HistoryInfo>();
		int total = 0;

		if(histories <= 0){
			histories = MAX_DISPLAY_NUMBER;
		}
		Integer limit = Integer.valueOf(histories + 1);

		//全セッションを取得
		List<CrunSessionEntity> sessionList
		= QueryUtil.getCrunSessionByHistoryFilter(null, null, null, null, null, null, limit);

		// 履歴数をカウント
		if(sessionList.size() > histories){
			// 最大表示件数より大きい場合
			List<CrunSessionEntity> allHistory
			= QueryUtil.getCrunSessionByHistoryFilter(null, null, null, null, null, null, null);
			total = allHistory.size();

		} else {
			total = sessionList.size();
		}

		Iterator<CrunSessionEntity> itr = sessionList.iterator();
		while(itr.hasNext()){
			//セッションを取得
			CrunSessionEntity session = itr.next();
			CrunCmdMstEntity cmd = null;
			try {
				cmd = QueryUtil.getCrunCmdMstPK(session.getCommandId());
			} catch (CollectiveRunNotFound e) {
			}

			//履歴一覧の１行を作成
			HistoryInfo info = new HistoryInfo();
			info.setStatus(session.getStatus());
			info.setSessionId(session.getSessionId());
			if(session.getStartDate() != null){
				info.setStartDate(session.getStartDate().getTime());
			}
			if(session.getEndDate() != null){
				info.setEndDate(session.getEndDate().getTime());
			}

			String name = null;
			try{
				name = SelectMaster.typeIdToString(session.getTypeId(), Locale.getDefault());
			} catch(CollectiveRunNotFound e){
				m_log.info("getHistoryList() type (type_id=" + session.getTypeId()
						+ ") not found in session (session_id=" + session.getSessionId() + ")");
			}

			if(name != null){
				info.setType(name);
			}
			else{
				info.setType("");
			}

			if(cmd != null){
				info.setName(Messages.getString(cmd.getNameId(), Locale.getDefault()));
			}
			else{
				info.setName("");
			}
			info.setFacilityId(session.getFacilityId());
			info.setScopeText(session.getScopeText());
			info.setNodeCount(session.getNodeCnt());
			info.setEndNormalCount(session.getEndNomalCnt());
			info.setEndWarningCount(session.getEndWarningCnt());
			info.setEndAbnormalCount(session.getEndAbnomalCnt());
			info.setWaitCount(session.getWaitCnt());
			info.setRunningCount(session.getRunningCnt());
			history.add(info);

			//取得した履歴を最大表示件数まで格納したら終了
			if(history.size() >= histories)
				break;
		}
		list.setTotal(total);
		list.setCollectiverunHistoryList(history);

		return list;
	}

	/**
	 * 履歴一覧取得
	 * 
	 * @param property
	 * @param histories
	 * @return
	 */
	public ViewListInfo getHistoryList(HistoryFilterInfo filter, int histories) {
		String type = null;
		Integer status = null;
		int total = 0;
		ArrayList<HistoryInfo> history = null;
		ViewListInfo list = null;

		//検索条件に該当するセッションを取得
		List<CrunSessionEntity> sessionList = null;
		//管理種別取得
		if(filter.getType() instanceof String && filter.getType().length() > 0){
			type = SelectMaster.stringToTypeId(filter.getType(), Locale.getDefault());
		}
		//状態取得
		if(filter.getStatus() instanceof String){
			status = Integer.valueOf(StatusConstant.stringToType(filter.getStatus()));
		}

		list = new ViewListInfo();
		history = new ArrayList<HistoryInfo>();

		if(histories <= 0){
			histories = MAX_DISPLAY_NUMBER;
		}

		sessionList = QueryUtil.getCrunSessionByHistoryFilter(
				filter.getStartFromDate(),
				filter.getStartToDate(),
				filter.getEndFromDate(),
				filter.getEndToDate(),
				type,
				status,
				histories);

		if(sessionList != null){
			// 履歴数を取得
			total = sessionList.size();

			for(CrunSessionEntity session : sessionList){
				CrunCmdMstEntity cmd = null;
				try {
					cmd = QueryUtil.getCrunCmdMstPK(session.getCommandId());
				} catch (CollectiveRunNotFound e) {
				}

				//履歴一覧の１行を作成
				HistoryInfo info = new HistoryInfo();
				info.setStatus(session.getStatus());
				info.setSessionId(session.getSessionId());
				if(session.getStartDate() != null){
					info.setStartDate(session.getStartDate().getTime());
				}
				if(session.getEndDate() != null){
					info.setEndDate(session.getEndDate().getTime());
				}

				String name = null;
				try{
					name = SelectMaster.typeIdToString(session.getTypeId(), Locale.getDefault());
				}catch(CollectiveRunNotFound e){
					m_log.info("getHistoryList() type (type_id="
							+ session.getTypeId() + ") not found in session (session_id=" + session.getSessionId() + ")");
				}

				if(name != null){
					info.setName(Messages.getString(cmd.getNameId(), Locale.getDefault()));
				}
				else{
					info.setName("");
				}

				if(cmd != null){
					info.setType(name);
				}
				else{
					info.setType("");
				}
				info.setFacilityId(session.getFacilityId());
				info.setScopeText(session.getScopeText());
				info.setNodeCount(session.getNodeCnt());
				info.setEndNormalCount(session.getEndNomalCnt());
				info.setEndWarningCount(session.getEndWarningCnt());
				info.setEndAbnormalCount(session.getEndAbnomalCnt());
				info.setWaitCount(session.getWaitCnt());
				info.setRunningCount(session.getRunningCnt());
				history.add(info);

				//取得した履歴を最大表示件数まで格納したら終了
				if(history.size() >= histories)
					break;
			}
		}
		list.setTotal(total);
		list.setCollectiverunHistoryList(history);

		return list;
	}

	/**
	 * 一括制御詳細情報の一覧を取得します。
	 * 
	 * @param sessionId
	 * @return 一括制御詳細情報のリスト
	 */
	public ArrayList<DetailInfo> getDetailList(String sessionId) {
		ArrayList<DetailInfo> list = new ArrayList<DetailInfo>();

		//セッションディテールをセッションIDで検索し取得
		List<CrunSessionDetailEntity> detailList
		= QueryUtil.getCrunSessionDetailBySessionId(sessionId);

		for(CrunSessionDetailEntity detail : detailList){
			//履歴一覧の１行を作成
			DetailInfo info = new DetailInfo();
			info.setStatus(detail.getStatus());
			info.setEndStatus(detail.getEndStatus());
			info.setEndValue(detail.getEndValue());
			info.setFacilityId(detail.getId().getFacilityId());
			info.setFacilityName(detail.getFacilityName());
			if(detail.getStartDate() != null){
				info.setStartDate(detail.getStartDate().getTime());
			}
			if(detail.getEndDate() != null){
				info.setEndDate(detail.getEndDate().getTime());
			}
			StringBuffer message = new StringBuffer();
			if(detail.getMessage() != null){
				message.append(detail.getMessage());
			}
			if(detail.getErrorMessage() != null){
				message.append(detail.getErrorMessage());
			}

			info.setMessage(message.toString());
			list.add(info);
		}
		return list;
	}
}
