/*

Copyright (C) 2006 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.

 */

package com.clustercontrol.collectiverun.factory;

import java.util.ArrayList;
import java.util.List;

import javax.persistence.EntityExistsException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.collectiverun.bean.CommandInfo;
import com.clustercontrol.collectiverun.bean.CommandParameterInfo;
import com.clustercontrol.collectiverun.bean.EndStatusInfo;
import com.clustercontrol.collectiverun.bean.ItemConstant;
import com.clustercontrol.collectiverun.bean.ItemInfo;
import com.clustercontrol.collectiverun.bean.ParameterInfo;
import com.clustercontrol.collectiverun.bean.ParameterSelectInfo;
import com.clustercontrol.collectiverun.bean.TypeInfo;
import com.clustercontrol.collectiverun.model.CrunCmdMstEntity;
import com.clustercontrol.collectiverun.model.CrunCmdParamMstEntity;
import com.clustercontrol.collectiverun.model.CrunCmdParamMstEntityPK;
import com.clustercontrol.collectiverun.model.CrunEndMstEntity;
import com.clustercontrol.collectiverun.model.CrunEndMstEntityPK;
import com.clustercontrol.collectiverun.model.CrunParamMstEntity;
import com.clustercontrol.collectiverun.model.CrunParamSelectMstEntity;
import com.clustercontrol.collectiverun.model.CrunParamSelectMstEntityPK;
import com.clustercontrol.collectiverun.model.CrunTypeMstEntity;
import com.clustercontrol.collectiverun.util.QueryUtil;
import com.clustercontrol.fault.CollectiveRunNotFound;

/**
 * マスタ更新クラス
 *
 * @version 2.1.0
 * @since 2.1.0
 */
public class ModifyMaster {
	private static Log m_log = LogFactory.getLog( ModifyMaster.class );

	/**
	 * 一括制御マスタ情報更新
	 * 
	 * @param info
	 * @param parentInfo
	 * @throws EntityExistsException
	 */
	public void update(ItemInfo info, ItemInfo parentInfo) throws EntityExistsException {
		if(info == null || parentInfo == null){
			return;
		}

		//cast出来ない場合はreturnする
		if (!(info instanceof TypeInfo)
				&& !(info instanceof CommandInfo)
				&& !(info instanceof CommandParameterInfo)
				&& !(info instanceof ParameterInfo)
				&& !(info instanceof ParameterSelectInfo)
				&& !(parentInfo instanceof CommandInfo)
				&& !(parentInfo instanceof TypeInfo))
			return;

		if(info.getType() == ItemConstant.TYPE_TYPE){
			updateType((TypeInfo)info);
		}
		else if(info.getType() == ItemConstant.TYPE_COMMAND){
			updateCommand((CommandInfo)info,
					((TypeInfo)parentInfo).getId());
		}
		else if(info.getType() == ItemConstant.TYPE_COMMAND_PARAMETER){
			updateCommandParameter((CommandParameterInfo)info,
					((CommandInfo)parentInfo).getId());
		}
		else if(info.getType() == ItemConstant.TYPE_PARAMETER){
			updateParameter((ParameterInfo)info);
		}
		else if(info.getType() == ItemConstant.TYPE_SELECT_ITEM){
			updateParameterSelect((ParameterSelectInfo)info);
		}
	}

	/**
	 * 種別情報更新
	 * 
	 * @param info
	 * @throws EntityExistsException
	 */
	private void updateType(TypeInfo info) throws EntityExistsException {

		CrunTypeMstEntity type = null;
		try {
			// 検索
			type = QueryUtil.getCrunTypeMstPK(info.getId());
		} catch (CollectiveRunNotFound e) {
		}
		if (type == null) {
			try {
				// 新規作成
				type = new CrunTypeMstEntity(info.getId());
			} catch (EntityExistsException e) {
				m_log.info("updateType() : "
						+ e.getClass().getSimpleName() + ", " + e.getMessage());
				throw e;
			}
		}
		type.setNameId(info.getName());
		type.setOrderNo(Integer.valueOf(info.getOrderNo()));
	}

	/**
	 * コマンド情報更新
	 * 
	 * @param info
	 * @param typeId
	 * @throws EntityExistsException
	 */
	private void updateCommand(CommandInfo info, String typeId) throws EntityExistsException {

		try {
			CrunCmdMstEntity command = null;
			try {
				// 検索
				command = QueryUtil.getCrunCmdMstPK(info.getId());
			} catch (CollectiveRunNotFound e) {
			}
			if (command == null) {
				try {
					// 新規作成
					CrunTypeMstEntity type = null;
					try {
						type = QueryUtil.getCrunTypeMstPK(typeId);
					} catch (CollectiveRunNotFound e) {
					}
					command = new CrunCmdMstEntity(info.getId(), type);
				} catch (EntityExistsException e) {
					m_log.info("updateCommand() : "
							+ e.getClass().getSimpleName() + ", " + e.getMessage());
					throw e;
				}
			}
			command.setNameId(info.getName());
			command.setOrderNo(Integer.valueOf(info.getOrderNo()));
			command.setCommand(info.getCommand());
			command.setCommandType(Integer.valueOf(info.getCommandType()));

			//終了状態作成・更新
			if(info.getEndStatus() != null &&
					info.getEndStatus() instanceof ArrayList){

				List<CrunEndMstEntity> crunEndMstEntities = new ArrayList<CrunEndMstEntity>();

				//CrunEndMstEntityを作成
				for(EndStatusInfo endStatus : info.getEndStatus()){
					// 検索
					CrunEndMstEntityPK crunEndMstEntityPk = new CrunEndMstEntityPK(info.getId(), Integer.valueOf(endStatus.getEndStatus()));
					CrunEndMstEntity crunEndMstEntity = null;
					try {
						crunEndMstEntity = QueryUtil.getCrunEndMstPK(crunEndMstEntityPk);
					} catch (CollectiveRunNotFound e) {
					}

					if (crunEndMstEntity == null) {
						try {
							// 新規作成
							crunEndMstEntity = new CrunEndMstEntity(crunEndMstEntityPk, command);
						} catch (EntityExistsException e) {
							m_log.info("updateCommand() : "
									+ e.getClass().getSimpleName() + ", " + e.getMessage());
							throw e;
						}
					}
					crunEndMstEntity.setEndValueFrom(Integer.valueOf(endStatus.getStartRangeValue()));
					crunEndMstEntity.setEndValueTo(Integer.valueOf(endStatus.getEndRangeValue()));
					//終了状態作成・更新
					crunEndMstEntities.add(crunEndMstEntity);
				}
				command.setCrunEndMstEntities(crunEndMstEntities);
			}
		} catch (EntityExistsException e) {
			m_log.info("updateCommand() : "
					+ e.getClass().getSimpleName() + ", " + e.getMessage());
			throw e;
		}
	}

	/**
	 * コマンドパラメータ情報更新
	 * 
	 * @param info
	 * @param commandId
	 * @throws EntityExistsException
	 */
	private void updateCommandParameter(CommandParameterInfo info, String commandId) throws EntityExistsException {

		CrunCmdParamMstEntityPK commandParameterPk = new CrunCmdParamMstEntityPK(commandId, info.getId());
		CrunCmdParamMstEntity commandParameter = null;
		try {
			commandParameter = QueryUtil.getCrunCmdParamMstPK(commandParameterPk);
		} catch (CollectiveRunNotFound e) {
		}
		if (commandParameter == null) {
			try {
				// 新規作成
				CrunCmdMstEntity command = null;
				try {
					command = QueryUtil.getCrunCmdMstPK(commandId);
				} catch (CollectiveRunNotFound e) {
				}
				CrunParamMstEntity param = null;
				try {
					param = QueryUtil.getCrunParamMstPK(info.getId());
				} catch (CollectiveRunNotFound e) {
				}
				commandParameter = new CrunCmdParamMstEntity(commandParameterPk, command, param);
			} catch (EntityExistsException e) {
				m_log.info("updateCommandParameter() : "
						+ e.getClass().getSimpleName() + ", " + e.getMessage());
				throw e;
			}
		}
		commandParameter.setParamPrefix(info.getPrefix());
		commandParameter.setOrderNo(Integer.valueOf(info.getOrderNo()));
	}

	/**
	 * パラメータ情報更新
	 * 
	 * @param info
	 * @throws EntityExistsException
	 */
	private void updateParameter(ParameterInfo info) throws EntityExistsException {

		CrunParamMstEntity parameter = null;
		try {
			parameter = QueryUtil.getCrunParamMstPK(info.getId());
		} catch (CollectiveRunNotFound e) {
		}
		if (parameter == null) {
			try {
				// 新規作成
				parameter = new CrunParamMstEntity(info.getId());
			} catch (EntityExistsException e) {
				m_log.info("updateParameter() : "
						+ e.getClass().getSimpleName() + ", " + e.getMessage());
				throw e;
			}
		}
		parameter.setNameId(info.getName());
		parameter.setParamType(Integer.valueOf(info.getParamType()));
	}


	/**
	 * パラメータ選択肢情報更新
	 * 
	 * @param info
	 * @throws EntityExistsException
	 */
	private void updateParameterSelect(ParameterSelectInfo info) throws EntityExistsException {

		CrunParamSelectMstEntityPK selectPk = new CrunParamSelectMstEntityPK(info.getId(), Integer.valueOf(info.getOrderNo()));
		CrunParamSelectMstEntity select = null;
		try {
			select = QueryUtil.getCrunParamSelectMstPK(selectPk);
		} catch (CollectiveRunNotFound e) {
		}
		if (select == null) {
			try {
				// 新規作成
				CrunParamMstEntity param = null;
				try {
					param = QueryUtil.getCrunParamMstPK(info.getId());
				} catch (CollectiveRunNotFound e) {
				}
				select = new CrunParamSelectMstEntity(selectPk, param);
			} catch (EntityExistsException e) {
				m_log.info("updateParameterSelect() : "
						+ e.getClass().getSimpleName() + ", " + e.getMessage());
				throw e;
			}
		}
		select.setNameId(info.getName());
		select.setParamValue(info.getValue());
	}

	/**
	 * 一括制御マスタ情報表示順序更新
	 * 
	 * @param info1
	 * @param info2
	 * @param parentInfo
	 * @throws EntityExistsException
	 */
	public void changeOrder(ItemInfo info1, ItemInfo info2, ItemInfo parentInfo) throws EntityExistsException {
		int orderNo1 = info1.getOrderNo();
		int orderNo2 = info2.getOrderNo();

		//選択したアイテムを移動ターゲットアイテムの順序で作成
		info1.setOrderNo(orderNo2);
		update(info1, parentInfo);

		//移動ターゲットアイテムを選択アイテムの順序で作成
		info2.setOrderNo(orderNo1);
		update(info2, parentInfo);
	}
}
