/*
Copyright (C) 2013 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.
 */

package com.clustercontrol.ws.cloud;

import java.util.List;

import javax.jws.WebMethod;
import javax.jws.WebParam;
import javax.jws.WebService;

import com.clustercontrol.cloud.CloudManagerFault;
import com.clustercontrol.cloud.Filter;
import com.clustercontrol.cloud.bean.AvailableRole;
import com.clustercontrol.cloud.bean.CloudAccountResource;
import com.clustercontrol.cloud.bean.CloudInstance;
import com.clustercontrol.cloud.bean.CloudInstanceBackup;
import com.clustercontrol.cloud.bean.CloudRegion;
import com.clustercontrol.cloud.bean.CloudService;
import com.clustercontrol.cloud.bean.CloudStorage;
import com.clustercontrol.cloud.bean.CloudStorageBackup;
import com.clustercontrol.cloud.bean.CloudTemplate;
import com.clustercontrol.cloud.bean.CloudType;
import com.clustercontrol.cloud.bean.CloudUser;
import com.clustercontrol.cloud.bean.CreateAccountResourceRequest;
import com.clustercontrol.cloud.bean.CreateCloudUserRequest;
import com.clustercontrol.cloud.bean.CreateInstanceBackupRequest;
import com.clustercontrol.cloud.bean.CreateInstanceByTemplateRequest;
import com.clustercontrol.cloud.bean.CreateInstanceRequest;
import com.clustercontrol.cloud.bean.CreateStorageBackupRequest;
import com.clustercontrol.cloud.bean.CreateStorageRequest;
import com.clustercontrol.cloud.bean.CreateTemplateRequest;
import com.clustercontrol.cloud.bean.Image;
import com.clustercontrol.cloud.bean.InstanceStateChange;
import com.clustercontrol.cloud.bean.ModifyAccountBillingDetailRequest;
import com.clustercontrol.cloud.bean.ModifyAccountResourceRequest;
import com.clustercontrol.cloud.bean.ModifyCloudUserRequest;
import com.clustercontrol.cloud.bean.ModifyTemplateRequest;
import com.clustercontrol.cloud.bean.RegistNodeRequest;
import com.clustercontrol.cloud.bean.RestoreInstanceRequest;
import com.clustercontrol.cloud.bean.RestoreStorageRequest;
import com.clustercontrol.cloud.bean.Snapshot;
import com.clustercontrol.cloud.bean.StartInstanceRequest;
import com.clustercontrol.cloud.bean.StopInstanceRequest;
import com.clustercontrol.cloud.bean.Zone;
import com.clustercontrol.cloud.validation.annotation.Into;
import com.clustercontrol.cloud.validation.annotation.NotNull;
import com.clustercontrol.fault.InvalidRole;
import com.clustercontrol.fault.InvalidUserPass;

/**
 * クラウド管理オプションの Web サービス API 定義。
 * 
 *
 */
@WebService(targetNamespace = "http://cloud.ws.clustercontrol.com")
public interface CloudEndpoint {
	/**
	 * クラウドインスタンスの作成を対象のクラウドプラットフォームへ指示し、関連するノードも合わせて作成します。
	 * 
	 * @param roleId クラウドユーザーに紐づくロール ID。
	 * @param region リージョン名。
	 * @param request クラウドインスタンス作成情報。
	 * @return クラウド管理へ登録されたクライドインスタンスの詳細。
	 * @throws CloudManagerFault
	 * @throws InvalidUserPass
	 * @throws InvalidRole
	 */
	@WebMethod
	CloudInstance addInstance(@WebParam(name = "roleId") String roleId, @WebParam(name = "region") String region, @WebParam(name = "request") CreateInstanceRequest request) throws CloudManagerFault, InvalidUserPass, InvalidRole;

	/**
	 * テンプレートに従ったクラウドインスタンスの作成を対象のクラウドプラットフォームへ指示し、関連するノードも合わせて作成します。
	 * なお、インスタンスを作成時に、エージェントが自動起動するように設定された イメージを指定すると、起動後テンプレートジョブが実行できます。
	 * 
	 * @param roleId クラウドユーザーに紐づくロール ID。
	 * @param region リージョン名。
	 * @param request テンプレート付きのクラウドインスタンス作成情報。
	 * @return クラウド管理オプションに登録されたクライドインスタンスの詳細。
	 * @throws CloudManagerFault
	 * @throws InvalidUserPass
	 * @throws InvalidRole
	 */
	@WebMethod
	CloudInstance addInstanceByTemplate(@WebParam(name = "roleId") String roleId, @WebParam(name = "region") String region, @WebParam(name = "request") CreateInstanceByTemplateRequest request) throws CloudManagerFault, InvalidUserPass, InvalidRole;

	/**
	 * 作成済みのクラウドインスタンスバックアップ情報から、クラウドインスタンスを復元します。
	 * 
	 * @param roleId クラウドユーザーに紐づくロール ID。
	 * @param region リージョン名。
	 * @param request クラウドインスタンスバックアップ情報。
	 * @return クラウド管理オプションに登録されたクライドインスタンスの詳細。
	 * @throws CloudManagerFault
	 * @throws InvalidUserPass
	 * @throws InvalidRole
	 */
	@WebMethod
	CloudInstance restoreInstance(@WebParam(name = "roleId") String roleId, @WebParam(name = "region") String region, @WebParam(name = "request") RestoreInstanceRequest request) throws CloudManagerFault, InvalidUserPass, InvalidRole;

	/**
	 * 対象のクラウド からインスタンスを削除し、ノードが紐づいている場合は、併せて削除されます。
	 * 
	 * @param roleId クラウドユーザーに紐づくロール ID。
	 * @param region リージョン名。
	 * @param instanceId 削除するクラウドインスタンス Id。
	 * @throws CloudManagerFault
	 * @throws InvalidUserPass
	 * @throws InvalidRole
	 */
	@WebMethod
	void removeInstance(@WebParam(name = "roleId") String roleId, @WebParam(name = "region") String region, @WebParam(name = "instanceId") String instanceId) throws CloudManagerFault, InvalidUserPass, InvalidRole;

	/**
	 * 作成済みのクラウドインスタンス情報を取得します。
	 * 
	 * @param roleId クラウドユーザーに紐づくロール ID。
	 * @param region リージョン名。
	 * @param instanceId クラウドインスタンス Id。
	 * @return クラウドインスタンス情報。
	 * @throws CloudManagerFault
	 * @throws InvalidUserPass
	 * @throws InvalidRole
	 */
	@WebMethod
	CloudInstance getInstance(@WebParam(name = "roleId") String roleId, @WebParam(name = "region") String region, @WebParam(name = "instanceId") String instanceId) throws CloudManagerFault, InvalidUserPass, InvalidRole;

	/**
	 * 指定したリージョンに作成されているインスタンス情報のリストを取得します。
	 * 取得できるリストは、実行ユーザに紐づくクラウドユーザの参照可能なインスタンスに限ります。
	 * 
	 * @param roleId クラウドユーザーに紐づくロール ID。
	 * @param region リージョン名。
	 * @return クラウドインスタンス情報の一覧。
	 * @throws CloudManagerFault
	 * @throws InvalidUserPass
	 * @throws InvalidRole
	 */
	@WebMethod
	List<CloudInstance> getInstancesByRegion(@WebParam(name = "roleId") String roleId, @WebParam(name = "region") String region) throws CloudManagerFault, InvalidUserPass, InvalidRole;

	/**
	 * 指定したクラウドインスタンスを起動します。
	 * エージェントが自動起動可能なノードは、<br>
	 * クラウドインスタンス作成時にテンプレートを指定した場合、
	 * クラウドインスタンス起動後に、テンプレートに既定にて指定されているテンプレートジョブが実施されます。
	 * また、テンプレートに指定されている既定のテンプレートジョブを切り替えたり、テンプレートが指定されていないエージェントでもテンプレートジョブを指定して起動することもできます。
	 * 
	 * @param roleId クラウドユーザーに紐づくロール ID。
	 * @param region リージョン名。
	 * @param request クラウドインスタンス起動情報。
	 * @return 起動後のクラウドインスタンスの状態情報。
	 * @throws CloudManagerFault
	 * @throws InvalidUserPass
	 * @throws InvalidRole
	 */
	@WebMethod
	InstanceStateChange startInstance(@WebParam(name = "roleId") String roleId, @WebParam(name = "region") String region, @WebParam(name = "request") StartInstanceRequest request) throws CloudManagerFault, InvalidUserPass, InvalidRole;

	/**
	 * 指定したクラウドインスタンスを停止します。
	 * クラウドインスタンス作成時にテンプレートを指定した場合、
	 * クラウドインスタンス停止前に、テンプレートに既定にて指定されているテンプレートジョブが実施されます。
	 * また、テンプレートに指定されている既定のテンプレートジョブを切り替えたり、テンプレートが指定されていないエージェントでもテンプレートジョブを指定して停止することもできます。
	 * 
	 * @param roleId クラウドユーザーに紐づくロール ID。
	 * @param region リージョン名。
	 * @param request クラウドインスタンス停止情報。
	 * @return 停止後のクラウドインスタンスの状態情報。
	 * @throws CloudManagerFault
	 * @throws InvalidUserPass
	 * @throws InvalidRole
	 */
	@WebMethod
	InstanceStateChange stopInstance(@WebParam(name = "roleId") String roleId, @WebParam(name = "region") String region, @WebParam(name = "request") StopInstanceRequest request) throws CloudManagerFault, InvalidUserPass, InvalidRole;

	/**
	 * クラウドユーザを追加します。
	 * クラウドユーザと Hinemos ユーザは一対一の関係となり、クラウドユーザに紐づいていない Hinemos ユーザが存在しない場合は、例外が発生します。
	 * なお、クラウドユーザと紐づいた Hinemos ユーザーは、クラウドユーザを削除しないと削除することができません。
	 * 
	 * @param request クラウドユーザー情報。
	 * @return 作成したクラウドユーザーの詳細。
	 * @throws CloudManagerFault
	 * @throws InvalidUserPass
	 * @throws InvalidRole
	 */
	@WebMethod
	CloudUser addCloudUser(@NotNull @Into @WebParam(name = "request") CreateCloudUserRequest request) throws CloudManagerFault, InvalidUserPass, InvalidRole;

	/**
	 * 指定したクラウドユーザを取得します。
	 * 
	 * @param userId ユーザーId。
	 * @return クラウドユーザー情報。
	 * @throws CloudManagerFault
	 * @throws InvalidUserPass
	 * @throws InvalidRole
	 */
	@WebMethod
	CloudUser getCloudUser(@WebParam(name = "userId") String userId) throws CloudManagerFault, InvalidUserPass, InvalidRole;

	/**
	 * 指定したクラウドユーザを削除します。
	 * 
	 * @param userId ユーザーId。
	 * @throws CloudManagerFault
	 * @throws InvalidUserPass
	 * @throws InvalidRole
	 */
	@WebMethod
	void removeCloudUser(@WebParam(name = "userId") String userId) throws CloudManagerFault, InvalidUserPass, InvalidRole;

	/**
	 * 指定したアカウントリソースに所属する全てのクラウドユーザを取得します。
	 * 
	 * @param アカウントリソース ID。
	 * @return クラウドユーザー情報一覧。
	 * @throws CloudManagerFault
	 * @throws InvalidUserPass
	 * @throws InvalidRole
	 */
	@WebMethod
	List<CloudUser> getCloudUsersByAccountResource(@WebParam(name = "accountResourceId") String accountResourceId) throws CloudManagerFault, InvalidUserPass, InvalidRole;

	/**
	 * 指定したクラウドユーザの設定を変更します。
	 * クラウドユーザID 以外は、null 指定が可能となり、値が指定された項目のみ更新されます。
	 * 
	 * @param request クラウドユーザー情報。
	 * @return クラウドユーザー情報。
	 * @throws CloudManagerFault
	 * @throws InvalidUserPass
	 * @throws InvalidRole
	 */
	@WebMethod
	CloudUser modifyCloudUser(@WebParam(name = "request") ModifyCloudUserRequest request) throws CloudManagerFault, InvalidUserPass, InvalidRole;

	/**
	 * クラウドタイプを取得します。
	 * 
	 * @return クラウドタイプ種別。
	 * @throws CloudManagerFault
	 * @throws InvalidUserPass
	 * @throws InvalidRole
	 */
	@WebMethod
	List<CloudType> getAllSupportedCloudTypes() throws CloudManagerFault, InvalidUserPass, InvalidRole;

	/**
	 * AWS の全てのリージョンを返します。
	 * 
	 * @param cloudServiceId　クラウドサービス ID。
	 * @return リージョン情報。
	 * @throws CloudManagerFault
	 * @throws InvalidUserPass
	 * @throws InvalidRole
	 */
	@WebMethod
	List<CloudRegion> getRegionsByService(@WebParam(name = "cloudServiceId") String cloudServiceId) throws CloudManagerFault, InvalidUserPass, InvalidRole;
	
	/**
	 * 指定したリージョンに所属するアベイラビリティゾーンを返します。
	 * 
	 * @param roleId クラウドユーザーに紐づくロール ID。
	 * @param region リージョン名。
	 * @return ゾーン一覧
	 * @throws CloudManagerFault
	 * @throws InvalidUserPass
	 * @throws InvalidRole
	 */
	@WebMethod
	List<Zone> getZones(@WebParam(name = "roleId") String roleId, @WebParam(name = "region") String region) throws CloudManagerFault, InvalidUserPass, InvalidRole;

	/**
	 * 指定したリージョンおよびフィルター条件に該当する AMI イメージを返します。
	 * 
	 * @param roleId クラウドユーザーに紐づくロール ID。
	 * @param region リージョン名。
	 * @param filters フィルター情報。
	 * @return AMI イメージ。
	 * @throws CloudManagerFault
	 * @throws InvalidUserPass
	 * @throws InvalidRole
	 */
	@WebMethod
	List<Image> getImagesWithFilter(@WebParam(name = "roleId") String roleId, @WebParam(name = "region") String region, @WebParam(name = "filters") List<Filter> filters) throws CloudManagerFault, InvalidUserPass, InvalidRole;

	/**
	 * 選択可能なインスタンス種別の一覧を取得します。
	 * 
	 * @param serviceId　クラウドサービス ID。
	 * @return インスタンス種別。
	 * @throws CloudManagerFault
	 * @throws InvalidUserPass
	 * @throws InvalidRole
	 */
	@WebMethod
	List<String> getInstanceFlavorsByService(@WebParam(name = "serviceId") String serviceId) throws CloudManagerFault, InvalidUserPass, InvalidRole;

	/**
	 * 指定したリージョンおよびフィルターに該当するスナップショットの一覧を取得します。
	 * フィルターの設定方法は、以下のリンクを参考にしてください。
	 * 
	 * @param roleId クラウドユーザーに紐づくロール ID。
	 * @param region リージョン名。
	 * @param filter フィルター一覧。
	 * @return スナップショット一覧。
	 * @throws CloudManagerFault
	 * @throws InvalidUserPass
	 * @throws InvalidRole
	 */
	@WebMethod
	List<Snapshot> getSnapshotsWithFilter(@WebParam(name = "roleId") String roleId, @WebParam(name = "region") String region, @WebParam(name = "filter") List<Filter> filter) throws CloudManagerFault, InvalidUserPass, InvalidRole;

	/**
	 * 指定したインスタンスおよびストレージをアタッチします。
	 * 
	 * @param roleId クラウドユーザーに紐づくロール ID。
	 * @param region リージョン名。
	 * @param instanceId インスタンス ID。
	 * @param storageId ストレージ ID。
	 * @param device デバイス名。
	 * @throws CloudManagerFault
	 * @throws InvalidUserPass
	 * @throws InvalidRole
	 */
	@WebMethod
	void attachStorage(@WebParam(name = "roleId") String roleId, @WebParam(name = "region") String region, @WebParam(name = "instanceId") String instanceId, @WebParam(name = "storageId") String storageId, @WebParam(name = "device") String device) throws CloudManagerFault, InvalidUserPass, InvalidRole ;

	/**
	 * 指定したインスタンスからストレージをデタッチします。
	 * 
	 * @param roleId クラウドユーザーに紐づくロール ID。
	 * @param region リージョン名。
	 * @param instanceId インスタンス ID。
	 * @param storageId ストレージ ID。
	 * @throws CloudManagerFault
	 * @throws InvalidUserPass
	 * @throws InvalidRole
	 */
	@WebMethod
	void detachStorage(@WebParam(name = "roleId") String roleId, @WebParam(name = "region") String region, @WebParam(name = "instanceId") String instanceId, @WebParam(name = "storageId") String storageId) throws CloudManagerFault, InvalidUserPass, InvalidRole ;

	/**
	 * ストレージを追加します。
	 * 
	 * @param roleId クラウドユーザーに紐づくロール ID。
	 * @param region リージョン名。
	 * @param request ストレージ作成情報
	 * @return ストレージ情報
	 * @throws CloudManagerFault
	 * @throws InvalidUserPass
	 * @throws InvalidRole
	 */
	@WebMethod
	CloudStorage addStorage(@WebParam(name = "roleId") String roleId, @WebParam(name = "region") String region, @WebParam(name = "request") CreateStorageRequest request) throws CloudManagerFault, InvalidUserPass, InvalidRole ;

	/**
	 * 作成済みのストレージバックアップ情報から、ストレージを復元します。
	 * 
	 * @param roleId クラウドユーザーに紐づくロール ID。
	 * @param region リージョン名。
	 * @param request リストア情報
	 * @return ストレージ情報
	 * @throws CloudManagerFault
	 * @throws InvalidUserPass
	 * @throws InvalidRole
	 */
	@WebMethod
	CloudStorage restoreStorage(@WebParam(name = "roleId") String roleId, @WebParam(name = "region") String region, @WebParam(name = "request") RestoreStorageRequest request) throws CloudManagerFault, InvalidUserPass, InvalidRole;

	/**
	 * 指定したストレージを削除します。
	 * 
	 * @param roleId クラウドユーザーに紐づくロール ID。
	 * @param region リージョン名。
	 * @param storageId ストレージ Id。
	 * @throws CloudManagerFault
	 * @throws InvalidUserPass
	 * @throws InvalidRole
	 */
	@WebMethod
	void removeStorage(@WebParam(name = "roleId") String roleId, @WebParam(name = "region") String region, @WebParam(name = "storageId") String storageId) throws CloudManagerFault, InvalidUserPass, InvalidRole;

	/**
	 * 指定したストレージを取得します。
	 * 
	 * @param roleId クラウドユーザーに紐づくロール ID。
	 * @param region リージョン名。
	 * @param storageId ストレージ ID。
	 * @return ストレージ情報。
	 * @throws CloudManagerFault
	 * @throws InvalidUserPass
	 * @throws InvalidRole
	 */
	@WebMethod
	CloudStorage getStorage(@WebParam(name = "roleId") String roleId, @WebParam(name = "region") String region,  @WebParam(name = "storageId") String storageId) throws CloudManagerFault, InvalidUserPass, InvalidRole;

	/**
	 * 指定したリージョンに登録されているストレージの一覧を取得します。
	 * 
	 * @param roleId クラウドユーザーに紐づくロール ID。
	 * @param region リージョン名。
	 * @return ストレージ情報一覧。
	 * @throws CloudManagerFault
	 * @throws InvalidUserPass
	 * @throws InvalidRole
	 */
	@WebMethod
	List<CloudStorage> getStoragesByRegion(@WebParam(name = "roleId") String roleId, @WebParam(name = "region") String region) throws CloudManagerFault, InvalidUserPass, InvalidRole;

	/**
	 * 選択可能なストレージ種別の一覧を取得します。
	 * 
	 * @param cloudServiceId クラウドサービス ID。
	 * @return ストレージ種別。
	 * @throws CloudManagerFault
	 * @throws InvalidUserPass
	 * @throws InvalidRole
	 */
	@WebMethod
	List<String> getStorageFlavorsByService(@WebParam(name = "cloudServiceId") String cloudServiceId) throws CloudManagerFault, InvalidUserPass, InvalidRole;

	/**
	 * 指定したストレージのバックアップを作成します。
	 * 
	 * @param roleId クラウドユーザーに紐づくロール ID。
	 * @param region リージョン名。
	 * @param request ストレージバックアップ作成情報。
	 * @return ストレージバックアップ情報。
	 * @throws CloudManagerFault
	 * @throws InvalidUserPass
	 * @throws InvalidRole
	 */
	@WebMethod
	CloudStorageBackup addStorageBackup(@WebParam(name = "roleId") String roleId, @WebParam(name = "region") String region, @WebParam(name = "request") CreateStorageBackupRequest request) throws CloudManagerFault, InvalidUserPass, InvalidRole;

	/**
	 * 指定したストレージのバックアップ情報を削除します。
	 * 
	 * @param roleId クラウドユーザーに紐づくロール ID。
	 * @param region リージョン名。
	 * @param storageBackupId ストレージバックアップ ID。
	 * @throws CloudManagerFault
	 * @throws InvalidUserPass
	 * @throws InvalidRole
	 */
	@WebMethod
	void removeStorageBackup(@WebParam(name = "roleId") String roleId, @WebParam(name = "region") String region, @WebParam(name = "storageBackupId") String storageBackupId) throws CloudManagerFault, InvalidUserPass, InvalidRole;
	
	/**
	 * 指定したストレージバックアップの情報を取得します。
	 * 
	 * @param roleId クラウドユーザーに紐づくロール ID。
	 * @param region リージョン名。
	 * @param storageBackupId ストレージバックアップ ID。
	 * @return ストレージバックアップ情報。
	 * @throws CloudManagerFault
	 * @throws InvalidUserPass
	 * @throws InvalidRole
	 */
	@WebMethod
	CloudStorageBackup getStorageBackup(@WebParam(name = "roleId") String roleId, @WebParam(name = "region") String region, @WebParam(name = "storageBackupId") String storageBackupId) throws CloudManagerFault, InvalidUserPass, InvalidRole;
	
	/**
	 * 指定したリージョンに登録されているストレージバックアップ情報の一覧を取得します。
	 * 
	 * @param roleId クラウドユーザーに紐づくロール ID。
	 * @param region リージョン名。
	 * @return ストレージバックアップ情報。
	 * @throws CloudManagerFault
	 * @throws InvalidUserPass
	 * @throws InvalidRole
	 */
	@WebMethod
	List<CloudStorageBackup> getStorageBackupsByRegion(@WebParam(name = "roleId") String roleId, @WebParam(name = "region") String region) throws CloudManagerFault, InvalidUserPass, InvalidRole;
	
	/**
	 * 指定したインスタンスのバックアップ情報を取得します。
	 * 
	 * @param roleId クラウドユーザーに紐づくロール ID。
	 * @param region リージョン名。
	 * @param request クラウドインスタンスバックアップ作成情報。
	 * @return クラウドインスタンスバックアップ情報。
	 * @throws CloudManagerFault
	 * @throws InvalidUserPass
	 * @throws InvalidRole
	 */
	@WebMethod
	CloudInstanceBackup addInstanceBackup(@WebParam(name = "roleId") String roleId, @WebParam(name = "region") String region, @WebParam(name = "request") CreateInstanceBackupRequest request) throws CloudManagerFault, InvalidUserPass, InvalidRole;

	/**
	 * 指定したインスタンスのバックアップ情報を削除します。
	 * 
	 * @param roleId クラウドユーザーに紐づくロール ID。
	 * @param region リージョン名。
	 * @param instanceBackupId インスタンスバックアップ ID。
	 * @throws CloudManagerFault
	 * @throws InvalidUserPass
	 * @throws InvalidRole
	 */
	@WebMethod
	void removeInstanceBackup(@WebParam(name = "roleId") String roleId, @WebParam(name = "region") String region, @WebParam(name = "instanceBackupId") String instanceBackupId) throws CloudManagerFault, InvalidUserPass, InvalidRole;
	
	/**
	 * 指定したインスタンスのバックアップ情報を取得します。
	 * 
	 * @param roleId クラウドユーザーに紐づくロール ID。
	 * @param region リージョン名。
	 * @param instanceBackupId インスタンスバックアップ ID。
	 * @return クラウドインスタンスバックアップ情報。
	 * @throws CloudManagerFault
	 * @throws InvalidUserPass
	 * @throws InvalidRole
	 */
	@WebMethod
	CloudInstanceBackup getInstanceBackup(@WebParam(name = "roleId") String roleId, @WebParam(name = "region") String region, @WebParam(name = "instanceBackupId") String instanceBackupId) throws CloudManagerFault, InvalidUserPass, InvalidRole;
	
	/**
	 * 指定したリージョンに登録されているインスタンスバックアップ情報のリストを取得します。
	 * 
	 * @param roleId クラウドユーザーに紐づくロール ID。
	 * @param region リージョン名。
	 * @return クラウドインスタンスバックアップ情報一覧。
	 * @throws CloudManagerFault
	 * @throws InvalidUserPass
	 * @throws InvalidRole
	 */
	@WebMethod
	List<CloudInstanceBackup> getInstanceBackupsByRegion(@WebParam(name = "roleId") String roleId, @WebParam(name = "region") String region) throws CloudManagerFault, InvalidUserPass, InvalidRole;

	/**
	 * ノードと紐づいていないインスタンスにノードを紐づけます。
	 * 
	 * @param roleId クラウドユーザーに紐づくロール ID。
	 * @param region リージョン名。
	 * @param request ノード登録情報。
	 * @return クラウドインスタンス情報。
	 * @throws InvalidUserPass
	 * @throws InvalidRole
	 * @throws CloudManagerFault
	 */
	@WebMethod
	CloudInstance registNode(@WebParam(name = "roleId") String roleId, @WebParam(name = "region") String region, @WebParam(name = "request") RegistNodeRequest request) throws InvalidUserPass, InvalidRole, CloudManagerFault;

	/**
	 * ノードに紐づいたインスタンスで、既に EC2 インスタンスが削除されているインスタンスを削除します。
	 * 
	 * @param roleId クラウドユーザーに紐づくロール ID。
	 * @param region リージョン名。
	 * @param instanceId インスタンス Id。
	 * @throws InvalidUserPass
	 * @throws InvalidRole
	 * @throws CloudManagerFault
	 */
	@WebMethod
	void unregistNode(@WebParam(name = "roleId") String roleId, @WebParam(name = "region") String region, @WebParam(name = "instanceId") String instanceId) throws InvalidUserPass, InvalidRole, CloudManagerFault;
	
	/**
	 * テンプレートを追加します。
	 * 
	 * @param roleId クラウドユーザーに紐づくロール ID。
	 * @param template テンプレート情報。
	 * @return テンプレート情報。
	 * @throws CloudManagerFault
	 * @throws InvalidUserPass
	 * @throws InvalidRole
	 */
	@WebMethod
	CloudTemplate addTemplate(@WebParam(name = "roleId") String roleId, @WebParam(name = "request") CreateTemplateRequest request) throws CloudManagerFault, InvalidUserPass, InvalidRole;

	/**
	 * 指定したテンプレートを削除します。
	 * 
	 * @param roleId クラウドユーザーに紐づくロール ID。
	 * @param templateId テンプレート Id。
	 * @throws CloudManagerFault
	 * @throws InvalidUserPass
	 * @throws InvalidRole
	 */
	@WebMethod
	void removeTemplate(@WebParam(name = "roleId") String roleId, @WebParam(name = "templateId") String templateId) throws CloudManagerFault, InvalidUserPass, InvalidRole;
	
	/**
	 * 指定したテンプレートを更新します。
	 * 
	 * @param roleId クラウドユーザーに紐づくロール ID。
	 * @param template テンプレート情報。
	 * @return テンプレート情報。
	 * @throws CloudManagerFault
	 * @throws InvalidUserPass
	 * @throws InvalidRole
	 */
	@WebMethod
	CloudTemplate modifyTemplate(@WebParam(name = "roleId") String roleId, @WebParam(name = "request") ModifyTemplateRequest request) throws CloudManagerFault, InvalidUserPass, InvalidRole;
	
	/**
	 * 指定したテンプレートを取得します。
	 * 
	 * @param roleId クラウドユーザーに紐づくロール ID。
	 * @param templateId テンプレート ID。
	 * @return テンプレート情報。
	 * @throws CloudManagerFault
	 * @throws InvalidUserPass
	 * @throws InvalidRole
	 */
	@WebMethod
	CloudTemplate getTemplate(@WebParam(name = "roleId") String roleId, @WebParam(name = "templateId") String templateId) throws CloudManagerFault, InvalidUserPass, InvalidRole;
	
	/**
	 * 実行ユーザが参照可能な全てのテンプレートを取得します。
	 * 
	 * @param roleId クラウドユーザーに紐づくロール ID。
	 * @return テンプレート情報一覧。
	 * @throws CloudManagerFault
	 * @throws InvalidUserPass
	 * @throws InvalidRole
	 */
	@WebMethod
	List<CloudTemplate> getTemplates(@WebParam(name = "roleId") String roleId) throws CloudManagerFault, InvalidUserPass, InvalidRole;

	/**
	 * テンプレートに設定可能なスクリプトの一覧を取得します。
	 * 
	 * @param roleId クラウドユーザーに紐づくロール ID。
	 * @return スクリプト一覧。
	 * @throws CloudManagerFault
	 */
	@WebMethod
	List<String> getTemplateScripts(@WebParam(name = "roleId") String roleId) throws CloudManagerFault;

	/**
	 * アカウントリソースを作成します。。
	 * 
	 * @param request アカウントリソース情報。
	 * @return 作成されたアカウントリソース。
	 * @throws CloudManagerFault
	 * @throws InvalidUserPass
	 * @throws InvalidRole
	 */
	@WebMethod
	CloudAccountResource addAccountResource(@WebParam(name = "request") CreateAccountResourceRequest request) throws CloudManagerFault, InvalidUserPass, InvalidRole;

	/**
	 * 指定したアカウントリソースを変更します。
	 * 
	 * @param request アカウントリソース情報。
	 * @return 変更後のアカウントリソース。
	 * @throws CloudManagerFault
	 * @throws InvalidUserPass
	 * @throws InvalidRole
	 */
	@WebMethod
	CloudAccountResource modifyAccountResource(@WebParam(name = "request") ModifyAccountResourceRequest request) throws CloudManagerFault, InvalidUserPass, InvalidRole;

	/**
	 * 指定したアカウントリソースの課金設定を変更します。
	 * 
	 * @param request アカウントリソース情報。
	 * @return 変更後のアカウントリソース。
	 * @throws CloudManagerFault
	 * @throws InvalidUserPass
	 * @throws InvalidRole
	 */
	@WebMethod
	CloudAccountResource modifyAccountBillingDetail(@WebParam(name = "request") ModifyAccountBillingDetailRequest request) throws CloudManagerFault, InvalidUserPass, InvalidRole;

	/**
	 * 指定したアカウントリソースを削除します。
	 * 
	 * @param accountResourceId アカウントリソース ID。
	 * @throws CloudManagerFault
	 * @throws InvalidUserPass
	 * @throws InvalidRole
	 */
	@WebMethod
	void removeAccountResource(@WebParam(name = "accountResourceId") String accountResourceId) throws CloudManagerFault, InvalidUserPass, InvalidRole;

	/**
	 * 指定したアカウントリソースを取得する。
	 * 
	 * @param accountResourceId アカウントリソース ID。
	 * @return アカウントリソース。
	 * @throws CloudManagerFault
	 * @throws InvalidUserPass
	 * @throws InvalidRole
	 */
	@WebMethod
	CloudAccountResource getAccountResource(@WebParam(name = "accountResourceId") String accountResourceId) throws CloudManagerFault, InvalidUserPass, InvalidRole;

	/**
	 * アカウントリソースの一覧を取得します。
	 * 
	 * @return アカウントリソースの一覧。
	 * @throws CloudManagerFault
	 * @throws InvalidUserPass
	 * @throws InvalidRole
	 */
	@WebMethod
	List<CloudAccountResource> getAllAccountResources() throws CloudManagerFault, InvalidUserPass, InvalidRole;

	/**
	 * 指定したクラウドサービスに紐付くリージョンの一覧を取得します。
	 * 
	 * @param cloudServiceId クラウドサービス ID。
	 * @return リージョン一覧。
	 * @throws CloudManagerFault
	 * @throws InvalidUserPass
	 * @throws InvalidRole
	 */
	@WebMethod
	List<CloudRegion> getCloudRegionsByService(@WebParam(name = "cloudServiceId") String cloudServiceId) throws CloudManagerFault, InvalidUserPass, InvalidRole;

	/**
	 * クラウドサービスを取得します。
	 * 
	 * @param cloudServiceId クラウドサービス ID。
	 * @return クラウドサービス。
	 * @throws CloudManagerFault
	 * @throws InvalidUserPass
	 * @throws InvalidRole
	 */
	@WebMethod
	CloudService getCloudService(@WebParam(name = "cloudServiceId") String cloudServiceId) throws CloudManagerFault, InvalidUserPass, InvalidRole;

	/**
	 * クラウドサービスの一覧を取得します。
	 * 
	 * @return クラウドサービスの一覧。
	 * @throws CloudManagerFault
	 * @throws InvalidUserPass
	 * @throws InvalidRole
	 */
	@WebMethod
	List<CloudService> getAllCloudServices() throws CloudManagerFault, InvalidUserPass, InvalidRole;
	
	/**
	 * クラウド課金監視のターゲットになるサービスの一覧を取得する。
	 * 
	 * @param accountResourceId アカウントリソース ID。
	 * @return クラウド課金監視のターゲット一覧。
	 * @throws CloudManagerFault
	 * @throws InvalidUserPass
	 * @throws InvalidRole
	 */
	@WebMethod
	List<String> getCloudServicesForBilling(@WebParam(name = "accountResourceId") String accountResourceId) throws CloudManagerFault, InvalidUserPass, InvalidRole;
	
	/**
	 * マネージャからの呼び出し待機中エージェントに対して、通信を開始する。
	 * 
	 * @param facilityId ファシリティ ID。
	 * @throws CloudManagerFault
	 * @throws InvalidUserPass
	 * @throws InvalidRole
	 */
	@WebMethod
	void establishAgentConnection(@WebParam(name = "facilityId") String facilityId) throws CloudManagerFault, InvalidUserPass, InvalidRole;
	
	/**
	 * クラウドユーザーに紐付け可能なロールの一覧を取得する。
	 * 戻される結果は、現在のアカウントが利用可能なロールの一覧から、既に紐付けられているロールを除外した内容になる。
	 * 
	 * @return クラウドユーザーに紐付け可能なロールの一覧。
	 * @throws CloudManagerFault
	 * @throws InvalidUserPass
	 * @throws InvalidRole
	 */
	@WebMethod
	List<AvailableRole> getAvailableRoles() throws CloudManagerFault, InvalidUserPass, InvalidRole;
}