/*
Copyright (C) 2013 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.
 */
package com.clustercontrol.cloud.factory;

import java.util.Date;
import java.util.List;
import java.util.Locale;
import java.util.concurrent.Callable;

import org.apache.log4j.Logger;

import com.clustercontrol.bean.PriorityConstant;
import com.clustercontrol.cloud.CloudManagerFault;
import com.clustercontrol.cloud.Messages;
import com.clustercontrol.cloud.SessionService;
import com.clustercontrol.cloud.bean.TemplateJobsArgument;
import com.clustercontrol.cloud.commons.CloudConstants;
import com.clustercontrol.cloud.commons.CloudPropertyConstants;
import com.clustercontrol.cloud.util.JobControllerBeanWrapper;
import com.clustercontrol.fault.FacilityNotFound;
import com.clustercontrol.fault.HinemosUnknown;
import com.clustercontrol.fault.InvalidObjectPrivilege;
import com.clustercontrol.fault.InvalidRole;
import com.clustercontrol.fault.JobInfoNotFound;
import com.clustercontrol.fault.JobInvalid;
import com.clustercontrol.fault.JobMasterNotFound;
import com.clustercontrol.fault.JobSessionDuplicate;
import com.clustercontrol.fault.NotifyNotFound;
import com.clustercontrol.fault.UserNotFound;
import com.clustercontrol.jobmanagement.bean.JobInfo;
import com.clustercontrol.jobmanagement.bean.JobParameterInfo;
import com.clustercontrol.jobmanagement.bean.JobTreeItem;
import com.clustercontrol.jobmanagement.bean.JobTriggerInfo;
import com.clustercontrol.jobmanagement.bean.JobTriggerTypeConstant;
import com.clustercontrol.jobmanagement.session.JobControllerBean;
import com.clustercontrol.jobmanagement.util.JobUtil;
import com.clustercontrol.notify.bean.OutputBasicInfo;
import com.clustercontrol.notify.factory.NotifyEventTaskFactory;
import com.clustercontrol.plugin.impl.AsyncWorkerPlugin;
import com.clustercontrol.repository.bean.NodeInfo;
import com.clustercontrol.repository.session.RepositoryControllerBean;

public class TemplateJobUtil implements CloudConstants {
	public interface JobEventListener {
		void jobEventOccurerd(String facilityId, String jobunitId, String jobId) throws Exception;
	}
	 
	public static void asyncExecute(String facilityId, String jobunitId, String jobId, List<TemplateJobsArgument> arguments, JobEventListener beforeExecute, JobEventListener afterExecute) {
		JobExecutor jobExecutor = new JobExecutor(facilityId, jobunitId, jobId, arguments, beforeExecute, afterExecute);
		jobExecutor.asyncExecute();
	}
	
	public static void syncExecute(String facilityId, String jobunitId, String jobId, List<TemplateJobsArgument> arguments, JobEventListener beforeExecute, JobEventListener afterExecute) throws Exception {
		JobExecutor jobExecutor = new JobExecutor(facilityId, jobunitId, jobId, arguments, beforeExecute, afterExecute);
		jobExecutor.syncExecute();
	}
	
	private static class JobExecutor {
		private String facilityId;
		private String jobunitId;
		private String jobId;
		private List<TemplateJobsArgument> arguments;
		private JobEventListener beforeExecute;
		private JobEventListener afterExecute;
		
		public JobExecutor(String facilityId, String jobunitId, String jobId, List<TemplateJobsArgument> arguments, JobEventListener beforeExecute, JobEventListener afterExecute) {
			this.jobunitId = jobunitId;
			this.jobId = jobId;
			this.facilityId = facilityId;
			this.arguments = arguments;
			this.beforeExecute = beforeExecute;
			this.afterExecute = afterExecute;
		}
		
		public void asyncExecute() {
			SessionService.execute(new Runnable() {
				@Override
				public void run() {
					try {
						syncExecute();
					}
					catch (Exception e) {
						Logger.getLogger(this.getClass()).error(e.getMessage(), e);
					}
				}
			});
		}		
		
		public void syncExecute() throws Exception {
			Logger logger = Logger.getLogger(this.getClass());
			if (beforeExecute != null) {
				beforeExecute.jobEventOccurerd(facilityId, jobunitId, jobId);
			}

			String sessionId = executeJob();

			if (afterExecute != null) {
				int maxCount = Integer.parseInt(CloudPropertyConstants.templatejob_endcheck_count.value());
				long interval = Long.parseLong(CloudPropertyConstants.templatejob_endcheck_interval.value());
				int counter = 0;
				while (counter < maxCount) {
					++counter;
					logger.debug("try to check end of job. facilityId:" + facilityId + ",jobId:" + jobId + ",sessionId:" + sessionId + " " + counter + "/" + maxCount + " time.");
					
					// isRunJob が不具合。終了したなら、true を返す。
					if (!JobUtil.isRunJob(sessionId, jobunitId, jobId)) {
						afterExecute.jobEventOccurerd(facilityId, jobunitId, jobId);
						break;
					}
					
					if (counter >= maxCount) {
						throw new CloudManagerFault("timed out waiting for end of job.");
					}
					
					Thread.sleep(interval);
				}
			}
		}		

		private String executeJob() throws CloudManagerFault, NotifyNotFound, HinemosUnknown, JobMasterNotFound, UserNotFound, InvalidRole, JobInvalid, InvalidObjectPrivilege, FacilityNotFound, JobInfoNotFound, JobSessionDuplicate {
			Logger logger = Logger.getLogger(this.getClass());
			logger.debug("runJob : jobunitId=" + jobunitId + ", jobId=" + jobId + ", facilityId=" + facilityId);
			
			JobControllerBean bean = JobControllerBeanWrapper.bean();
			
			JobTreeItem root = bean.getJobTree(null, true, Locale.getDefault());
			JobTreeItem awsRoot = null;
			for (JobTreeItem item: root.getChildren().get(0).getChildren()) {
				if (item.getData().getId().equals(jobunitId)) {
					awsRoot = item;
					break;
				}
			}
			
			if (awsRoot != null) {
				JobInfo jobInfo = bean.getJobFull(awsRoot.getData());
				awsRoot.setData(jobInfo);
				List<JobParameterInfo> paramList = jobInfo.getParam();
				if (paramList != null && arguments != null && arguments.size() > 0) {
					for (JobParameterInfo param: paramList) {
						for (TemplateJobsArgument arg: arguments) {
//							if (param.getParamId().equals(CONSTANT_TEMPLATE_JOB_PARAMETER_PREFIX + arg.getJobId())) {
							//仕様変更に伴い変更　v2.0.0　JobunitId　= AccountResourceId　のため成立する
							if (param.getParamId().equals(jobunitId + "_" + arg.getJobId())) {
								param.setValue(arg.getArg());
							}
						}
					}
//						jobInfo.setEdit(true);
					awsRoot.setData(jobInfo);
					
					bean.registerJobunit(awsRoot);
				}
			}
			else {
				throw new CloudManagerFault("Jobunit AWS not found.");
			}
							
			JobTriggerInfo triggerInfo = new JobTriggerInfo();
			triggerInfo.setTrigger_info(JobTriggerTypeConstant.STRING_MANUAL);
			triggerInfo.setTrigger_type(JobTriggerTypeConstant.TYPE_MANUAL);
			OutputBasicInfo info = new OutputBasicInfo();
			info.setFacilityId(facilityId);
			return bean.runJob(jobunitId, jobId, info, triggerInfo);
		}
	}
	
	public static class InternalEventNotifier implements Callable<Boolean> {
		private static final String MONITOR_ID = "SYS";
		private static final String APPLICATION = Messages.messages().getString("cloud.template.message.notify.application");
		private static final String MESSAGE_ID = "001";
		private static final int PRIORITY = PriorityConstant.TYPE_WARNING;
		
		private String pluginId;
		private String facilityId;
		private String message;
		
		public InternalEventNotifier(String pluginId, String facilityId, String message){
			this.pluginId = pluginId;
			this.facilityId = facilityId;
			this.message = message;
		}
		
		@Override
		public Boolean call() throws Exception {
			try {
				OutputBasicInfo output = new OutputBasicInfo();
				output.setPluginId(pluginId);
				output.setMonitorId(MONITOR_ID);
				output.setFacilityId(facilityId);

				RepositoryControllerBean repositoryLocal = new RepositoryControllerBean();
				NodeInfo nodeInfo = repositoryLocal.getNode(facilityId);
				output.setScopeText(nodeInfo.getFacilityName());
				output.setApplication(APPLICATION);
				output.setMessageId(MESSAGE_ID);

				output.setMessage(message);
//				output.setMessageOrg("-");
				output.setPriority(PRIORITY);

				output.setGenerationDate(new Date().getTime());

				//メッセージ送信
				AsyncWorkerPlugin.addTask(NotifyEventTaskFactory.class.getSimpleName(), output, false);
			}
			catch (Exception e) {
				Logger logger = Logger.getLogger(TemplateJobUtil.class);
				logger.error(e.getMessage(), e);
				return false;
			}
			return true;
		}
	}
}
