/*

 Copyright (C) 2011 NTT DATA Corporation

 This program is free software; you can redistribute it and/or
 Modify it under the terms of the GNU General Public License
 as published by the Free Software Foundation, version 2.

 This program is distributed in the hope that it will be
 useful, but WITHOUT ANY WARRANTY; without even the implied
 warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
 PURPOSE.  See the GNU General Public License for more details.

 */


package com.clustercontrol.systemlog.service;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.bean.HinemosModuleConstant;
import com.clustercontrol.commons.util.HinemosProperties;
import com.clustercontrol.monitor.run.bean.MonitorInfo;
import com.clustercontrol.monitor.run.bean.MonitorStringValueInfo;
import com.clustercontrol.notify.bean.OutputBasicInfo;
import com.clustercontrol.notify.session.NotifyControllerBean;
import com.clustercontrol.repository.bean.FacilityTreeAttributeConstant;
import com.clustercontrol.repository.session.RepositoryControllerBean;
import com.clustercontrol.systemlog.bean.SyslogMessage;

public class SystemLogNotifier {
	/** ログ出力のインスタンス  */
	private final static Log m_log = LogFactory.getLog(SystemLogNotifier.class);

	/** オリジナルメッセージ出力文字数 */
	public static final String _LOG_LINE_MAX_LENGTH_KEY = "log.line.max.length";
	public static final int _LOG_LINE_MAX_LENGTH_DEFAULT = 256;

	private final static int LOG_LINE_LENGTH;

	static {
		// intialize log line
		String logLineLengthStr = HinemosProperties.getProperty(_LOG_LINE_MAX_LENGTH_KEY);
		int logLineLength = _LOG_LINE_MAX_LENGTH_DEFAULT;
		try {
			if (logLineLengthStr != null) {
				logLineLength = Integer.parseInt(logLineLengthStr);
			}
		} catch (NumberFormatException e) {
			m_log.info("static() invalid configuration. (" + _LOG_LINE_MAX_LENGTH_KEY + " = " + logLineLengthStr + ")");
		}
		LOG_LINE_LENGTH = logLineLength;
		m_log.info("static() NotifyDirectSender initialized : LOG_LINE_LENGTH = " + LOG_LINE_LENGTH);
	}

	public void put(String receiverId, SyslogMessage syslog,
			MonitorInfo monitorInfo, MonitorStringValueInfo rule, String logFacilityId) {


		String message = "";
		if (HinemosModuleConstant.MONITOR_LOGFILE.equals(monitorInfo.getMonitorTypeId())) {
			message += "log.file=" + monitorInfo.getLogfileCheckInfo().getLogfile() + "\n";
		}
		message += "pattern=" + rule.getPattern() + "\nlog.line=" + syslog.rawSyslog;

		//マッチした場合
		// メッセージ作成（未登録ノードの時は、送信元ホスト名をスコープ名にする。）
		m_log.debug("call makeMessage.");
		OutputBasicInfo logOutput  = makeMessage(receiverId, message, syslog,
				monitorInfo, rule, logFacilityId, syslog.hostname);
		m_log.debug("called makeMessage.");

		//メッセージ送信処理
		m_log.debug("Process:" + rule.getDescription() + ":" + message);
		new NotifyControllerBean().notify(logOutput, monitorInfo.getNotifyGroupId());
		m_log.debug("sended.");
	}

	/**
	 * 引数で指定された情報より、ログ出力メッセージを生成し返します。
	 * 
	 * @param msg メッセージ
	 * @param logmsg syslogメッセージ情報
	 * @param moninfo 監視項目情報
	 * @param filterInfo フィルタ情報
	 * @param facilityID ファシリティID
	 * @param nodeName ノード名
	 * @return ログ出力メッセージ
	 * 
	 * @since 1.0.0
	 */
	private static OutputBasicInfo makeMessage(
			String receiverId,
			String msg,
			SyslogMessage syslog,
			MonitorInfo monInfo,
			MonitorStringValueInfo filterInfo,
			String facilityId,
			String nodeName) {

		m_log.debug("Make ObjectMsg");

		OutputBasicInfo output = new OutputBasicInfo();

		output.setMonitorId(filterInfo.getMonitorId());
		output.setFacilityId(facilityId);

		if (HinemosModuleConstant.MONITOR_LOGFILE.equals(monInfo.getMonitorTypeId())) {
			output.setPluginId(HinemosModuleConstant.MONITOR_LOGFILE);
		} else if (HinemosModuleConstant.MONITOR_SYSTEMLOG.equals(monInfo.getMonitorTypeId())) {
			output.setPluginId(HinemosModuleConstant.MONITOR_SYSTEMLOG);
		} else if (HinemosModuleConstant.MONITOR_WINEVENT.equals(monInfo.getMonitorTypeId())){
			output.setPluginId(HinemosModuleConstant.MONITOR_WINEVENT);
		} else {
			m_log.info("makeMessage() pluginId not found (monitorTypeId=" + monInfo.getMonitorTypeId() +")");
		}

		// 通知抑制を監視項目の「パターンマッチ表現」単位にするため、通知抑制用サブキーを設定する。
		output.setSubKey(filterInfo.getPattern());

		if(FacilityTreeAttributeConstant.UNREGISTEREFD_SCOPE.equals(facilityId)){
			//未登録ノードの場合には送信元のホスト名を表示する。
			output.setScopeText(nodeName);
		}else{
			// ファシリティのパスを設定する]
			try {
				m_log.debug("call getFacilityPath.");
				String facilityPath = new RepositoryControllerBean().getFacilityPath(facilityId, null);
				m_log.debug("called getFacilityPath.");

				output.setScopeText(facilityPath);
			} catch (Exception e) {
				m_log.warn("makeMessage() cannot get facility path.(facilityId = " + facilityId + ") : "
						+ e.getClass().getSimpleName() + ", " + e.getMessage(), e);
			}
		}

		output.setApplication(monInfo.getApplication());
		output.setMessageId(filterInfo.getMessageId());

		// メッセージに#[LOG_LINE]が入力されている場合は、
		// オリジナルメッセージに置換する。
		if (filterInfo.getMessage() != null) {
			String str = filterInfo.getMessage().replace("#[LOG_LINE]", syslog.message);
			if (str.length() > LOG_LINE_LENGTH) {
				str = str.substring(0, LOG_LINE_LENGTH);
			}
			output.setMessage(str);
		}

		output.setMessageOrg(msg);
		output.setPriority(filterInfo.getPriority());
		output.setGenerationDate(syslog.date);

		if(receiverId != null && !"".equals(receiverId)){
			output.setMultiId(receiverId);
		}

		return output;
	}
}
