/*

Copyright (C) 2012 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.

 */

package com.clustercontrol.systemlog.bean;

import java.io.Serializable;
import java.text.MessageFormat;
import java.text.ParseException;
import java.util.Calendar;
import java.util.Date;
import java.util.Locale;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

/**
 * syslog情報を保持するクラス
 * @author takahatat
 */
public class SyslogMessage implements Serializable {

	private static final long serialVersionUID = 1L;

	private static Log log = LogFactory.getLog(SyslogMessage.class);

	public Facility facility;
	public Severity severity;
	public long date;
	public String hostname;
	public String message;
	public String rawSyslog;

	public enum Facility { KERN, USER, MAIL, DAEMON, AUTH, SYSLOG, LPR, NEWS, UUCP, CRON, AUTHPRIV, FTP, NTP, SECURITY, CONSOLE,
		LOCAL0, LOCAL1, LOCAL2, LOCAL3, LOCAL4, LOCAL5, LOCAL6, LOCAL7 };

		public enum Severity { EMERG, ALERT, CRIT, ERR, WARNING, NOTICE, INFO, DEBUG };

		public SyslogMessage() {

		}

		public SyslogMessage(Facility facility, Severity severity, long date, String hostname, String message, String rawSyslog) {
			this.facility = facility;
			this.severity = severity;
			this.date = date;
			this.hostname = hostname;
			this.message = message;
			this.rawSyslog = rawSyslog;
		}

		/**
		 * syslogの文字列をパースし、その解析結果をSyslogMessageインスタンスとして返すAPI
		 * @param syslog syslogの文字列
		 * @return SyslogMessageインスタンス
		 * @throws ParseException syslogのフォーマットに従っていない文字列が与えられた場合
		 */
		public static SyslogMessage parse(String syslog) throws ParseException {
			if (log.isDebugEnabled()) {
				log.debug("parsing syslog : " + syslog);
			}

			MessageFormat syslogFormat = new MessageFormat("<{0,number,integer}>{1,date,MMM dd HH:mm:ss} {2} {3}", Locale.ENGLISH);

			// 文字列をパースしてヘッダ情報およびメッセージを取得する
			Object[] syslogArgs = syslogFormat.parse(syslog);

			if (log.isDebugEnabled() && syslogArgs != null) {
				int i = 0;
				for (Object arg : syslogArgs) {
					log.debug(String.format("syslog args [%d] : %s", i++, arg.toString()));
				}
			}

			long pri = (Long)syslogArgs[0];
			Date date = (Date)syslogArgs[1];
			String hostname = (String)syslogArgs[2];
			String msg = (String)syslogArgs[3];

			// ヘッダ情報に含まれない年を埋め込む
			Calendar nowCal = Calendar.getInstance();
			int year = nowCal.get(Calendar.YEAR);
			Calendar syslogCal = Calendar.getInstance();
			syslogCal.setTime(date);
			if (syslogCal.get(Calendar.MONTH) == Calendar.DECEMBER && nowCal.get(Calendar.MONTH) == Calendar.JANUARY) {
				syslogCal.set(Calendar.YEAR, year - 1);
			} else {
				syslogCal.set(Calendar.YEAR, year);
			}
			date = syslogCal.getTime();

			// インスタンスの生成
			SyslogMessage instance = new SyslogMessage(getFacility((int)pri), getSeverity((int)pri), date.getTime(), hostname, msg, syslog);
			if (log.isDebugEnabled()) {
				log.debug("parsed syslog : " + instance);
			}

			return instance;
		}

		@Override
		public String toString() {
			return String.format("%s [facility = %s, severity = %s, date = %s, hostname = %s, message = %s]",
					this.getClass().getSimpleName(), facility, severity, new Date(date), hostname, message);
		}

		/**
		 * syslogヘッダの<PRI>から該当するFacility値を返すAPI
		 * @param pri syslogヘッダの<PRI>部の値
		 * @return Facility値
		 */
		public static Facility getFacility(int pri) {
			int facility = (pri & 0xFFFFFFF8) >> 3;

			switch (facility) {
			case 0 :
				return Facility.KERN;
			case 1 :
				return Facility.USER;
			case 2 :
				return Facility.MAIL;
			case 3 :
				return Facility.DAEMON;
			case 4 :
				return Facility.AUTH;
			case 5 :
				return Facility.SYSLOG;
			case 6 :
				return Facility.LPR;
			case 7 :
				return Facility.NEWS;
			case 8 :
				return Facility.UUCP;
			case 9 :
				return Facility.CRON;
			case 10 :
				return Facility.AUTHPRIV;
			case 11 :
				return Facility.FTP;
			case 12 :
				return Facility.NTP;
			case 13 :
				return Facility.SECURITY;
			case 14 :
				return Facility.CONSOLE;
			case 16 :
				return Facility.LOCAL0;
			case 17 :
				return Facility.LOCAL1;
			case 18 :
				return Facility.LOCAL2;
			case 19 :
				return Facility.LOCAL3;
			case 20 :
				return Facility.LOCAL4;
			case 21 :
				return Facility.LOCAL5;
			case 22 :
				return Facility.LOCAL6;
			case 23 :
				return Facility.LOCAL7;
			}
			return null;
		}

		/**
		 * syslogヘッダの<PRI>から該当するSeverity値を返すAPI
		 * @param pri syslogヘッダの<PRI>部の値
		 * @return Severity値
		 */
		public static Severity getSeverity(int pri) {
			int severity = (pri & 0x00000007);

			switch (severity) {
			case 0 :
				return Severity.EMERG;
			case 1 :
				return Severity.ALERT;
			case 2 :
				return Severity.CRIT;
			case 3 :
				return Severity.ERR;
			case 4 :
				return Severity.WARNING;
			case 5 :
				return Severity.NOTICE;
			case 6 :
				return Severity.INFO;
			case 7 :
				return Severity.DEBUG;
			}

			return null;
		}

		public static void main(String args[]) throws Exception {

			String syslog1 = "<34>Oct 11 22:14:15 mymachine su: 'su root' failed for lonvick on /dev/pts/8";

			System.out.println(SyslogMessage.parse(syslog1));

		}

}
