/*

Copyright (C) 2010 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.

 */

package com.clustercontrol.selfcheck.monitor;

import javax.persistence.EntityManager;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.commons.util.JpaTransactionManager;
import com.clustercontrol.util.apllog.AplLogger;

/**
 * ジョブセッション数を確認する処理の実装クラス
 */
public class JobRunSessionMonitor extends SelfCheckMonitorBase {

	private static Log m_log = LogFactory.getLog( JobRunSessionMonitor.class );

	public final long threshold;

	public final String monitorId = "SYS_JOB_SESSION";
	public final String subKey = "";
	public final String application = "SELFCHECK (Running Job Session)";

	public JobRunSessionMonitor(long threshold) {
		this.threshold = threshold;
	}

	/**
	 * セルフチェック処理名
	 */
	@Override
	public String toString() {
		return "monitoring running job session (thresdhold = " + threshold + ")";
	}

	/**
	 * 監視項目ID
	 */
	@Override
	public String getMonitorId() {
		return monitorId;
	}

	/**
	 * 監視項目subKey
	 */
	@Override
	public String getSubKey() {
		return subKey;
	}

	/**
	 * ジョブセッション数の確認
	 * @return 通知情報（アプリケーション名は未格納）
	 */
	@Override
	public void execute() {
		/** ローカル変数 */
		JpaTransactionManager tm = null;
		long count = -1;
		boolean warn = true;

		/** メイン処理 */
		if (m_log.isDebugEnabled())
			m_log.debug("monitoring job session running. (threshold = " + threshold + ")");

		try {
			// データソースのオブジェクトをJNDI経由で取得し、取得したコネクションが正しく動作するかを確認する
			tm = new JpaTransactionManager();
			tm.begin();

			// 判定対象値を取得する
			count = getJobRunSessionCount();

			if (count == -1) {
				m_log.info("skipped monitoring job session running. (threshold=" + threshold);
				return;
			} else if (count <= threshold) {
				m_log.debug("job session running size is low. (count = " + count + ", threshold = " + threshold + ")");
				warn = false;
			}
		} catch (Exception e) {
			m_log.warn("monitoring job run session failure. (threshold=" + threshold, e);
		} finally {
			if (tm != null) {
				tm.close();
			}
		}

		if (warn) {
			m_log.info("job session running size is too high. (count= "  + count + ", threshold = " + threshold + ")");
		}
		if (!isNotify(warn)) {
			return;
		}
		String[] msgAttr1 = { new Long(count).toString(), new Long(threshold).toString() };
		AplLogger aplLogger = new AplLogger(PLUGIN_ID, APL_ID);
		aplLogger.put(MESSAGE_ID, "007", msgAttr1,
				"job run session count is too large (" +
						count +
						" > threshold " +
						threshold +
				").");

		return;
	}


	/**
	 * 実行中のジョブセッションレコード数を返すメソッド
	 * @param conn 問い合わせに利用するコネクション
	 * @return レコード数
	 */
	private static long getJobRunSessionCount() {
		// ローカル変数
		JpaTransactionManager tm = null;
		EntityManager em =null;

		String query = "SELECT count(*) " +
				"FROM cc_job_session s, cc_job_session_job j " +
				"WHERE s.session_id = j.session_id AND s.jobunit_id = j.jobunit_id " +
				"AND s.job_id = j.job_id AND status NOT IN (300,301)";

		long count = -1;

		// メイン処理
		try {
			tm = new JpaTransactionManager();
			tm.begin();
			em = tm.getEntityManager();

			Long row = (Long)em.createNativeQuery(query).getSingleResult();
			if (row != null) {
				count = row;
			}

			tm.commit();
		} catch (Exception e) {
			m_log.warn("database query execution failure. (" + query + ")", e);
		} finally {
			if ( tm != null) {
				tm.close();
			}
		}

		return count;
	}

}
