/*

Copyright (C) 2011 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.

 */

package com.clustercontrol.custom.bean;

import java.util.List;

import javax.xml.bind.annotation.XmlType;

import com.clustercontrol.fault.CustomInvalid;
import com.clustercontrol.calendar.bean.CalendarInfo;

/**
 * 情報収集に利用するコマンド実行情報の格納クラス<BR />
 * エージェントのJava VM上で、監視設定1つに対して1つのWSインスタンスが存在する。<BR />
 * 
 * @since 4.0
 * @author Tomoya Takahata
 */
@XmlType(namespace = "http://monitor.ws.clustercontrol.com")
public class CommandExecuteDTO {

	// コマンド実行に起因する監視項目ID
	private String monitorId;

	// コマンドの実効ユーザ
	private Integer specifyUser;
	private String effectiveUser;
	// コマンド文字列(変数込み)
	private String command;
	private Integer timeout = new Integer(0);

	private Integer interval = new Integer(0);
	private CalendarInfo calendar;

	// 対象となっているノード情報のハッシュ(facilityId -> hash of variables)
	//   ノードごとに実行する場合 : 1セットのみ
	//   特定のノードでまとめて実行する場合 : 監視対象スコープに含まれるノード数分のセット
	private List<CommandVariableDTO> variables;

	/**
	 * コンストラクタ(no-argument for JAXB)
	 */
	public CommandExecuteDTO() {

	}

	/**
	 * コンストラクタ
	 * @throws CustomInvalid メンバ変数に不整合が存在する場合
	 */
	public CommandExecuteDTO(String monitorId, Integer specifyUser, String effectiveUser, String command, Integer timeout,
			Integer interval, CalendarInfo calendar, List<CommandVariableDTO> variables) throws CustomInvalid {
		this.monitorId = monitorId;
		this.specifyUser = specifyUser;
		this.effectiveUser = effectiveUser;
		this.command = command;
		this.timeout = timeout;
		this.interval = interval;
		this.calendar = calendar;
		this.variables = variables;

		validate();
	}

	/**
	 * コマンド実行情報に対応する監視項目IDを付与する。<br/>
	 * @param monitorId 監視項目ID
	 */
	public void setMonitorId(String monitorId) {
		this.monitorId = monitorId;
	}

	/**
	 * コマンド実行情報に対応する監視項目IDを返す。<br/>
	 * @return 監視項目ID
	 */
	public String getMonitorId() {
		return monitorId;
	}

	/**
	 * 実効ユーザの種別を付与する。<br/>
	 * @param specifyUser 実効ユーザ種別
	 */
	public void setSpecifyUser(Integer specifyUser) {
		this.specifyUser = specifyUser;
	}

	/**
	 * 実効ユーザの種別を返す。<br/>
	 * @return 実効ユーザ種別
	 */
	public Integer getSpecifyUser() {
		return specifyUser;
	}

	/**
	 * コマンドの実効ユーザを付与する。<br/>
	 * @param effectiveUser 実効ユーザ
	 */
	public void setEffectiveUser(String effectiveUser) {
		this.effectiveUser = effectiveUser;
	}

	/**
	 * コマンドの実効ユーザを返す。<br/>
	 * @return 実効ユーザ
	 */
	public String getEffectiveUser() {
		return effectiveUser;
	}

	/**
	 * 実行されるコマンド文字列を付与する。<br/>
	 * @param command コマンド文字列
	 */
	public void setCommand(String command) {
		this.command = command;
	}

	/**
	 * 実行されるコマンド文字列を返す。<br/>
	 * @return コマンド文字列
	 */
	public String getCommand() {
		return command;
	}

	/**
	 * コマンド実行のタイムアウト時間を付与する。<br/>
	 * @param timeout タイムアウト時間[msec]
	 */
	public void setTimeout(Integer timeout) {
		this.timeout = timeout;
	}

	/**
	 * コマンド実行のタイムアウト時間を返す。<br/>
	 * @return タイムアウト時間[msec]
	 */
	public Integer getTimeout() {
		return timeout;
	}

	/**
	 * コマンド実行情報に割り当てられたカレンダ情報を付与する。<br/>
	 * @param calendar カレンダ情報
	 */
	public void setCalendar(CalendarInfo calendar) {
		this.calendar = calendar;
	}

	/**
	 * コマンド実行情報に割り当てられたカレンダ情報を返す。<br/>
	 * @return カレンダ情報
	 */
	public CalendarInfo getCalendar() {
		return calendar;
	}

	/**
	 * コマンドの実行間隔を付与する。<br/>
	 * @param interval コマンドの実行間隔[msec]
	 */
	public void setInterval(Integer interval) {
		this.interval = interval;
	}

	/**
	 * コマンドの実行間隔を返す。<br/>
	 * @return コマンド実行間隔[msec]
	 */
	public Integer getInterval() {
		return interval;
	}

	/**
	 * 実行対象となっているノード情報の一覧を付与する。<br/>
	 * このノード情報に基づいて、変数が埋め込まれたコマンドが実行される。<br/>
	 * @param variables 実行対象となっているノード情報の一覧
	 */
	public void setVariables(List<CommandVariableDTO> variables) {
		this.variables = variables;
	}

	/**
	 * 実行対象となっているノード情報の一覧を返す。<br/>
	 * @return 実行対象となっているノード情報の一覧
	 */
	public List<CommandVariableDTO> getVariables() {
		return variables;
	}

	/**
	 * メンバ変数の妥当性を確認する。<br/>
	 * @throws CustomInvalid 妥当でない値が含まれる場合
	 */
	public void validate() throws CustomInvalid {
		// null check
		if (effectiveUser == null) {
			throw new CustomInvalid("effectiveUser must be defined(not null). (" + this + ")");
		}
		if (command == null) {
			throw new CustomInvalid("command must be defined(not null). (" + this + ")");
		}
		if (variables == null) {
			throw new CustomInvalid("variables must be defined(not null). 0 length list is allowed. (" + this + ")");
		}

		// validate consistency
		if (timeout <= 0) {
			throw new CustomInvalid("interval must be greater than 0. (" + this + ")");
		}
		if (timeout > interval) {
			throw new CustomInvalid("timeout must be less than interval. (" + this + ")");
		}
	}

	@Override
	public String toString() {
		String variablesStr = "";

		if (variables != null) {
			variablesStr = "";
			for (CommandVariableDTO variable : variables) {
				variablesStr += variablesStr.length() == 0 ? variable : ", " +variable;
			}
		}

		return this.getClass().getCanonicalName() + " [monitorId = " + monitorId
				+ ", effectiveUser = " + effectiveUser
				+ ", command = " + command
				+ ", timeout = " + timeout
				+ ", interval = " + interval
				+ "]";
	}

}
