package com.clustercontrol.calendar.util;

import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.ConcurrentHashMap;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.bean.ValidConstant;
import com.clustercontrol.calendar.bean.CalendarDetailInfo;
import com.clustercontrol.calendar.bean.CalendarInfo;
import com.clustercontrol.calendar.model.CalDetailInfoEntity;
import com.clustercontrol.calendar.model.CalInfoEntity;
import com.clustercontrol.fault.CalendarNotFound;

public class CalendarCache {

	private static Log m_log = LogFactory.getLog( CalendarCache.class );
	private static Object calendarLock = new Object();
	private static ConcurrentHashMap<String, CalendarInfo> map =
			new ConcurrentHashMap<String,CalendarInfo>();

	/**
	 * キャッシュをリフレッシュする。
	 * カレンダの登録、変更、削除時に呼ぶ。
	 */
	public static void remove(String id) {
		m_log.info("remove() calendar cache is removed");

		synchronized (calendarLock) {
			map.remove(id);
		}
	}

	/**
	 * id一致するCalendarInfoを返す。
	 * 一致しなければ、キャッシュに追加する
	 * @param id
	 * @return
	 * @throws CalendarNotFound
	 */
	public static CalendarInfo getCalendarInfo(String id) throws CalendarNotFound {
		CalendarInfo ret = null;
		if (id == null) {
			return null;
		}
		synchronized (calendarLock) {
			ret = map.get(id);
			if (ret != null) {
				return ret;
			}
			ret = getCalendarInfoDB(id);
			map.put(id, ret);
		}
		return ret;
	}
	/**
	 * IDと一致するカレンダ情報一覧をDBより取得します。
	 * @param id
	 * @return
	 * @throws CalendarNotFound
	 */
	private static CalendarInfo getCalendarInfoDB(String id) throws CalendarNotFound{
		//カレンダ取得
		CalInfoEntity entity = null;

		entity = QueryUtil.getCalInfoPK_NONE(id);

		//カレンダ情報のDTOを生成
		CalendarInfo ret = new CalendarInfo();
		//id
		ret.setId(entity.getCalendarId());
		//名前
		ret.setName(entity.getCalendarName());
		//有効期間(From)
		if (entity.getValidTimeFrom() != null) {
			ret.setValidTimeFrom(entity.getValidTimeFrom().getTime());
		}
		//有効期間(To)
		if (entity.getValidTimeTo() != null) {
			ret.setValidTimeTo(entity.getValidTimeTo().getTime());
		}
		//説明
		ret.setDescription(entity.getDescription());
		//登録者
		ret.setRegUser(entity.getRegUser());
		//登録日時
		if (entity.getRegDate() != null) {
			ret.setRegDate(entity.getRegDate().getTime());
		}
		//更新者
		ret.setUpdateUser(entity.getUpdateUser());
		//更新日時
		if (entity.getUpdateDate() != null) {
			ret.setUpdateDate(entity.getUpdateDate().getTime());
		}
		//カレンダ詳細情報
		ArrayList<CalendarDetailInfo> detailList = getCalDetailList(id);
		ret.getCalendarDetailList().addAll(detailList);

		return ret;
	}
	/**
	 * IDと一致するカレンダ詳細情報一覧をDBより取得します。
	 * @param id
	 * @return カレンダ詳細情報のリスト
	 */
	private static ArrayList<CalendarDetailInfo> getCalDetailList(String id) {
		ArrayList<CalendarDetailInfo> list = new ArrayList<CalendarDetailInfo>();

		//カレンダIDの曜日別情報を取得
		List<CalDetailInfoEntity> ct = QueryUtil.getCalDetailByCalendarId(id);

		for (CalDetailInfoEntity cal : ct) {
			CalendarDetailInfo info = new CalendarDetailInfo();
			//説明
			if(cal.getDescription() != null){
				info.setDescription(cal.getDescription());
			}
			//年
			info.setYear(cal.getYearNo());
			//月
			info.setMonth(cal.getMonthNo());
			//曜日選択
			info.setDayType(cal.getDayType());
			//曜日
			if(cal.getWeekNo() != null){
				info.setDayOfWeek(cal.getWeekNo());
			}
			//第x週
			if(cal.getWeekXth() != null){
				info.setDayOfWeekInMonth(cal.getWeekXth());
			}
			//日
			if(cal.getDayNo() != null){
				info.setDate(cal.getDayNo());
			}
			//カレンダパターン
			if(cal.getCalPatternId() != null){
				info.setCalPatternId(cal.getCalPatternId());
			}
			//上記の日程からx日後
			info.setAfterday(cal.getAfterDay());
			//開始時間
			if(cal.getStartTime() != null){
				info.setTimeFrom(cal.getStartTime().getTime());
			}
			//終了時間
			if(cal.getEndTime() != null){
				info.setTimeTo(cal.getEndTime().getTime());
			}
			//稼動・非稼動
			info.setOperateFlg(ValidConstant.typeToBoolean(cal.getExecuteFlg()));

			list.add(info);
		}

		return list;
	}

}
