/*

Copyright (C) 2006 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.

 */

package com.clustercontrol.accesscontrol.util;

import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.ConcurrentHashMap;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.accesscontrol.bean.PrivilegeConstant.SystemPrivilegeMode;
import com.clustercontrol.accesscontrol.bean.SystemPrivilegeInfo;
import com.clustercontrol.accesscontrol.model.RoleEntity;
import com.clustercontrol.accesscontrol.model.SystemPrivilegeEntity;
import com.clustercontrol.accesscontrol.model.UserEntity;
import com.clustercontrol.commons.util.JpaTransactionManager;
import com.clustercontrol.fault.HinemosUnknown;
import com.clustercontrol.fault.RoleNotFound;

/**
 * ロールIDとユーザIDとの関連をマップで管理するクラス。
 */
public class UserRoleCache {
	private static Log m_log = LogFactory.getLog( UserRoleCache.class );

	// ロールとそれに所属するユーザを管理
	private static ConcurrentHashMap<String, List<String>> m_roleUserMap = new ConcurrentHashMap<String, List<String>>();

	// ロールとそれに割り当てられたシステム権限を管理
	private static ConcurrentHashMap<String, List<SystemPrivilegeInfo>> m_roleSystemPrivilegeMap = new ConcurrentHashMap<String, List<SystemPrivilegeInfo>>();

	// ユーザとユーザが所属するロールを管理
	private static ConcurrentHashMap<String, List<String>> m_userRoleMap = new ConcurrentHashMap<String, List<String>>();

	private static Object userRoleMapLock = new Object();

	static {
		refresh();
	}

	/**
	 * ロールIDリストを返す。
	 * 
	 * @return ロールIDのリスト。エラー時は空のリストを返す。
	 */
	public static List<String> getAllRoleIdList(){
		ArrayList<String> roleIdList = new ArrayList<String>();
		if (m_roleUserMap != null
				&& m_roleUserMap.keySet() != null) {
			roleIdList = new ArrayList<String>(m_roleUserMap.keySet());
		}
		return roleIdList;
	}


	/**
	 * ユーザIDリストを返す。
	 * 
	 * @return ユーザIDのリスト。エラー時は空のリストを返す。
	 */
	public static List<String> getAllUserIdList(){
		ArrayList<String> userIdList = new ArrayList<String>();
		if (m_userRoleMap != null
				&& m_userRoleMap.keySet() != null) {
			userIdList = new ArrayList<String>(m_userRoleMap.keySet());
		}
		return userIdList;
	}

	/**
	 * ロールが関連を持つユーザIDのリストを返す。
	 * 
	 * @param roleId ロールID
	 * @return ユーザIDのリスト。エラー時は空のリストを返す。
	 */
	public static List<String> getUserIdList(String roleId){
		m_log.debug("getUserIdList() : roleId " + roleId);
		try {
			// キャッシュから取得
			List<String> userIdList = m_roleUserMap.get(roleId);
			if(userIdList != null){
				return userIdList;
			}
			RoleEntity roleEntity = QueryUtil.getRolePK(roleId);
			userIdList = new ArrayList<String>();
			if (roleEntity != null && roleEntity.getUserEntities() != null) {
				for(UserEntity userEntity : roleEntity.getUserEntities()){
					userIdList.add(userEntity.getUserId());
				}
			}
			m_roleUserMap.put(roleId, userIdList);
			return userIdList;
		} catch (Exception e) {
			m_log.warn("getUserIdList() : "
					+ e.getClass().getSimpleName() + ", " + e.getMessage(), e);
			return new ArrayList<String>(); // 空のリストを返す
		}
	}

	/**
	 * ユーザが所属するロールIDのリストを返す。
	 * 
	 * @param userId ユーザID
	 * @return ロールIDのリスト。エラー時は空のリストを返す。
	 */
	public static List<String> getRoleIdList(String userId){
		m_log.debug("getRoleIdList() : userId " + userId);
		try {
			// キャッシュから取得
			List<String> roleIdList = m_userRoleMap.get(userId);
			if(roleIdList != null){
				return roleIdList;
			}
			UserEntity userEntity = QueryUtil.getUserPK(userId);
			roleIdList = new ArrayList<String>();
			if (userEntity != null && userEntity.getRoleEntities() != null) {
				for(RoleEntity roleEntity : userEntity.getRoleEntities()){
					roleIdList.add(roleEntity.getRoleId());
				}
			}
			m_roleUserMap.put(userId, roleIdList);
			return roleIdList;
		} catch (Exception e) {
			m_log.warn("getRoleIdList() : "
					+ e.getClass().getSimpleName() + ", " + e.getMessage(), e);
			return new ArrayList<String>(); // 空のリストを返す
		}
	}

	/**
	 * ロールが保持するシステム権限のリストを返す。
	 * 
	 * @param roleId ロールID
	 * @return システム権限のリストを返す。
	 * @throws HinemosUnknown 
	 */
	public static List<SystemPrivilegeInfo> getSystemPrivilegeList(String roleId) throws HinemosUnknown{
		m_log.debug("getSystemPrivilegeList() : roleId " + roleId);
		try {
			// キャッシュから取得
			List<SystemPrivilegeInfo> systemPrivilegeList = m_roleSystemPrivilegeMap.get(roleId);
			if(systemPrivilegeList != null){
				return systemPrivilegeList;
			}
			RoleEntity roleEntity = QueryUtil.getRolePK(roleId);
			systemPrivilegeList = new ArrayList<SystemPrivilegeInfo>();
			if (roleEntity != null && roleEntity.getSystemPrivilegeEntities() != null) {
				for(SystemPrivilegeEntity systemPrivilegeEntity : roleEntity.getSystemPrivilegeEntities()){
					systemPrivilegeList.add(new SystemPrivilegeInfo(
							systemPrivilegeEntity.getId().getSystemFunction(), 
							SystemPrivilegeMode.valueOf(systemPrivilegeEntity.getId().getSystemPrivilege())));
				}
			}
			m_roleSystemPrivilegeMap.put(roleId, systemPrivilegeList);

			return systemPrivilegeList;
		} catch (RoleNotFound e) {
			return new ArrayList<SystemPrivilegeInfo>(); // 空のリストを返す
		} catch (Exception e) {
			m_log.warn("getSystemPrivilegeList() : "
					+ e.getClass().getSimpleName() + ", " + e.getMessage(), e);
			throw new HinemosUnknown(e.getMessage(), e);
		}
	}

	/**
	 * ユーザが指定された権限を保持しているか確認する。
	 * 
	 * @param userId ユーザID
	 * @param systemPrivilegeInfo チェック対象のシステム権限
	 * @return true:権限あり、false:権限なし
	 * @throws HinemosUnknown 
	 */
	public static boolean isSystemPrivilege(String userId, SystemPrivilegeInfo info) throws HinemosUnknown{
		m_log.debug("isSystemPrivilege() : userId " + userId);
		boolean rtn = false;
		try {
			// ユーザに対応したロールを取得
			List<String> roleIdList = getRoleIdList(userId);
			
			for (String roleId : roleIdList) {
				List<SystemPrivilegeInfo> systemPrivilegeList = getSystemPrivilegeList(roleId);
				if (systemPrivilegeList.contains(info)) {
					// 権限あり
					rtn = true;
					break;
				}
			}
			return rtn;
		} catch (HinemosUnknown e) {
			throw e;
		} catch (Exception e) {
			m_log.warn("isSystemPrivilege() : "
					+ e.getClass().getSimpleName() + ", " + e.getMessage(), e);
			throw new HinemosUnknown(e.getMessage(), e);
		}
	}

	/**
	 * 全件リフレッシュする。
	 * 
	 */
	public static void refresh(){
		m_log.debug("refresh()");
		JpaTransactionManager jtm = new JpaTransactionManager();
		if (!jtm.isNestedEm()) {
			m_log.warn("refresh() : transactioin has not been begined.");
			jtm.close();
			return;
		}

		synchronized (userRoleMapLock) {
			// ロールに所属するユーザを取得
			m_roleUserMap = new ConcurrentHashMap<String, List<String>>();
			List<RoleEntity> roleEntities = QueryUtil.getAllRole();
			
			for (RoleEntity roleEntity : roleEntities) {
				List<String> userIdList = new ArrayList<String>();
				if (roleEntity.getUserEntities() != null) {
					for (UserEntity userEntity : roleEntity.getUserEntities()) {
						userIdList.add(userEntity.getUserId());
					}
				}
				m_roleUserMap.put(roleEntity.getRoleId(), userIdList);
			}

			// ロールに割り当てられたシステム権限を取得
			m_roleSystemPrivilegeMap = new ConcurrentHashMap<String, List<SystemPrivilegeInfo>>();
			
			for (RoleEntity roleEntity : roleEntities) {
				List<SystemPrivilegeInfo> systemPrivilegeList = new ArrayList<SystemPrivilegeInfo>();
				if (roleEntity.getSystemPrivilegeEntities() != null) {
					for (SystemPrivilegeEntity systemPrivilegeEntity : roleEntity.getSystemPrivilegeEntities()) {
						systemPrivilegeList.add(new SystemPrivilegeInfo(
								systemPrivilegeEntity.getId().getSystemFunction(), 
								SystemPrivilegeMode.valueOf(systemPrivilegeEntity.getId().getSystemPrivilege())));
					}
				}
				m_roleSystemPrivilegeMap.put(roleEntity.getRoleId(), systemPrivilegeList);
			}

			// ユーザが所属するロールを取得
			m_userRoleMap = new ConcurrentHashMap<String, List<String>>();
			List<UserEntity> userEntities = QueryUtil.getAllUser();
			
			for (UserEntity userEntity : userEntities) {
				List<String> roleIdList = new ArrayList<String>();
				if (userEntity.getRoleEntities() != null) {
					for (RoleEntity roleEntity : userEntity.getRoleEntities()) {
						roleIdList.add(roleEntity.getRoleId());
					}
				}
				m_userRoleMap.put(userEntity.getUserId(), roleIdList);
			}
		}
	}
}
