/*

Copyright (C) 2006 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.

 */

package com.clustercontrol.jobmanagement.composite;

import java.util.ArrayList;
import java.util.List;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eclipse.jface.dialogs.IDialogConstants;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.viewers.DoubleClickEvent;
import org.eclipse.jface.viewers.IDoubleClickListener;
import org.eclipse.jface.viewers.ISelectionChangedListener;
import org.eclipse.jface.viewers.SelectionChangedEvent;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.swt.SWT;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Tree;
import org.eclipse.ui.PlatformUI;

import com.clustercontrol.jobmanagement.bean.JobConstant;
import com.clustercontrol.jobmanagement.dialog.JobDialog;
import com.clustercontrol.jobmanagement.util.JobEditStateUtil;
import com.clustercontrol.jobmanagement.util.JobEndpointWrapper;
import com.clustercontrol.jobmanagement.util.JobUtil;
import com.clustercontrol.jobmanagement.view.JobListView;
import com.clustercontrol.jobmanagement.viewer.JobTreeContentProvider;
import com.clustercontrol.jobmanagement.viewer.JobTreeLabelProvider;
import com.clustercontrol.jobmanagement.viewer.JobTreeViewer;
import com.clustercontrol.util.Messages;
import com.clustercontrol.ws.jobmanagement.InvalidRole_Exception;
import com.clustercontrol.ws.jobmanagement.JobInfo;
import com.clustercontrol.ws.jobmanagement.JobTreeItem;

/**
 * ジョブツリー用のコンポジットクラスです。
 * 
 * @version 1.0.0
 * @since 1.0.0
 */
public class JobTreeComposite extends Composite {

	// ログ
	private static Log m_log = LogFactory.getLog( JobTreeComposite.class );

	/** ツリービューア */
	private JobTreeViewer m_viewer = null;
	/** 選択ジョブツリーアイテム */
	private JobTreeItem m_selectItem = null;
	/** ツリーのみ */
	private boolean m_treeOnly = false;
	private String m_jobId = null;
	private JobListView m_view = null;
	private boolean m_useForView = false; //ジョブ[一覧]ビュー、ジョブツリービュー(ジョブマップオプション)で使うか
	private static List<JobTreeViewer>m_treeViewerList = new ArrayList<JobTreeViewer>(); // JobTreeViewerをまとめて更新するためのリスト

	/**
	 * 表示ツリーの形式
	 * 値として、JobConstantクラスで定義したものが入る
	 * @see com.clustercontrol.jobmanagement.bean.JobConstant
	 *  -1 : 未選択
	 *  TYPE_REFERJOB以外: 選択したユニット、ネットの子のみ表示する
	 *  TYPE_REFERJOB : 選択したユニット、ネットの所属するジョブユニット以下すべて表示する
	 */
	private int mode = -1;

	/** オーナーロールID */
	private String ownerRoleId = null;

	/**
	 * コンストラクタ
	 * 
	 * @param parent 親コンポジット
	 * @param style スタイル
	 * 
	 * @see org.eclipse.swt.SWT
	 * @see org.eclipse.swt.widgets.Composite#Composite(Composite parent, int style)
	 * @see #initialize()
	 */
	public JobTreeComposite(JobListView view , Composite parent, int style, String ownerRoleId) {
		super(parent, style);

		m_treeOnly = false;
		m_view = view;
		this.ownerRoleId = ownerRoleId;
		m_useForView = true;
		initialize();
	}

	/**
	 * コンストラクタ
	 * 
	 * @param parent 親コンポジット
	 * @param style スタイル
	 * @param treeOnly true：ツリーのみ、false：ジョブ情報を含む
	 * 
	 * @see org.eclipse.swt.SWT
	 * @see org.eclipse.swt.widgets.Composite#Composite(Composite parent, int style)
	 * @see #initialize()
	 */
	public JobTreeComposite(Composite parent, int style,
			String ownerRoleId,
			boolean treeOnly,
			boolean useForView) {
		super(parent, style);

		m_treeOnly = treeOnly;
		this.ownerRoleId = ownerRoleId;
		m_useForView = useForView;
		initialize();
	}

	/**
	 * コンストラクタ
	 * 
	 * @param parent 親コンポジット
	 * @param style スタイル
	 * @param parentJobId 親ジョブID
	 * @param jobId ジョブID
	 * 
	 * @see org.eclipse.swt.SWT
	 * @see org.eclipse.swt.widgets.Composite#Composite(Composite parent, int style)
	 * @see #initialize()
	 */
	public JobTreeComposite(Composite parent, int style,
			String ownerRoleId,
			JobTreeItem selectItem) {
		super(parent, style);

		m_treeOnly = true;
		m_selectItem = selectItem;
		m_jobId = selectItem.getData().getId();
		this.ownerRoleId = ownerRoleId;
		m_useForView = false;
		initialize();
	}
	public JobTreeComposite(Composite parent, int style,
			String ownerRoleId,
			JobTreeItem selectItem,
			int mode) {
		super(parent, style);

		m_treeOnly = true;
		m_selectItem = selectItem;
		m_jobId = selectItem.getData().getId();
		this.mode = mode;
		this.ownerRoleId = ownerRoleId;
		m_useForView = false;
		initialize();
	}

	/**
	 * コンポジットを構築します。
	 */
	private void initialize() {
		GridLayout layout = new GridLayout(1, true);
		this.setLayout(layout);
		layout.marginHeight = 0;
		layout.marginWidth = 0;

		Tree tree = new Tree(this, SWT.SINGLE | SWT.BORDER);
		GridData gridData = new GridData();
		gridData.horizontalAlignment = GridData.FILL;
		gridData.verticalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		gridData.grabExcessVerticalSpace = true;
		tree.setLayoutData(gridData);

		m_viewer = new JobTreeViewer(tree);
		m_viewer.setContentProvider(new JobTreeContentProvider());
		m_viewer.setLabelProvider(new JobTreeLabelProvider(m_useForView));

		// 選択アイテム取得イベント定義
		m_viewer.addSelectionChangedListener(new ISelectionChangedListener() {
			@Override
			public void selectionChanged(SelectionChangedEvent event) {
				StructuredSelection selection = (StructuredSelection) event.getSelection();
				m_selectItem = (JobTreeItem) selection.getFirstElement();
			}
		});

		// ダブルクリックしたらジョブを開く
		if (m_useForView) {
			m_viewer.addDoubleClickListener(
					new IDoubleClickListener() {
						@Override
						public void doubleClick(DoubleClickEvent event) {
							StructuredSelection selection = (StructuredSelection) event.getSelection();
							JobTreeItem item = (JobTreeItem) selection.getFirstElement();
							int type = item.getData().getType();
							m_log.info("double click. type=" + type);
							if (type != JobConstant.TYPE_REFERJOB &&
									type != JobConstant.TYPE_FILEJOB &&
									type != JobConstant.TYPE_JOB &&
									type != JobConstant.TYPE_JOBUNIT &&
									type != JobConstant.TYPE_JOBNET) {
								return;
							}
							boolean readOnly = !JobEditStateUtil.isLockedJobunitId(item.getData().getJobunitId());
							JobDialog dialog = new JobDialog(
									PlatformUI.getWorkbench().getActiveWorkbenchWindow().getShell(),
									readOnly);
							dialog.setJobTreeItem(item);
							//ダイアログ表示
							if (dialog.open() == IDialogConstants.OK_ID) {
								if (!readOnly) {
									// 編集できるときにOKが押されたときは、設定が変更されたと判定する
									JobEditStateUtil.addEditedJobunit(item);
									if (item.getData().getType() == JobConstant.TYPE_JOBUNIT) {
										JobUtil.setJobunitIdAll(item, item.getData().getJobunitId());
									}
								}
								m_viewer.sort(item.getParent());
								m_viewer.refresh(item.getParent());
								m_viewer.refresh(item);
								m_viewer.setSelection(new StructuredSelection(item), true);
							}
						}
					});
		}

		updateTree(m_useForView);
	}

	/**
	 * このコンポジットが利用するツリービューアを返します。
	 * 
	 * @return ツリービューア
	 */
	public JobTreeViewer getTreeViewer() {
		return m_viewer;
	}

	/**
	 * このコンポジットが利用するツリーを返します。
	 * 
	 * @return ツリー
	 */
	public Tree getTree() {
		return m_viewer.getTree();
	}

	/**
	 * ツリービューアーを更新します。<BR>
	 * ジョブツリー情報を取得し、ツリービューアーにセットします。
	 * <p>
	 * <ol>
	 * <li>ジョブツリー情報を取得します。</li>
	 * <li>ツリービューアーにジョブツリー情報をセットします。</li>
	 * </ol>
	 * 
	 * @see com.clustercontrol.jobmanagement.action.GetJobTree#getJobTree(boolean)
	 */
	@Override
	public void update() {
		updateTree(false);
	}
	
	/**
	 * ツリービューアーを更新します。<BR>
	 * ジョブツリー情報を取得し、ツリービューアーにセットします。
	 * <p>
	 * <ol>
	 * <li>ジョブツリー情報を取得します。</li>
	 * <li>ツリービューアーにジョブツリー情報をセットします。</li>
	 * </ol>
	 * 
	 * @param useChache キャッシュがあればそれを利用する(ジョブ[一覧]ビュー以外では考慮されない)
	 */
	public void updateTree(boolean useChache) {
		// 非Javadoc 継承の関係でupdate(boolean)はoverrideの警告が出るのでメソッド名を変更した
		JobTreeItem jobTree = null;

		//　ジョブ一覧情報取得
		if (m_useForView) {
			if (useChache) {
				jobTree = JobEditStateUtil.getJobTreeItem();
			}
			if (jobTree == null) {
				jobTree = JobEditStateUtil.updateJobTree(ownerRoleId, m_treeOnly);
			}
		} else if (m_jobId == null) {
			try {
				jobTree = JobEndpointWrapper.getJobTree(ownerRoleId, m_treeOnly);
			} catch (InvalidRole_Exception e) {
				// アクセス権なしの場合、エラーダイアログを表示する
				MessageDialog.openInformation(
						null,
						Messages.getString("message"),
						Messages.getString("message.accesscontrol.16"));
			} catch (Exception e) {
				m_log.warn("update() getJobTree, " + e.getMessage(), e);
				MessageDialog.openError(
						null,
						Messages.getString("failed"),
						Messages.getString("message.hinemos.failure.unexpected") + ", " + e.getMessage());
			}
		} else if (mode == JobConstant.TYPE_REFERJOB) {
			jobTree = getJobTreeOneUnit(m_selectItem);
		} else {
			jobTree = getJobTreeOneLevel(m_selectItem);
		}
		m_selectItem = null;

		m_viewer.setInput(jobTree);
		if (m_useForView) {
			// ビューを更新した場合は、開いているジョブツリーの情報を全て更新する
			for (JobTreeViewer viewer : m_treeViewerList) {
				if (viewer == m_viewer) {
					continue;
				}
				viewer.setInput(jobTree);
			}
		}

		//ジョブユニットのレベルまで展開
		m_viewer.expandToLevel(2);
	}
	
	/**
	 * ツリービューアーを更新します。<BR>
	 * ジョブツリー情報を取得し、ツリービューアーにセットします。
	 * その際に、引数のjobIdに該当するアイテムにフォーカスを当てます
	 * <p>
	 * <ol>
	 * <li>ジョブツリー情報を取得します。</li>
	 * <li>ツリービューアーにジョブツリー情報をセットします。</li>
	 * <li>その際に、引数のjobIdに該当するアイテムにフォーカスを当てます。</li>
	 * </ol>
	 */
	public void update(String jobId){
		JobTreeItem jobTree = null;
		//ジョブ一覧情報取得
		if (m_useForView) {
			jobTree = JobEditStateUtil.updateJobTree(ownerRoleId, m_treeOnly);
		} else if (m_jobId == null) {
			try {
				jobTree = JobEndpointWrapper.getJobTree(ownerRoleId, m_treeOnly);
			} catch (InvalidRole_Exception e) {
				// アクセス権なしの場合、エラーダイアログを表示する
				MessageDialog.openInformation(
						null,
						Messages.getString("message"),
						Messages.getString("message.accesscontrol.16"));
			} catch (Exception e) {
				m_log.warn("update() getJobTree, " + e.getMessage(), e);
				MessageDialog.openError(
						null,
						Messages.getString("failed"),
						Messages.getString("message.hinemos.failure.unexpected") + ", " + e.getMessage());
			}
		} else {
			jobTree = getJobTreeOneLevel(m_selectItem);
		}
		m_selectItem = null;

		m_viewer.setInput(jobTree);
		//引数のjobIdに該当するアイテムにフォーカスを当てる
		List<JobTreeItem> treeList = jobTree.getChildren();
		setFocus(treeList,jobId);
	}
	
	/**
	 * 引数のjobIdと一致するJobTreeItemにフォーカスを当てる
	 * itemには子がいるので、再帰呼び出しでループ
	 * @param treeList
	 * @param jobId
	 */
	private void setFocus(List<JobTreeItem> treeList,String jobId){
		for(JobTreeItem item : treeList){
			//該当アイテムが存在したらフォーカスを当て、ループを抜ける
			if(jobId.equals(item.getData().getId())){
				m_log.trace("item : " + item.getData().getId());
				m_viewer.setSelection(new StructuredSelection(item), true);
				break;
			}
			//子が存在する場合、再帰呼び出し
			if(item.getChildren() != null && item.getChildren().size() > 0){
				setFocus(item.getChildren(),jobId);
			}
		}
	}
	/**
	 * 選択ジョブツリーアイテムを返します。
	 * 
	 * @return ジョブツリーアイテム
	 */
	public JobTreeItem getSelectItem() {
		return m_selectItem;
	}

	/**
	 * 選択ジョブツリーアイテムを設定
	 * 
	 * @param item ジョブツリーアイテム
	 */
	public void setSelectItem(JobTreeItem item) {
		m_selectItem = item;
	}
	/**
	 * ジョブ[一覧]ビューのジョブツリー情報から、<BR>
	 * 引数で渡された親ジョブIDの直下のジョブツリーアイテムを取得する。<BR><BR>
	 * 取得したジョブツリーアイテムから、<BR>
	 * 引数で渡されたジョブIDと一致するジョブツリーアイテムを除いたジョブツリーアイテムを返す。
	 * 
	 * @return ジョブツリー情報{@link com.clustercontrol.jobmanagement.bean.JobTreeItem}の階層オブジェクト
	 */
	public JobTreeItem getJobTreeOneLevel(JobTreeItem self) {
		JobTreeItem parentOrg = self.getParent();

		// selfの親
		JobTreeItem parentClone = new JobTreeItem();
		parentClone.setData(copyJobInfo(parentOrg.getData()));

		// selfの兄弟
		String jobId = self.getData().getId();
		for (JobTreeItem brotherOrg : self.getParent().getChildren()) {
			if (!jobId.equals(brotherOrg.getData().getId())) {
				JobTreeItem brotherClone = new JobTreeItem();
				brotherClone.setParent(parentClone);
				parentClone.getChildren().add(brotherClone);
				brotherClone.setData(copyJobInfo(brotherOrg.getData()));
			}
		}
		return parentClone;
	}
	/**
	 * 自分が所属するジョブユニット以下のJobTreeItemを取得する。
	 * @version 4.1.0
	 * @param self
	 * @return
	 */
	public JobTreeItem getJobTreeOneUnit(JobTreeItem self) {
		// selfの親
		JobTreeItem parentOrg = self.getParent();
		// selfの親のクローン
		JobTreeItem ret = new JobTreeItem();
		//selfの所属するジョブユニットのジョブID
		String jobUnitId = self.getData().getJobunitId();
		//selfの階層から、root階層まで、探索する
		while (!(parentOrg.getData().getId()).equals("")) {
			//
			if (parentOrg.getData().getId().equals(jobUnitId)) {
				ret.setData(copyJobInfo(parentOrg.getData()));
				break;
			}
			//現在のJobTreeItemの親を取得する
			parentOrg = parentOrg.getParent();
		}
		//選択したJobTreeItemのジョブID
		String jobId = self.getData().getId();
		//deep copy
		ret = cloneChildren(jobId,ret,parentOrg.getChildren());
		return ret;
	}
	/**
	 * 子が存在した場合、下位のJobTreeItemをdeep copyする
	 * @version 4.1.0
	 * @param id 選択したJobTreeItemのジョブID
	 * @param parent 参照している階層の親
	 * @param itemList
	 * @return
	 */
	private JobTreeItem cloneChildren(String id, JobTreeItem parent, List<JobTreeItem> itemList){
		for (JobTreeItem childrenOrg : itemList) {
			if (!id.equals(childrenOrg.getData().getId())) {
				JobTreeItem childrenClone = new JobTreeItem();

				childrenClone.setData(copyJobInfo(childrenOrg.getData()));
				if(!childrenOrg.getChildren().isEmpty()){
					childrenClone = cloneChildren(id,childrenClone,childrenOrg.getChildren());
				}
				childrenClone.setParent(parent);
				parent.getChildren().add(childrenClone);
			}
		}
		return parent;
	}

	/**
	 * 引数で渡されたジョブ情報のコピーインスタンスを作成する。
	 * 
	 * @param orgInfo コピー元ジョブ情報
	 * @return ジョブ情報
	 */
	private JobInfo copyJobInfo(JobInfo orgInfo) {

		JobInfo info = new JobInfo();
		info.setJobunitId(orgInfo.getJobunitId());
		info.setId(orgInfo.getId());
		info.setName(orgInfo.getName());
		info.setType(orgInfo.getType());

		return info;
	}
	
	/**
	 * 表示しているすべてのジョブツリーの表示をリフレッシュする
	 */
	public void refresh() {
		for (JobTreeViewer viewer : m_treeViewerList) {
			m_log.debug("refresh : " + viewer);
			viewer.refresh();
		}
	}
	
	/**
	 * 表示しているすべてのジョブツリーの表示をリフレッシュする
	 * @param element
	 */
	public void refresh(Object element) {
		for (JobTreeViewer viewer : m_treeViewerList) {
			m_log.debug("refresh : " + viewer);
			viewer.refresh(element);
		}
	}
	
	/**
	 * 現在クライアントが表示しているジョブツリーのリストにこのインスタンスのツリーを追加する<BR>
	 * ビューを開く際に呼ぶこと
	 */
	public void addToTreeViewerList() {
		m_log.debug("add treeViewerList: " + m_viewer);
		m_log.debug(m_treeViewerList.size());
		m_treeViewerList.add(this.m_viewer);
		m_log.debug(m_treeViewerList.size());
	}
	
	/**
	 * 現在クライアントが表示しているジョブツリーのリストからこのインスタンスのツリーを削除する<BR>
	 * ビューを閉じる際に呼ぶこと
	 */
	public void removeFromTreeViewerList() {
		m_log.debug("remove treeViewerList: " + m_viewer);
		m_log.debug(m_treeViewerList.size());
		m_treeViewerList.remove(this.m_viewer);
		m_log.debug(m_treeViewerList.size());
		super.dispose();
	}
	
}