/*

Copyright (C) 2012 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.

 */

package com.clustercontrol.commons.util;

import java.util.Collections;
import java.util.List;

import javax.persistence.Cache;
import javax.persistence.EntityManager;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eclipse.persistence.internal.jpa.CacheImpl;

import com.clustercontrol.fault.HinemosUnknown;
import com.clustercontrol.hinemosagent.util.AgentConnectUtil;
import com.clustercontrol.jobmanagement.util.JobMultiplicityCache;
import com.clustercontrol.notify.model.MonitorStatusEntity;
import com.clustercontrol.notify.model.NotifyHistoryEntity;
import com.clustercontrol.plugin.impl.AsyncWorkerPlugin;
import com.clustercontrol.plugin.impl.SchedulerInfo;
import com.clustercontrol.plugin.impl.SchedulerPlugin;
import com.clustercontrol.plugin.impl.SharedTablePlugin;
import com.clustercontrol.plugin.impl.SnmpTrapPlugin;
import com.clustercontrol.plugin.impl.SystemLogPlugin;
import com.clustercontrol.plugin.impl.SchedulerPlugin.SchedulerType;
import com.clustercontrol.poller.PollerManager;
import com.clustercontrol.repository.util.FacilityTreeCache;
import com.clustercontrol.selfcheck.SelfCheckConfig;

public class ManagerCliUtil {

	private static final Log log = LogFactory.getLog(ManagerCliUtil.class);

	/**
	 * エージェントの管理情報を文字列で返す。<br/>
	 * @return エージェントの管理情報文字列
	 */
	public static String getValidAgentStr() {
		String str = "";
		String lineSeparator = System.getProperty("line.separator");

		List<String> validAgent = AgentConnectUtil.getValidAgent();
		Collections.sort(validAgent);

		for (String facilityId : validAgent) {
			String agentString = AgentConnectUtil.getAgentString(facilityId);
			if (agentString == null) {
				continue;
			}
			str += facilityId + ", " + agentString + lineSeparator;
		}
		return str;
	}

	/**
	 * Pollerの管理情報を文字列で返す。<br/>
	 * @return Pollerの管理情報文字列
	 */
	public static String getPollerInfoStr() {
		return PollerManager.getInstnace().getPollerDebugInfo();
	}

	/**
	 * SharedTableの管理情報を文字列で返す。<br/>
	 * @return SharedTableの管理情報文字列
	 */
	public static String getSharedTableInfoStr() {
		return SharedTablePlugin.getSharedTable().getTableListDebugInfo();
	}

	/**
	 * 管理しているスケジューラ情報を文字列で返す。
	 * @return スケジューラ情報文字列
	 * @throws HinemosUnknown
	 */
	public static String getSchedulerInfoStr() throws HinemosUnknown {
		String str = "";
		String lineSeparator = System.getProperty("line.separator");

		str += "----- " + SchedulerType.DBMS + " -----" + lineSeparator;
		List<SchedulerInfo> dbms = SchedulerPlugin.getSchedulerList(SchedulerType.DBMS);
		for (SchedulerInfo trigger : dbms) {
			str += lineSeparator;

			String start = String.format("%1$tY-%1$tm-%1$td %1$tH:%1$tM:%1$tS", trigger.startTime);
			String prev = String.format("%1$tY-%1$tm-%1$td %1$tH:%1$tM:%1$tS", trigger.previousFireTime);
			String next = String.format("%1$tY-%1$tm-%1$td %1$tH:%1$tM:%1$tS", trigger.nextFireTime);

			str += String.format("%s (in %s) :", trigger.name, trigger.group) + lineSeparator;
			str += String.format("   start fire time - %s", start) + lineSeparator;
			str += String.format("   last fire time  - %s", prev) + lineSeparator;
			str += String.format("   next fire time  - %s", next) + lineSeparator;
			str += String.format("   is paused       - %s", trigger.isPaused) + lineSeparator;
		}

		str += lineSeparator;

		str += "----- " + SchedulerType.RAM + " -----" + lineSeparator;
		List<SchedulerInfo> ram = SchedulerPlugin.getSchedulerList(SchedulerType.RAM);
		for (SchedulerInfo trigger : ram) {
			str += lineSeparator;

			String start = String.format("%1$tY-%1$tm-%1$td %1$tH:%1$tM:%1$tS", trigger.startTime);
			String prev = String.format("%1$tY-%1$tm-%1$td %1$tH:%1$tM:%1$tS", trigger.previousFireTime);
			String next = String.format("%1$tY-%1$tm-%1$td %1$tH:%1$tM:%1$tS", trigger.nextFireTime);

			str += String.format("%s (in %s) :", trigger.name, trigger.group) + lineSeparator;
			str += String.format("   start fire time - %s", start) + lineSeparator;
			str += String.format("   last fire time  - %s", prev) + lineSeparator;
			str += String.format("   next fire time  - %s", next) + lineSeparator;
			str += String.format("   is paused       - %s", trigger.isPaused) + lineSeparator;
		}

		return str;
	}

	/**
	 * セルフチェック機能が最後に動作した日時文字列を返す。<br/>
	 * @return セルフチェック機能が最後に動作した日時文字列を
	 */
	public static String getSelfCheckLastFireTimeStr() {
		return String.format("%1$tY-%1$tm-%1$td %1$tH:%1$tM:%1$tS", SelfCheckConfig.lastMonitorDate);
	}

	/**
	 * syslogの統計情報を返す。<br/>
	 * @return syslogの統計情報
	 */
	public static String getSyslogStatistics() {
		String str = "";
		str += "[Syslog Statistics]" + System.getProperty("line.separator");
		str += "received : " + SystemLogPlugin.getReceivedCount() + System.getProperty("line.separator");
		str += "queued : " + SystemLogPlugin.getQueuedCount() + System.getProperty("line.separator");
		str += "discarded : " + SystemLogPlugin.getDiscardedCount() + System.getProperty("line.separator");
		str += "notified : " + SystemLogPlugin.getNotifiedCount() + System.getProperty("line.separator");
		return str;
	}

	/**
	 * snmptrapの統計情報を返す。<br/>
	 * @return snmptrapの統計情報
	 */
	public static String getSnmpTrapStatistics() {
		String str = "";
		str += "[SnmpTrap Statistics]" + System.getProperty("line.separator");
		str += "received : " + SnmpTrapPlugin.getReceivedCount() + System.getProperty("line.separator");
		str += "queued : " + SnmpTrapPlugin.getQueuedCount() + System.getProperty("line.separator");
		str += "discarded : " + SnmpTrapPlugin.getDiscardedCount() + System.getProperty("line.separator");
		str += "notified : " + SnmpTrapPlugin.getNotifiedCount() + System.getProperty("line.separator");
		return str;
	}

	/**
	 * 非同期タスクの蓄積数を返す。<br/>
	 * @return 非同期タスクの蓄積数
	 * @throws HinemosUnknown
	 */
	public static String getAsyncWorkerStatistics() throws HinemosUnknown {
		String str = "";
		str += "[AsyncWorker Statistics]" + System.getProperty("line.separator");
		for (String worker : AsyncWorkerPlugin.getWorkerList()) {
			str += "queued tasks [" + worker + "] : " + AsyncWorkerPlugin.getTaskCount(worker) + System.getProperty("line.separator");
		}
		return str;
	}

	/**
	 * 通知抑制の履歴情報（最終重要度および通知日時）をリセットする
	 * @return
	 * @throws HinemosUnknown
	 */
	public static String resetNotificationLogger() throws HinemosUnknown {
		log.info("resetting notification counter...");

		JpaTransactionManager tm = null;
		EntityManager em = null;

		try {
			tm = new JpaTransactionManager();
			tm.begin();

			// EJB Container内の抑制情報を初期化
			em = tm.getEntityManager();
			em.createNamedQuery("MonitorStatusEntity.deleteAll", MonitorStatusEntity.class).executeUpdate();
			em.createNamedQuery("NotifyHistoryEntity.deleteAll", NotifyHistoryEntity.class).executeUpdate();

			tm.commit();
		} catch (Exception e) {
			log.warn("notify counter reset failure...", e);
			tm.rollback();
			throw new HinemosUnknown("notify counter reset failure...", e);
		} finally {
			if (tm != null) {
				tm.close();
			}
		}

		log.info("notify counter resetted successfully.");
		return null;
	}

	/**
	 * JPAのキャッシュを全て出力する。
	 * 現状、hinemos_manager.logに出力されてしまいます。
	 */
	public static void printJpaCacheAll() {
		JpaTransactionManager tm = null;
		EntityManager em = null;
		try {
			tm = new JpaTransactionManager();
			tm.begin();
			em = tm.getEntityManager();
			Cache cache = em.getEntityManagerFactory().getCache();
			CacheImpl jpaCache = (CacheImpl) cache;
			jpaCache.print();
			tm.commit();
		} catch (Exception e) {
			log.warn("printJpaCache failure...", e);
		} finally {
			if (tm != null) {
				tm.close();
			}
		}
	}

	/**
	 * ジョブ多重度のキューの状態を出力する。
	 * @return
	 */
	public static String getJobQueueStr() {
		return JobMultiplicityCache.getJobQueueStr();
	}
	
	/**
	 * リポジトリのキャッシュ情報を、hinemos_manager.logに出力する
	 */
	public static void printFacilityTreeCacheAll() {
		JpaTransactionManager jtm = null;
		try {
			jtm = new JpaTransactionManager();
			jtm.begin();
			
			/* メイン処理 */
			FacilityTreeCache.printCache();
			
			jtm.commit();
		} catch (Exception e) {
			log.warn("printFacilityTreeCacheAll failure...", e);
		} finally {
			if (jtm != null) {
				jtm.close();
			}
		}
	}
	
	/**
	 * リポジトリのキャッシュ情報をリフレッシュする
	 */
	public static void refreshFacilityTreeCache() {
		JpaTransactionManager jtm = null;
		try {
			jtm = new JpaTransactionManager();
			jtm.begin();
			
			/* メイン処理 */
			FacilityTreeCache.refresh_ALL();
			
			jtm.commit();
		} catch (Exception e) {
			log.warn("refreshFacilityTreeCache failure...", e);
		} finally {
			if (jtm != null) {
				jtm.close();
			}
		}
	}
}
