/*
Copyright (C) 2010 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.
 */
package com.clustercontrol.ws.endpoint;

import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.Locale;

import javax.annotation.Resource;
import javax.ejb.CreateException;
import javax.naming.NamingException;
import javax.xml.ws.WebServiceContext;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.jnp.interfaces.NamingContext;

import com.clustercontrol.fault.CalendarNotFound;
import com.clustercontrol.fault.FacilityNotFound;
import com.clustercontrol.fault.HinemosUnknown;
import com.clustercontrol.fault.InvalidRole;
import com.clustercontrol.fault.InvalidSetting;
import com.clustercontrol.fault.InvalidUserPass;
import com.clustercontrol.fault.JobInfoNotFound;
import com.clustercontrol.fault.JobInvalid;
import com.clustercontrol.fault.JobMasterNotFound;
import com.clustercontrol.fault.NotifyNotFound;
import com.clustercontrol.fault.UserNotFound;
import com.clustercontrol.accesscontrol.bean.RoleConstant;
import com.clustercontrol.bean.HinemosModuleConstant;
import com.clustercontrol.jobmanagement.bean.JobForwardFile;
import com.clustercontrol.jobmanagement.bean.JobHistoryFilter;
import com.clustercontrol.jobmanagement.bean.JobHistoryList;
import com.clustercontrol.jobmanagement.bean.JobInfo;
import com.clustercontrol.jobmanagement.bean.JobNodeDetail;
import com.clustercontrol.jobmanagement.bean.JobSchedule;
import com.clustercontrol.jobmanagement.bean.JobTreeItem;
import com.clustercontrol.jobmanagement.bean.JobTriggerInfo;
import com.clustercontrol.jobmanagement.bean.JobTriggerTypeConstant;
import com.clustercontrol.jobmanagement.bean.NodeOperationInfo;
import com.clustercontrol.jobmanagement.bean.OperationConstant;
import com.clustercontrol.jobmanagement.ejb.session.JobControllerLocal;
import com.clustercontrol.jobmanagement.ejb.session.JobControllerLocalHome;
import com.clustercontrol.notify.bean.OutputBasicInfo;

/**
 * ジョブ操作用のWebAPIエンドポイント
 */
@javax.jws.WebService(targetNamespace = "http://jobmanagement.ws.clustercontrol.com")
public class JobEndpoint {
	@Resource
	WebServiceContext wsctx;

	private static Log m_log = LogFactory.getLog( JobEndpoint.class );
	private static Log m_opelog = LogFactory.getLog("HinemosOperation");

	/**
	 * echo(WebサービスAPI疎通用)
	 * 
	 * 権限必要なし（ユーザ名チェックのみ実施）
	 * 
	 * @param str
	 * @return
	 * @throws InvalidUserPass
	 * @throws InvalidRole
	 * @throws HinemosUnknown
	 */
	public String echo(String str) throws InvalidUserPass, InvalidRole, HinemosUnknown {
		ArrayList<String> roleList = new ArrayList<String>();
		HttpAuthenticator.authCheck(wsctx, roleList);

		return str + ", " + str;
	}

	/**
	 * ジョブ管理用のSessionBeanの取得
	 * 
	 * @return
	 * @throws HinemosUnknown
	 * @throws InvalidUserPass
	 */
	private JobControllerLocal getJobControllerLocal() throws HinemosUnknown, InvalidUserPass {
		JobControllerLocal local = null;
		try {
			String account = HttpAuthenticator.getAccount(wsctx);
			NamingContext namingContext = HttpAuthenticator.getContext(account);
			JobControllerLocalHome localHome =
				(JobControllerLocalHome)
				namingContext.lookup(JobControllerLocalHome.JNDI_NAME);
			local = localHome.create();
		} catch (CreateException e) {
			m_log.error("getJobControllerLocal CreateException : " + e,e);
			throw new HinemosUnknown(e.getMessage(), e);
		} catch (NamingException e) {
			m_log.error("getJobControllerLocal NamingException : " + e,e);
			throw new HinemosUnknown(e.getMessage(), e);
		}
		return local;
	}


	/**
	 * ジョブツリー情報を取得する。<BR>
	 * 
	 * JobManagementRead権限が必要
	 * 
	 * @param treeOnly
	 * @throws UserNotFound
	 * @throws JobMasterNotFound
	 * @throws HinemosUnknown
	 * @throws NotifyNotFound
	 * @throws InvalidRole
	 * @throws InvalidUserPass
	 */
	public JobTreeItem getJobTree(boolean treeOnly) throws NotifyNotFound, HinemosUnknown, JobMasterNotFound, UserNotFound, InvalidUserPass, InvalidRole {
		m_log.debug("getJobTree : treeOnly=" + treeOnly);
		ArrayList<String> roleList = new ArrayList<String>();
		roleList.add(RoleConstant.JOBMANAGEMENT_READ);
		HttpAuthenticator.authCheck(wsctx, roleList);

		// 認証済み操作ログ
		StringBuffer msg = new StringBuffer();
		msg.append(", TreeOnly=");
		msg.append(treeOnly);
		m_opelog.debug(HinemosModuleConstant.LOG_PREFIX_JOB + " Get, Method=getJobTree, User="
				+ HttpAuthenticator.getUserAccountString(wsctx)
				+ msg.toString());

		JobControllerLocal job = getJobControllerLocal();
		return job.getJobTree(treeOnly, Locale.getDefault());
	}

	/**
	 * ジョブツリー情報を取得する。<BR>
	 * 
	 * JobManagementRead権限が必要
	 * 
	 * @param jobInfo ジョブ情報(ツリー情報のみ)
	 * @return ジョブ情報(Full)
	 */
	public JobInfo getJobFull(JobInfo jobInfo) throws NotifyNotFound,
	JobMasterNotFound, UserNotFound, HinemosUnknown, InvalidUserPass, InvalidRole {
		m_log.debug("getJobFull : jobInfo=" + jobInfo);
		ArrayList<String> roleList = new ArrayList<String>();
		roleList.add(RoleConstant.JOBMANAGEMENT_READ);
		HttpAuthenticator.authCheck(wsctx, roleList);

		// 認証済み操作ログ
		if(jobInfo != null){
			StringBuffer msg = new StringBuffer();
			msg.append(", JobID=");
			msg.append(jobInfo.getId());
			m_opelog.debug(HinemosModuleConstant.LOG_PREFIX_JOB + " Get, Method=getJobFull, User="
					+ HttpAuthenticator.getUserAccountString(wsctx)
					+ msg.toString());
		}

		JobControllerLocal job = getJobControllerLocal();
		return job.getJobFull(jobInfo);
	}

	/**
	 * ジョブツリー情報を登録する。<BR>
	 * 
	 * JobManagementWrite権限が必要
	 * 
	 * @param item ジョブツリー情報{@link com.clustercontrol.jobmanagement.bean.JobTreeItem}の階層オブジェクト
	 * @throws JobInvalid
	 * @throws JobMasterNotFound
	 * @throws HinemosUnknown
	 * @throws InvalidRole
	 * @throws InvalidUserPass
	 * @see com.clustercontrol.jobmanagement.factory.ModifyJob#registerJob(JobTreeItem, String)
	 */
	public void registerJob(JobTreeItem item) throws HinemosUnknown, JobMasterNotFound, JobInvalid, InvalidUserPass, InvalidRole, InvalidSetting {
		JobControllerLocal job = getJobControllerLocal();
		String id = null;
		if (item != null && item.getData() != null) {
			id = item.getData().getId();
		}
		m_log.debug("registerJob : Id=" + id + ", item="+ item);
		ArrayList<String> roleList = new ArrayList<String>();
		roleList.add(RoleConstant.JOBMANAGEMENT_WRITE);
		HttpAuthenticator.authCheck(wsctx, roleList);

		// 認証済み操作ログ
		m_opelog.info(HinemosModuleConstant.LOG_PREFIX_JOB + " Set Job, Method=registerJob, User="
				+ HttpAuthenticator.getUserAccountString(wsctx));

		job.registerJob(item);
	}

	/**
	 * ジョブユニット情報を登録する。<BR>
	 * 
	 * JobManagementWrite権限が必要
	 * 
	 * @param item ジョブユニット情報{@link com.clustercontrol.jobmanagement.bean.JobTreeItem}の階層オブジェクト
	 * @throws JobInvalid
	 * @throws JobMasterNotFound
	 * @throws HinemosUnknown
	 * @throws InvalidRole
	 * @throws InvalidUserPass
	 * @see com.clustercontrol.jobmanagement.factory.ModifyJob#registerJob(JobTreeItem, String)
	 */
	public void registerJobunit(JobTreeItem item) throws HinemosUnknown, JobMasterNotFound, JobInvalid, InvalidUserPass, InvalidRole, InvalidSetting, NotifyNotFound {
		JobControllerLocal job = getJobControllerLocal();
		String id = null;
		if (item != null && item.getData() != null) {
			id = item.getData().getId();
		}
		m_log.debug("registerJobunit : Id=" + id + ", item="+ item);
		ArrayList<String> roleList = new ArrayList<String>();
		roleList.add(RoleConstant.JOBMANAGEMENT_WRITE);
		HttpAuthenticator.authCheck(wsctx, roleList);


		// 認証済み操作ログ
		if (item != null && item.getData() != null) {
			StringBuffer msg = new StringBuffer();
			msg.append(", JobunitID=");
			msg.append(item.getData().getId());
			m_opelog.info(HinemosModuleConstant.LOG_PREFIX_JOB + " Set Jobunit, Method=registerJobunit, User="
					+ HttpAuthenticator.getUserAccountString(wsctx)
					+ msg.toString());
		}


		job.registerJobunit(item);
	}

	/**
	 * ジョブユニット単位でジョブツリー情報を削除する。<BR>
	 * 
	 * JobManagementWrite権限が必要
	 * 
	 * @param jobunitId 削除対象ジョブユニットのジョブID
	 * @throws JobMasterNotFound
	 * @throws HinemosUnknown
	 * @throws NotifyNotFound
	 * @throws InvalidRole
	 * @throws InvalidUserPass
	 */
	public void deleteJobunit(String jobunitId) throws NotifyNotFound, HinemosUnknown, JobMasterNotFound, InvalidUserPass, InvalidRole, InvalidSetting {
		m_log.debug("deleteJobunit : Id=" + jobunitId);
		ArrayList<String> roleList = new ArrayList<String>();
		roleList.add(RoleConstant.JOBMANAGEMENT_WRITE);
		HttpAuthenticator.authCheck(wsctx, roleList);

		// 認証済み操作ログ
		StringBuffer msg = new StringBuffer();
		msg.append(", JobunitID=");
		msg.append(jobunitId);
		m_opelog.info(HinemosModuleConstant.LOG_PREFIX_JOB + " Delete Jobunit, Method=deleteJobunit, User="
				+ HttpAuthenticator.getUserAccountString(wsctx)
				+ msg.toString());

		JobControllerLocal job = getJobControllerLocal();
		job.deleteJobunit(jobunitId);
	}


	/**
	 * ジョブ操作開始用プロパティを返します。<BR>
	 * 
	 * JobManagementExecute権限が必要
	 * 
	 * @param sessionId セッションID
	 * @param jobunitId 所属ジョブユニットのジョブID
	 * @param jobId ジョブID
	 * @param facilityId ファシリティID
	 * @param locale ロケール情報
	 * @return ジョブ操作開始用プロパティ
	 * @throws HinemosUnknown
	 * @throws InvalidRole
	 * @throws InvalidUserPass
	 * 
	 * @see com.clustercontrol.jobmanagement.factory.JobOperationProperty#getStartProperty(String, String, String, String, Locale)
	 */
	@SuppressWarnings("unchecked")
	public ArrayList<String> getAvailableStartOperation(String sessionId, String jobunitId, String jobId, String facilityId) throws InvalidUserPass, InvalidRole, HinemosUnknown {
		m_log.debug("getAvailableStartOperation : sessionId=" + sessionId +
				", jobunitId=" + jobunitId + ", jobId=" + jobId + ", facilityId=" + facilityId);
		ArrayList<String> roleList = new ArrayList<String>();
		roleList.add(RoleConstant.JOBMANAGEMENT_EXECUTE);
		HttpAuthenticator.authCheck(wsctx, roleList);

		// 認証済み操作ログ
		StringBuffer msg = new StringBuffer();
		msg.append(", SessionID=");
		msg.append(sessionId);
		msg.append(", JobunitID=");
		msg.append(jobunitId);
		msg.append(", JobID=");
		msg.append(jobId);
		msg.append(", FacilityID=");
		msg.append(facilityId);
		m_opelog.debug(HinemosModuleConstant.LOG_PREFIX_JOB + " Get, Method=getAvailableStartOperation, User="
				+ HttpAuthenticator.getUserAccountString(wsctx)
				+ msg.toString());

		JobControllerLocal job = getJobControllerLocal();
		return job.getAvailableStartOperation(sessionId, jobunitId, jobId, facilityId, Locale.getDefault());
	}

	/**
	 * ジョブ操作停止用プロパティを返します。<BR>
	 * 
	 * JobManagementExecute権限が必要
	 * 
	 * @param sessionId セッションID
	 * @param jobunitId 所属ジョブユニットのジョブID
	 * @param jobId ジョブID
	 * @param facilityId ファシリティID
	 * @param locale ロケール情報
	 * @return ジョブ操作停止用プロパティ
	 * @throws HinemosUnknown
	 * @throws InvalidRole
	 * @throws InvalidUserPass
	 * 
	 * @see com.clustercontrol.jobmanagement.factory.JobOperationProperty#getStopProperty(String, String, String, String, Locale)
	 */
	@SuppressWarnings("unchecked")
	public ArrayList<String> getAvailableStopOperation(String sessionId, String jobunitId,  String jobId, String facilityId) throws InvalidUserPass, InvalidRole, HinemosUnknown {
		m_log.debug("getAvailableStopOperation : sessionId=" + sessionId +
				", jobunitId=" + jobunitId + ", jobId=" + jobId + ", facilityId=" + facilityId);
		ArrayList<String> roleList = new ArrayList<String>();
		roleList.add(RoleConstant.JOBMANAGEMENT_EXECUTE);
		HttpAuthenticator.authCheck(wsctx, roleList);

		// 認証済み操作ログ
		StringBuffer msg = new StringBuffer();
		msg.append(", SessionID=");
		msg.append(sessionId);
		msg.append(", JobunitID=");
		msg.append(jobunitId);
		msg.append(", JobID=");
		msg.append(jobId);
		msg.append(", FacilityID=");
		msg.append(facilityId);
		m_opelog.debug(HinemosModuleConstant.LOG_PREFIX_JOB + " Get, Method=getAvailableStopOperation, User="
				+ HttpAuthenticator.getUserAccountString(wsctx)
				+ msg.toString());

		JobControllerLocal job = getJobControllerLocal();
		return job.getAvailableStopOperation(sessionId, jobunitId, jobId, facilityId, Locale.getDefault());
	}

	/**
	 * ジョブを実行します。<BR>
	 * 
	 * JobManagementExecute権限が必要
	 * 
	 * @param jobunitId 所属ジョブユニットのジョブID
	 * @param jobId ジョブID
	 * @param info ログ出力情報
	 * @param triggerInfo 実行契機情報
	 * @throws JobMasterNotFound
	 * @throws JobInfoNotFound
	 * @throws HinemosUnknown
	 * @throws FacilityNotFound
	 * @throws InvalidRole
	 * @throws InvalidUserPass
	 * @see com.clustercontrol.jobmanagement.ejb.session.JobControllerBean#createJobInfo(String, String, NotifyRequestMessage, JobTriggerInfo}
	 * @see com.clustercontrol.jobmanagement.ejb.session.JobRunManagementBean#runJob(String, String)
	 */
	public String runJob(String jobunitId, String jobId, OutputBasicInfo info, JobTriggerInfo triggerInfo) throws FacilityNotFound, HinemosUnknown, JobInfoNotFound, JobMasterNotFound, InvalidUserPass, InvalidRole
	{
		m_log.debug("runJob : jobunitId=" + jobunitId + ", jobId=" + jobId + ", info=" + info +
				", triggerInfo=" + triggerInfo);
		ArrayList<String> roleList = new ArrayList<String>();
		roleList.add(RoleConstant.JOBMANAGEMENT_EXECUTE);
		HttpAuthenticator.authCheck(wsctx, roleList);

		// 認証済み操作ログ
		StringBuffer msg = new StringBuffer();
		msg.append(", JobID=");
		msg.append(jobId);
		msg.append(", Trigger=");
		msg.append(triggerInfo==null?null:triggerTypeToString(triggerInfo.getTrigger_type()));
		m_opelog.info(HinemosModuleConstant.LOG_PREFIX_JOB + " Run Job, Method=runJob, User="
				+ HttpAuthenticator.getUserAccountString(wsctx)
				+ msg.toString());

		JobControllerLocal job = getJobControllerLocal();
		return job.runJob(jobunitId, jobId, info, triggerInfo);
	}

	/**
	 * ジョブ操作を行います。<BR>
	 * 
	 * JobManagementExecute権限が必要
	 * 
	 * @param property ジョブ操作用プロパティ
	 * @throws JobInfoNotFound
	 * @throws HinemosUnknown
	 * @throws InvalidRole
	 * @throws InvalidUserPass
	 * @see com.clustercontrol.jobmanagement.ejb.session.JobRunManagementBean#operationJob(Property)
	 */
	public void operationJob(NodeOperationInfo property) throws HinemosUnknown, JobInfoNotFound, InvalidUserPass, InvalidRole {
		m_log.debug("operationJob : nodeOperationInfo=" + property);
		ArrayList<String> roleList = new ArrayList<String>();
		roleList.add(RoleConstant.JOBMANAGEMENT_EXECUTE);
		HttpAuthenticator.authCheck(wsctx, roleList);

		// 認証済み操作ログ
		if(property != null){
			StringBuffer msg = new StringBuffer();
			msg.append(", SessionID=");
			msg.append(property.getSessionId());
			msg.append(", JobID=");
			msg.append(property.getJobId());
			msg.append(", FacilityID=");
			msg.append(property.getFacilityId()==null?"(not set)":property.getFacilityId());
			msg.append(", Operation=");
			msg.append(getOperationString(property.getControl()));
			msg.append(", EndValue=");
			msg.append(property.getEndValue()==null?"(not set)":property.getEndValue());
			m_opelog.info(HinemosModuleConstant.LOG_PREFIX_JOB + " Operate Job, Method=operationJob, User="
					+ HttpAuthenticator.getUserAccountString(wsctx)
					+ msg.toString());
		}

		JobControllerLocal job = getJobControllerLocal();
		job.operationJob(property);
	}


	/**
	 * ジョブ履歴一覧情報を返します。<BR>
	 * 
	 * JobManagementRead権限が必要
	 * 
	 * @param property 履歴フィルタ用プロパティ
	 * @param histories 表示履歴数
	 * @return ジョブ履歴一覧情報（Objectの2次元配列）
	 * @throws HinemosUnknown
	 * @throws JobInfoNotFound
	 * @throws InvalidRole
	 * @throws InvalidUserPass
	 * @see com.clustercontrol.jobmanagement.factory.SelectJob#getHistoryList(Property, int)
	 */
	public JobHistoryList getJobHistoryList(JobHistoryFilter property, int histories) throws JobInfoNotFound, HinemosUnknown, InvalidUserPass, InvalidRole {
		m_log.debug("getHistoryList : jobHistoryFilter=" + property + ", histories=" + histories);
		ArrayList<String> roleList = new ArrayList<String>();
		roleList.add(RoleConstant.JOBMANAGEMENT_READ);
		HttpAuthenticator.authCheck(wsctx, roleList);

		// 認証済み操作ログ
		StringBuffer msg = new StringBuffer();
		if(property != null){
			SimpleDateFormat sdf = new SimpleDateFormat("yyyy/MM/dd HH:mm:ss");
			msg.append(", StartFromDate=");
			msg.append(property.getStartFromDate()==null?null:sdf.format(new Date(property.getStartFromDate())));
			msg.append(", StartToDate=");
			msg.append(property.getStartToDate()==null?null:sdf.format(new Date(property.getStartToDate())));
			msg.append(", EndFromDate=");
			msg.append(property.getEndFromDate()==null?null:sdf.format(new Date(property.getEndFromDate())));
			msg.append(", EndToDate=");
			msg.append(property.getEndToDate()==null?null:sdf.format(new Date(property.getEndToDate())));
			msg.append(", JobID=");
			msg.append(property.getJobId());
			msg.append(", Status=");
			msg.append(property.getStatus());
			msg.append(", TriggerType=");
			msg.append(property.getTriggerType());
			msg.append(", TriggerInfo=");
			msg.append(property.getTriggerInfo());
		}
		msg.append(", Count=");
		msg.append(histories);
		m_opelog.debug(HinemosModuleConstant.LOG_PREFIX_JOB + " Get, Method=getJobHistoryList, User="
				+ HttpAuthenticator.getUserAccountString(wsctx)
				+ msg.toString());

		JobControllerLocal job = getJobControllerLocal();
		return job.getJobHistoryList(property, histories);
	}

	/**
	 * ジョブ詳細一覧情報を返します。<BR>
	 * 
	 * JobManagementRead権限が必要
	 * 
	 * @param sessionId セッションID
	 * @return ジョブ詳細一覧情報（Objectの2次元配列）
	 * @throws JobMasterNotFound
	 * @throws HinemosUnknown
	 * @throws JobInfoNotFound
	 * @throws InvalidRole
	 * @throws InvalidUserPass
	 * @see com.clustercontrol.jobmanagement.factory.SelectJob#getDetailList(String)
	 */
	public JobTreeItem getJobDetailList(String sessionId) throws JobInfoNotFound, HinemosUnknown, JobMasterNotFound, InvalidUserPass, InvalidRole {
		JobControllerLocal job = getJobControllerLocal();
		ArrayList<String> roleList = new ArrayList<String>();
		roleList.add(RoleConstant.JOBMANAGEMENT_READ);
		HttpAuthenticator.authCheck(wsctx, roleList);

		// 認証済み操作ログ
		StringBuffer msg = new StringBuffer();
		msg.append(", SessionID=");
		msg.append(sessionId);
		m_opelog.debug(HinemosModuleConstant.LOG_PREFIX_JOB + " Get, Method=getJobDetailList, User="
				+ HttpAuthenticator.getUserAccountString(wsctx)
				+ msg.toString());

		return job.getJobDetailList(sessionId);
	}

	/**
	 * ノード詳細一覧情報を返します。<BR>
	 * 
	 * JobManagementRead権限が必要
	 * 
	 * @param sessionId セッションID
	 * @param jobunitId 所属ジョブユニットのジョブID
	 * @param jobId ジョブID
	 * @param locale ロケール情報
	 * @return ノード詳細一覧情報（Objectの2次元配列）
	 * @throws HinemosUnknown
	 * @throws InvalidRole
	 * @throws InvalidUserPass
	 * @throws JobInfoNotFound
	 * @see com.clustercontrol.jobmanagement.factory.SelectJob#getNodeDetailList(String, String, String, Locale)
	 */
	@SuppressWarnings("unchecked")
	public ArrayList<JobNodeDetail> getNodeDetailList(String sessionId, String jobunitId, String jobId) throws InvalidUserPass, InvalidRole, HinemosUnknown, JobInfoNotFound {
		JobControllerLocal job = getJobControllerLocal();
		ArrayList<String> roleList = new ArrayList<String>();
		roleList.add(RoleConstant.JOBMANAGEMENT_READ);
		HttpAuthenticator.authCheck(wsctx, roleList);

		// 認証済み操作ログ
		StringBuffer msg = new StringBuffer();
		msg.append(", SessionID=");
		msg.append(sessionId);
		msg.append(", JobunitID=");
		msg.append(jobunitId);
		msg.append(", JobID=");
		msg.append(jobId);
		m_opelog.debug(HinemosModuleConstant.LOG_PREFIX_JOB + " Get, Method=getNodeDetailList, User="
				+ HttpAuthenticator.getUserAccountString(wsctx)
				+ msg.toString());

		return job.getNodeDetailList(sessionId, jobunitId, jobId, Locale.getDefault());
	}

	/**
	 * ファイル転送一覧情報を返します。<BR>
	 * 
	 * JobManagementRead権限が必要
	 * 
	 * @param sessionId セッションID
	 * @param jobunitId 所属ジョブユニットのジョブID
	 * @param jobId ジョブID
	 * @return ファイル転送一覧情報（Objectの2次元配列）
	 * @throws HinemosUnknown
	 * @throws JobInfoNotFound
	 * @throws InvalidRole
	 * @throws InvalidUserPass
	 * @see com.clustercontrol.jobmanagement.factory.SelectJob#getForwardFileList(String, String)
	 */
	@SuppressWarnings("unchecked")
	public ArrayList<JobForwardFile> getForwardFileList(String sessionId, String jobunitId, String jobId) throws JobInfoNotFound, HinemosUnknown, InvalidUserPass, InvalidRole {
		JobControllerLocal job = getJobControllerLocal();
		ArrayList<String> roleList = new ArrayList<String>();
		roleList.add(RoleConstant.JOBMANAGEMENT_READ);
		HttpAuthenticator.authCheck(wsctx, roleList);

		// 認証済み操作ログ
		StringBuffer msg = new StringBuffer();
		msg.append(", SessionID=");
		msg.append(sessionId);
		msg.append(", JobunitID=");
		msg.append(jobunitId);
		msg.append(", JobID=");
		msg.append(jobId);
		m_opelog.debug(HinemosModuleConstant.LOG_PREFIX_JOB + " Get, Method=getForwardFileList, User="
				+ HttpAuthenticator.getUserAccountString(wsctx)
				+ msg.toString());

		return job.getForwardFileList(sessionId, jobunitId, jobId);
	}

	/**
	 * スケジュール情報を登録します。<BR>
	 * 
	 * JobManagementWrite権限が必要
	 * 
	 * @param info スケジュール情報
	 * @throws HinemosUnknown
	 * @throws InvalidRole
	 * @throws InvalidUserPass
	 * @throws InvalidSetting
	 * 
	 * @see com.clustercontrol.jobmanagement.factory.ModifySchedule#addSchedule(JobSchedule, String)
	 */
	public void addSchedule(JobSchedule info) throws HinemosUnknown, InvalidUserPass, InvalidRole, InvalidSetting {
		m_log.debug("addSchedule : jobSchedule=" + info);
		ArrayList<String> roleList = new ArrayList<String>();
		roleList.add(RoleConstant.JOBMANAGEMENT_WRITE);
		HttpAuthenticator.authCheck(wsctx, roleList);

		// 認証済み操作ログ
		if(info != null){
			StringBuffer msg = new StringBuffer();
			msg.append(", ScheduleID=");
			msg.append(info.getId());
			m_opelog.info(HinemosModuleConstant.LOG_PREFIX_JOB + " Add Schedule, Method=addSchedule, User="
					+ HttpAuthenticator.getUserAccountString(wsctx)
					+ msg.toString());
		}

		JobControllerLocal job = getJobControllerLocal();
		job.addSchedule(info);
	}

	/**
	 * スケジュール情報の有効/無効を変更します。<BR>
	 * 
	 * JobManagementWrite権限が必要
	 * 
	 * @param scheduleId スケジュールID
	 * @param validFlag 有効/無効
	 * @throws HinemosUnknown
	 * @throws InvalidUserPass
	 * @throws InvalidRole
	 * @throws InvalidSetting
	 */
	public void setScheduleStatus(String scheduleId, boolean validFlag) throws HinemosUnknown, InvalidUserPass, InvalidRole, InvalidSetting {
		m_log.debug("setScheduleStatus : scheduleId=" + scheduleId + ", validFlag=" + validFlag);
		ArrayList<String> roleList = new ArrayList<String>();
		roleList.add(RoleConstant.JOBMANAGEMENT_WRITE);
		HttpAuthenticator.authCheck(wsctx, roleList);

		// 認証済み操作ログ
		StringBuffer msg = new StringBuffer();
		msg.append(", ScheduleID=");
		msg.append(scheduleId);
		msg.append(", ValidFlag=");
		msg.append(validFlag);
		m_opelog.info(HinemosModuleConstant.LOG_PREFIX_JOB + " Change Valid, Method=setScheduleStatus, User="
				+ HttpAuthenticator.getUserAccountString(wsctx)
				+ msg.toString());


		JobControllerLocal job = getJobControllerLocal();
		job.setScheduleStatus(scheduleId, validFlag);
	}

	/**
	 * スケジュール情報を変更します。<BR>
	 * 
	 * JobManagementWrite権限が必要
	 * 
	 * @param info スケジュール情報
	 * @throws HinemosUnknown
	 * @throws InvalidRole
	 * @throws InvalidUserPass
	 * @throws InvalidSetting
	 * @see com.clustercontrol.jobmanagement.factory.ModifySchedule#modifySchedule(JobSchedule, String)
	 */
	public void modifySchedule(JobSchedule info) throws HinemosUnknown, InvalidUserPass, InvalidRole, InvalidSetting {
		m_log.debug("modifySchedule : jobSchedule=" + info);
		ArrayList<String> roleList = new ArrayList<String>();
		roleList.add(RoleConstant.JOBMANAGEMENT_WRITE);
		HttpAuthenticator.authCheck(wsctx, roleList);

		// 認証済み操作ログ
		if(info != null){
			StringBuffer msg = new StringBuffer();
			msg.append(", ScheduleID=");
			msg.append(info.getId());
			m_opelog.info(HinemosModuleConstant.LOG_PREFIX_JOB + " Change Schedule, Method=modifySchedule, User="
					+ HttpAuthenticator.getUserAccountString(wsctx)
					+ msg.toString());
		}

		JobControllerLocal job = getJobControllerLocal();
		job.modifySchedule(info);
	}

	/**
	 * スケジュール情報を削除します。<BR>
	 * 
	 * JobManagementWrite権限が必要
	 * 
	 * @param scheduleId スケジュールID
	 * @throws HinemosUnknown
	 * @throws InvalidRole
	 * @throws InvalidUserPass
	 * 
	 * @see com.clustercontrol.jobmanagement.factory.ModifySchedule#deleteSchedule(String)
	 */
	public void deleteSchedule(String scheduleId) throws HinemosUnknown, InvalidUserPass, InvalidRole {
		m_log.debug("deleteSchedule : scheduleId=" + scheduleId);
		ArrayList<String> roleList = new ArrayList<String>();
		roleList.add(RoleConstant.JOBMANAGEMENT_WRITE);
		HttpAuthenticator.authCheck(wsctx, roleList);

		// 認証済み操作ログ
		StringBuffer msg = new StringBuffer();
		msg.append(", ScheduleID=");
		msg.append(scheduleId);
		m_opelog.info(HinemosModuleConstant.LOG_PREFIX_JOB + " Delete Schedule, Method=deleteSchedule, User="
				+ HttpAuthenticator.getUserAccountString(wsctx)
				+ msg.toString());

		JobControllerLocal job = getJobControllerLocal();
		job.deleteSchedule(scheduleId);
	}

	/**
	 * スケジュール一覧情報を返します。<BR>
	 * 
	 * JobManagementRead権限が必要
	 * 
	 * @return スケジュール一覧情報（Objectの2次元配列）
	 * @throws HinemosUnknown
	 * @throws JobMasterNotFound
	 * @throws InvalidRole
	 * @throws InvalidUserPass
	 * @see com.clustercontrol.jobmanagement.factory.SelectSchedule#getScheduleList()
	 */
	@SuppressWarnings("unchecked")
	public ArrayList<JobSchedule> getScheduleList() throws JobMasterNotFound, HinemosUnknown, InvalidUserPass, InvalidRole {
		m_log.debug("getScheduleList :");
		ArrayList<String> roleList = new ArrayList<String>();
		roleList.add(RoleConstant.JOBMANAGEMENT_READ);
		HttpAuthenticator.authCheck(wsctx, roleList);

		// 認証済み操作ログ
		m_opelog.debug(HinemosModuleConstant.LOG_PREFIX_JOB + " Get, Method=getScheduleList, User="
				+ HttpAuthenticator.getUserAccountString(wsctx));

		JobControllerLocal job = getJobControllerLocal();
		return job.getScheduleList();
	}

	/**
	 * カレンダIDリストを返します。<BR>
	 * 
	 * JobManagementRead権限が必要
	 * 
	 * @return カレンダIDのリスト
	 * @throws HinemosUnknown
	 * @throws InvalidRole
	 * @throws InvalidUserPass
	 * @throws CalendarNotFound
	 * @throws
	 * 
	 * @see com.clustercontrol.calendar.ejb.session.CalendarControllerBean#getCalendarIdList()
	 */
	@SuppressWarnings("unchecked")
	public ArrayList<String> getCalendarIdList() throws InvalidUserPass, InvalidRole, HinemosUnknown, CalendarNotFound {
		m_log.debug("getCalendarIdList :");
		ArrayList<String> roleList = new ArrayList<String>();
		roleList.add(RoleConstant.JOBMANAGEMENT_READ);
		roleList.add(RoleConstant.CALENDAR_READ);
		HttpAuthenticator.authCheck(wsctx, roleList);

		// 認証済み操作ログ
		m_opelog.debug(HinemosModuleConstant.LOG_PREFIX_JOB + " Get, Method=getCalendarIdList, User="
				+ HttpAuthenticator.getUserAccountString(wsctx));

		JobControllerLocal job = getJobControllerLocal();
		return job.getCalendarIdList();
	}

	/**
	 * セッションジョブ情報を返します。<BR>
	 * 
	 * JobManagementRead権限が必要
	 * 
	 * @param sessionId セッションID
	 * @param jobunitId 所属ジョブユニットのジョブID
	 * @param jobId ジョブID
	 * @return ジョブツリー情報{@link com.clustercontrol.jobmanagement.bean.JobTreeItem}
	 * @throws HinemosUnknown
	 * @throws InvalidRole
	 * @throws InvalidUserPass
	 * @throws JobInfoNotFound
	 * @throws NotifyNotFound
	 * @see com.clustercontrol.jobmanagement.factory.SelectJob#getSessionJobInfo(String, String, String)
	 */
	public JobTreeItem getSessionJobInfo(String sessionId, String jobunitId, String jobId) throws InvalidUserPass, InvalidRole, HinemosUnknown, NotifyNotFound, JobInfoNotFound {
		m_log.debug("getSessionJobInfo : sessionId=" + sessionId + ", jobunitId=" + jobunitId +
				", jobId=" + jobId);
		ArrayList<String> roleList = new ArrayList<String>();
		roleList.add(RoleConstant.JOBMANAGEMENT_READ);
		HttpAuthenticator.authCheck(wsctx, roleList);

		// 認証済み操作ログ
		StringBuffer msg = new StringBuffer();
		msg.append(", SessionID=");
		msg.append(sessionId);
		msg.append(", JobunitID=");
		msg.append(jobunitId);
		msg.append(", JobID=");
		msg.append(jobId);
		m_opelog.debug(HinemosModuleConstant.LOG_PREFIX_JOB + " Get, Method=getSessionJobInfo, User="
				+ HttpAuthenticator.getUserAccountString(wsctx)
				+ msg.toString());

		JobControllerLocal job = getJobControllerLocal();
		return job.getSessionJobInfo(sessionId, jobunitId, jobId);
	}

	/**
	 * 実行契機の文字列取得
	 * 
	 * @param type 実行契機
	 * @return 実行契機文字列
	 */
	private String triggerTypeToString(Integer type){
		if (type != null) {
			if (type == JobTriggerTypeConstant.TYPE_SCHEDULE) {
				return "Schedule";
			} else if (type == JobTriggerTypeConstant.TYPE_MANUAL) {
				return "Manual";
			} else if (type == JobTriggerTypeConstant.TYPE_MONITOR) {
				return "Monitor";
			}
		}
		return "";
	}

	/**
	 * ログ出力用の操作名取得
	 * 
	 * @param op 操作名(画面表示)
	 * @return ログ出力用操作名
	 */
	private static String getOperationString(Integer op){
		if (op ==null) {
			return "[Unknown Operation]";
		}

		switch (op.intValue()) {
		case OperationConstant.TYPE_START_AT_ONCE://1
			return "Start[Start]";

		case OperationConstant.TYPE_START_SUSPEND://3
			return "Start[Cancel Suspend]";

		case OperationConstant.TYPE_START_SKIP://5
			return "Start[Cancel Skip]";

		case OperationConstant.TYPE_START_WAIT://7
			return "Start[Cancel Pause]";

		case OperationConstant.TYPE_STOP_AT_ONCE://0
			return "Stop[Command]";

		case OperationConstant.TYPE_STOP_SUSPEND://2
			return "Stop[Suspend]";

		case OperationConstant.TYPE_STOP_SKIP://4
			return "Stop[Skip]";

		case OperationConstant.TYPE_STOP_WAIT://6
			return "Stop[Pause]";

		case OperationConstant.TYPE_STOP_MAINTENANCE://8
			return "Stop[Change End Value]";

		case OperationConstant.TYPE_STOP_SET_END_VALUE://10
			return "Stop[Set End Value]";

		case OperationConstant.TYPE_STOP_FORCE://11
			return "Stop[Force]";

		default:
			return "[Unknown Operation]";
		}
	}
}
