/*
Copyright (C) 2011 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.
 */
package com.clustercontrol.ws.endpoint;

import java.io.File;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;

import javax.activation.DataHandler;
import javax.activation.FileDataSource;
import javax.annotation.Resource;
import javax.ejb.CreateException;
import javax.naming.NamingException;
import javax.xml.bind.annotation.XmlMimeType;
import javax.xml.ws.WebServiceContext;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.jnp.interfaces.NamingContext;

import com.clustercontrol.fault.FacilityNotFound;
import com.clustercontrol.fault.HinemosUnknown;
import com.clustercontrol.fault.InvalidRole;
import com.clustercontrol.fault.InvalidUserPass;
import com.clustercontrol.fault.JobInfoNotFound;
import com.clustercontrol.fault.CustomInvalid;
import com.clustercontrol.fault.MonitorNotFound;
import com.clustercontrol.accesscontrol.bean.RoleConstant;
import com.clustercontrol.commons.bean.SettingUpdateInfo;
import com.clustercontrol.custom.bean.CommandExecuteDTO;
import com.clustercontrol.custom.bean.CommandResultDTO;
import com.clustercontrol.custom.ejb.session.MonitorCustomControllerLocal;
import com.clustercontrol.custom.ejb.session.MonitorCustomControllerUtil;
import com.clustercontrol.hinemosagent.bean.AgentInfo;
import com.clustercontrol.hinemosagent.bean.AgentOutputBasicInfo;
import com.clustercontrol.hinemosagent.bean.TopicInfo;
import com.clustercontrol.hinemosagent.util.AgentConnectUtil;
import com.clustercontrol.jobmanagement.bean.RunResultInfo;
import com.clustercontrol.jobmanagement.ejb.session.JobRunManagementLocal;
import com.clustercontrol.jobmanagement.ejb.session.JobRunManagementLocalHome;
import com.clustercontrol.logfile.ejb.session.MonitorLogfileControllerLocal;
import com.clustercontrol.logfile.ejb.session.MonitorLogfileControllerUtil;
import com.clustercontrol.monitor.run.bean.MonitorInfo;
import com.clustercontrol.monitor.run.bean.MonitorStringValueInfo;
import com.clustercontrol.notify.bean.OutputBasicInfo;
import com.clustercontrol.notify.util.LogDirectNotifier;
import com.clustercontrol.repository.ejb.session.RepositoryControllerLocal;
import com.clustercontrol.repository.ejb.session.RepositoryControllerUtil;
import com.clustercontrol.repository.factory.FacilitySelector;
import com.clustercontrol.systemlog.service.MessageInfo;

/**
 * ジョブ操作用のWebAPIエンドポイント
 */
@javax.jws.WebService(targetNamespace = "http://agent.ws.clustercontrol.com")
public class AgentEndpoint {
	@Resource
	WebServiceContext wsctx;

	private static Log m_log = LogFactory.getLog( AgentEndpoint.class );

	/**
	 * echo(WebサービスAPI疎通用)
	 * 
	 * 権限必要なし（ユーザ名チェックのみ実施）
	 * 
	 * @param str
	 * @return
	 * @throws InvalidUserPass
	 * @throws InvalidRole
	 * @throws HinemosUnknown
	 */
	public String echo(String str) throws InvalidUserPass, InvalidRole, HinemosUnknown {
		ArrayList<String> roleList = new ArrayList<String>();
		HttpAuthenticator.authCheck(wsctx, roleList);

		return str + ", " + str;
	}

	/**
	 * ジョブ管理用のSessionBeanの取得
	 * 
	 * @return
	 * @throws HinemosUnknown
	 * @throws InvalidUserPass
	 */
	private JobRunManagementLocal getJobRunManagementLocal() throws HinemosUnknown, InvalidUserPass {
		JobRunManagementLocal local = null;
		try {
			String account = HttpAuthenticator.getAccount(wsctx);
			NamingContext namingContext = HttpAuthenticator.getContext(account);
			JobRunManagementLocalHome localHome =
				(JobRunManagementLocalHome)
				namingContext.lookup(JobRunManagementLocalHome.JNDI_NAME);
			local = localHome.create();
		} catch (CreateException e) {
			m_log.error("getJobRunManagementLocal CreateException : " + e,e);
			throw new HinemosUnknown(e.getMessage(), e);
		} catch (NamingException e) {
			m_log.error("getJobRunManagementLocal NamingException : " + e,e);
			throw new HinemosUnknown(e.getMessage(), e);
		}
		return local;
	}

	/**
	 * ログファイル監視のSessionBeanの取得
	 * 
	 * @return
	 * @throws HinemosUnknown
	 */
	private MonitorLogfileControllerLocal getMonitorLogfileControllerLocal() throws HinemosUnknown {
		MonitorLogfileControllerLocal local = null;
		try {
			local = MonitorLogfileControllerUtil.getLocalHome().create();
		} catch (CreateException e) {
			m_log.error("MonitorLogfileControllerLocal CreateException : " + e,e);
			throw new HinemosUnknown(e.getMessage(), e);
		} catch (NamingException e) {
			m_log.error("MonitorLogfileControllerLocal NamingException : " + e,e);
			throw new HinemosUnknown(e.getMessage(), e);
		}
		return local;
	}

	/**
	 * リポジトリ管理のSessionBeanの取得
	 * 
	 * @return
	 * @throws HinemosUnknown
	 */
	private RepositoryControllerLocal getRepositoryControllerLocal() throws HinemosUnknown {
		RepositoryControllerLocal local = null;
		try {
			local = RepositoryControllerUtil.getLocalHome().create();
		} catch (CreateException e) {
			m_log.error("RepositoryControllerLocal CreateException : " + e,e);
			throw new HinemosUnknown(e.getMessage(), e);
		} catch (NamingException e) {
			m_log.error("RepositoryControllerLocal NamingException : " + e,e);
			throw new HinemosUnknown(e.getMessage(), e);
		}
		return local;
	}

	/**
	 * [Agent Base] エージェントからwebサービスで利用。
	 * 
	 * HinemosAgentAccess権限が必要
	 * 
	 * @param agentInfo
	 * @return
	 * @throws InvalidRole
	 * @throws InvalidUserPass
	 * @throws HinemosUnknown
	 */
	public ArrayList<TopicInfo> getTopic (AgentInfo agentInfo) throws HinemosUnknown, InvalidUserPass, InvalidRole
	{
		ArrayList<String> roleList = new ArrayList<String>();
		roleList.add(RoleConstant.HINEMOS_AGENT_ACCESS);
		HttpAuthenticator.authCheck(wsctx, roleList);

		ArrayList<String> facilityIdList = getFacilityId(agentInfo);
		for (String facilityId : facilityIdList) {
			AgentConnectUtil.checkAgentLibMd5(facilityId);
		}

		ArrayList<TopicInfo> topicInfoList = new ArrayList<TopicInfo>();
		for (String facilityId : facilityIdList) {
			ArrayList<TopicInfo> list = AgentConnectUtil.getTopic(facilityId);
			/*
			 * Agent.propertiesでfacilityIdが直接指定されていない場合、
			 * agentInfoにfacilityIdが含まれていないので、ここで詰める。
			 */
			agentInfo.setFacilityId(facilityId);
			AgentConnectUtil.putAgentMap(agentInfo);
			if (list != null && list.size() != 0) {
				topicInfoList.addAll(list);
			}
		}
		return topicInfoList;
	}



	/**
	 * [Agent Base] エージェントからwebサービスで利用。
	 * エージェントがシャットダウンする際に利用。(shutdownhook に登録)
	 * 
	 * HinemosAgentAccess権限が必要
	 * 
	 * @param agentInfo
	 * @return
	 * @throws InvalidRole
	 * @throws InvalidUserPass
	 * @throws HinemosUnknown
	 */
	public void deleteAgent (AgentInfo agentInfo) throws HinemosUnknown, InvalidUserPass, InvalidRole {
		ArrayList<String> roleList = new ArrayList<String>();
		roleList.add(RoleConstant.HINEMOS_AGENT_ACCESS);
		HttpAuthenticator.authCheck(wsctx, roleList);

		ArrayList<String> facilityIdList = getFacilityId(agentInfo);
		for (String facilityId : facilityIdList) {
			m_log.info("deleteAgent " + facilityId + " is shutdown");
			AgentConnectUtil.deleteAgent(facilityId);
		}
	}

	/**
	 * [Agent Base] Internalイベントに出力する際に利用する。
	 * 
	 * HinemosAgentAccess権限が必要
	 * 
	 * @param message
	 * @throws HinemosUnknown
	 * @throws FacilityNotFound
	 * @throws InvalidRole
	 * @throws InvalidUserPass
	 */
	public void sendMessage(AgentOutputBasicInfo message)
	throws HinemosUnknown, FacilityNotFound,
	InvalidUserPass, InvalidRole {
		ArrayList<String> roleList = new ArrayList<String>();
		roleList.add(RoleConstant.HINEMOS_AGENT_ACCESS);
		HttpAuthenticator.authCheck(wsctx, roleList);

		ArrayList<String> facilityIdList = getFacilityId(message.getAgentInfo());
		OutputBasicInfo outputBasicInfo = message.getOutputBasicInfo();
		if (facilityIdList == null || facilityIdList.size() == 0) {
			m_log.warn("sendMessage facilityId is null");
			return;
		}
		if (facilityIdList.size() == 0) {
			m_log.warn("sendMessage facilityId.size() is 0");
			return;
		}
		AgentConnectUtil.sendMessageLocal(outputBasicInfo, facilityIdList);
	}

	/**
	 * [Agent Base] AgentInfoからFacilityIdのリストを取得する。
	 * 
	 * @param agentInfo
	 * @return
	 * @throws HinemosUnknown
	 */
	private ArrayList<String> getFacilityId(AgentInfo agentInfo) throws HinemosUnknown {
		ArrayList<String> facilityIdList = new ArrayList<String>();

		if (agentInfo.getFacilityId() != null && !agentInfo.getFacilityId().equals("")) {
			/*
			 * agentInfoにfacilityIdが入っている場合。
			 */
			facilityIdList.add(agentInfo.getFacilityId());
		} else {
			/*
			 * agentInfoにfacilityIdが入っていない場合。
			 * この場合は、ホスト名とIPアドレスからファシリティIDを決める。
			 */
			RepositoryControllerLocal local = null;
			try {
				local = RepositoryControllerUtil.getLocalHome().create();
			} catch(Exception e){
				m_log.warn("getFacilityId : " + e.getMessage());
				throw new HinemosUnknown("getFacilityId " + e.getMessage());
			}
			try {
				for (String ipAddress : agentInfo.getIpAddress()) {
					ArrayList<String> list = local.getFacilityIdList(agentInfo.getHostname(), ipAddress);
					if (list != null && list.size() != 0) {
						for (String facilityId : list) {
							m_log.debug("facilityId=" + facilityId + ", " + agentInfo.toString());
						}
						facilityIdList.addAll(list);
					}
				}
			} catch (Exception e) {
				m_log.error(e,e);
				throw new HinemosUnknown("getFacilityId " + e.getMessage());
			}
		}
		return facilityIdList;
	}

	/**
	 * [Job] ジョブの結果を送信する。
	 * 
	 * HinemosAgentAccess権限が必要
	 * 
	 * @param info
	 * @throws HinemosUnknown
	 * @throws InvalidRole
	 * @throws InvalidUserPass
	 */
	public void jobResult (RunResultInfo info)
	throws HinemosUnknown, JobInfoNotFound,
	InvalidUserPass, InvalidRole {
		m_log.info("jobResult : " +
				info.getSessionId() + ", " +
				info.getJobunitId() + ", " +
				info.getJobId() + ", " +
				info.getCommandType() + ", " +
				info.getStatus() + ", " +
				info.getFacilityId() + ", "
		);
		ArrayList<String> roleList = new ArrayList<String>();
		roleList.add(RoleConstant.HINEMOS_AGENT_ACCESS);
		HttpAuthenticator.authCheck(wsctx, roleList);

		JobRunManagementLocal local = getJobRunManagementLocal();
		local.endNode(info);
	}

	/**
	 * [Logfile] ログファイル監視の監視設定を取得
	 * 
	 * HinemosAgentAccess権限が必要
	 * 
	 * @param agentInfo
	 * @return
	 * @throws HinemosUnknown
	 * @throws InvalidRole
	 * @throws InvalidUserPass
	 * @throws MonitorNotFound
	 */
	public ArrayList<MonitorInfo> getMonitorLogfile (AgentInfo agentInfo) throws MonitorNotFound, HinemosUnknown, InvalidUserPass, InvalidRole
	{
		m_log.info("getMonitorLogFile : " + agentInfo);
		ArrayList<String> roleList = new ArrayList<String>();
		roleList.add(RoleConstant.HINEMOS_AGENT_ACCESS);
		HttpAuthenticator.authCheck(wsctx, roleList);
		ArrayList<String> facilityIdList = new ArrayList<String>();

		if (agentInfo.getFacilityId() != null && !agentInfo.getFacilityId().equals("")) {
			facilityIdList.add(agentInfo.getFacilityId());
		} else {
			try {
				RepositoryControllerLocal local = RepositoryControllerUtil.getLocalHome().create();
				for (String ipAddress : agentInfo.getIpAddress()) {
					ArrayList<String> list = local.getFacilityIdList(agentInfo.getHostname(), ipAddress);
					if (list != null && list.size() != 0) {
						for (String facilityId : list) {
							m_log.info("facilityId=" + facilityId + ", " + agentInfo.toString());
						}
						facilityIdList.addAll(list);
					}
				}
			} catch (Exception e) {
				m_log.error(e,e);
				return null;
			}
		}

		ArrayList<MonitorInfo> list = new ArrayList<MonitorInfo>();
		MonitorLogfileControllerLocal local = getMonitorLogfileControllerLocal();
		for (String facilityId : facilityIdList) {
			list.addAll(local.getLogfileListForFacilityId(facilityId, true));
		}
		return list;
	}

	/**
	 * [Logfile] ログファイル監視でマッチしたものに対して通知をマネージャに依頼する。
	 * 
	 * HinemosAgentAccess権限が必要
	 * 
	 * @throws HinemosUnknown
	 * @throws InvalidRole
	 * @throws InvalidUserPass
	 */
	public void sendNotify(String message, MessageInfo logmsg,
			MonitorInfo monitorInfo, MonitorStringValueInfo rule, AgentInfo agentInfo) throws HinemosUnknown, InvalidRole, InvalidUserPass {

		m_log.debug("sendNotify : " + message);
		ArrayList<String> roleList = new ArrayList<String>();
		roleList.add(RoleConstant.HINEMOS_AGENT_ACCESS);
		HttpAuthenticator.authCheck(wsctx, roleList);

		for (String logFacilityId : getFacilityId(agentInfo)) {
			String facilityId = monitorInfo.getFacilityId();
			ArrayList<String> facilityList = FacilitySelector.getFacilityIdList(facilityId, 0, false);
			String str = "";
			if (m_log.isDebugEnabled()) {
				for (String s : facilityList) {
					str += s + ", ";
				}
				m_log.debug("facilityId=" + logFacilityId + ", (" + str + ")");
			}
			if (facilityList.contains(logFacilityId)) {
				new LogDirectNotifier().put("", message, logmsg, monitorInfo, rule, logFacilityId);
			} else {
				m_log.debug("sendNotify facilityId=" + logFacilityId + ", " + str);
			}
		}
	}

	/**
	 * [Agent Base] 各設定の最終変更時刻情報を取得する。
	 * 
	 * HinemosAgentAccess権限が必要
	 * 
	 * @return 各設定の最終変更時刻情報
	 * @throws InvalidUserPass
	 * @throws InvalidRole
	 * @throws HinemosUnknown 予期せぬ内部エラーによりスローされる例外
	 */
	public SettingUpdateInfo getSettingUpdateInfo() throws InvalidUserPass, InvalidRole, HinemosUnknown {
		// MAIN
		ArrayList<String> roleList = new ArrayList<String>();
		roleList.add(RoleConstant.HINEMOS_AGENT_ACCESS);
		HttpAuthenticator.authCheck(wsctx, roleList);

		return SettingUpdateInfo.getInstance();
	}

	/**
	 * [Command Monitor] コマンド実行情報を問い合わせられた場合にコールされるメソッドであり、問い合わせたエージェントが実行すべきコマンド実行情報を返す。
	 * 
	 * HinemosAgentAccess権限が必要
	 * 
	 * @param agentInfo メソッドをコールしたエージェント情報
	 * @return コマンド実行情報の一覧
	 * @throws CustomInvalid コマンド監視設定に不整合が見つかった場合にスローされる例外
	 * @throws HinemosUnknown 予期せぬ内部エラーによりスローされる例外
	 * @throws InvalidRole
	 * @throws InvalidUserPass
	 */
	@SuppressWarnings("unchecked")
	public ArrayList<CommandExecuteDTO> getCommandExecuteDTO(AgentInfo agentInfo)
	throws CustomInvalid, HinemosUnknown,
	InvalidUserPass, InvalidRole
	{
		ArrayList<String> roleList = new ArrayList<String>();
		roleList.add(RoleConstant.HINEMOS_AGENT_ACCESS);
		HttpAuthenticator.authCheck(wsctx, roleList);

		// Local Variables
		MonitorCustomControllerLocal monitorCmdCtrl = null;
		ArrayList<CommandExecuteDTO> dtos = null;
		ArrayList<String> facilityIds = null;

		// MAIN
		facilityIds = getFacilityId(agentInfo);
		dtos = new ArrayList<CommandExecuteDTO>();
		try {
			monitorCmdCtrl = MonitorCustomControllerUtil.getLocalHome().create();
			for (String facilityId : facilityIds) {
				dtos.addAll(monitorCmdCtrl.getCommandExecuteDTO(facilityId));
			}
		} catch (CreateException e) {
			throw new HinemosUnknown("access failure to entity/session.", e);
		} catch (NamingException e) {
			throw new HinemosUnknown("access failure to naming service.", e);
		}

		return dtos;
	}

	/**
	 * [Command Monitor] コマンド監視において、コマンドの実行結果をマネージャに通知する。
	 * 
	 * HinemosAgentAccess権限が必要
	 * 
	 * @param result コマンドの実行結果
	 * @throws InvalidRole
	 * @throws InvalidUserPass
	 * @throws HinemosUnknown
	 */
	public void putCommandResultDTO(CommandResultDTO result) throws InvalidUserPass, InvalidRole, HinemosUnknown
	{
		ArrayList<String> roleList = new ArrayList<String>();
		roleList.add(RoleConstant.HINEMOS_AGENT_ACCESS);
		HttpAuthenticator.authCheck(wsctx, roleList);

		// Local Variables
		MonitorCustomControllerLocal monitorCmdCtrl = null;

		// MAIN
		try {
			monitorCmdCtrl = MonitorCustomControllerUtil.getLocalHome().create();
			monitorCmdCtrl.CustomResultDTO(result);
		} catch (MonitorNotFound e) {
			m_log.warn("monitor not found...", e);
		} catch (HinemosUnknown e) {
			m_log.warn("unexpected internal failure occurred...", e);
		} catch (CreateException e) {
			m_log.warn("unexpected internal failure occurred...", e);
		} catch (NamingException e) {
			m_log.warn("unexpected internal failure occurred...", e);
		} catch (CustomInvalid e) {
			m_log.warn("configuration of command monitor is not valid...", e);
		}
	}

	/**
	 * [Update] 新モジュールを取得
	 * 
	 * HinemosAgentAccess権限が必要
	 * 
	 * @throws HinemosUnknown
	 * @throws InvalidRole
	 * @throws InvalidUserPass
	 */
	@XmlMimeType("application/octet-stream")
	public DataHandler downloadModule(String filename) throws InvalidUserPass, InvalidRole, HinemosUnknown
	{
		ArrayList<String> roleList = new ArrayList<String>();
		roleList.add(RoleConstant.HINEMOS_AGENT_ACCESS);
		HttpAuthenticator.authCheck(wsctx, roleList);

		String filepath= "/opt/hinemos/lib/agent/" + filename;
		File file = new File(filepath);
		if(!file.exists()) {
			m_log.error("file not found : " + filepath);
			return null;
		}
		FileDataSource source = new FileDataSource(file);
		DataHandler dataHandler = new DataHandler(source);
		return dataHandler;
	}

	/**
	 * [Update] モジュール一覧を取得
	 * 
	 * HinemosAgentAccess権限が必要
	 * 
	 * @return
	 * @throws HinemosUnknown
	 * @throws InvalidRole
	 * @throws InvalidUserPass
	 */
	public ArrayList<String> getAgentLibMap() throws HinemosUnknown, InvalidRole, InvalidUserPass {
		RepositoryControllerLocal local = getRepositoryControllerLocal();

		ArrayList<String> roleList = new ArrayList<String>();
		roleList.add(RoleConstant.HINEMOS_AGENT_ACCESS);
		HttpAuthenticator.authCheck(wsctx, roleList);

		// TODO HashMapが利用できないのでArrayList<String>で実装。
		// あとで調査すること。
		HashMap<String, String> map = local.getAgentLibMap();
		ArrayList<String> ret = new ArrayList<String>();
		for (String str : map.keySet()) {
			ret.add(str);
			ret.add(map.get(str));
		}
		return ret;
	}

	/**
	 * [Update] ファイル名とMD5の組をマネージャに登録しておく。
	 * 
	 * HinemosAgentAccess権限が必要
	 * 
	 * TODO
	 * ArrayList<String>ではなく、HashMap<String, String>にしたい。
	 * 要調査。
	 * 
	 * @param filenameMd5
	 * @param agentInto
	 * @throws HinemosUnknown
	 * @throws InvalidRole
	 * @throws InvalidUserPass
	 */
	public void setAgentLibMd5(ArrayList<String> filenameMd5, AgentInfo agentInfo) throws InvalidUserPass, InvalidRole, HinemosUnknown {
		ArrayList<String> roleList = new ArrayList<String>();
		roleList.add(RoleConstant.HINEMOS_AGENT_ACCESS);
		HttpAuthenticator.authCheck(wsctx, roleList);

		HashMap<String, String> map = new HashMap<String, String>();
		Iterator<String> itr = filenameMd5.iterator();
		while(itr.hasNext()) {
			map.put(itr.next(), itr.next());
		}

		ArrayList<String> facilityIdList = getFacilityId(agentInfo);
		for (String facilityId : facilityIdList) {
			m_log.debug("setAgentLibMd5() : facilityId=" + facilityId);
			AgentConnectUtil.setAngetLibMd5(facilityId, map);
		}
	}
}
