/*

Copyright (C) 2006 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.

 */

package com.clustercontrol.util.apllog;


import java.text.MessageFormat;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.Map;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;


import com.clustercontrol.bean.PriorityConstant;
import com.clustercontrol.commons.util.HinemosProperties;
import com.clustercontrol.notify.bean.OutputBasicInfo;
import com.clustercontrol.notify.bean.QueueConstant;
import com.clustercontrol.notify.util.SendSyslog;




/**
 * 
 * Hinemosの内部ログ（HinemosApp.log）の出力を行うクラス<BR>
 * 
 * Hinemos内部で発生する事象をログやHinemosのイベントとして
 * 処理します。
 * 
 * AplLogger
 * @version $Revision:2230 $
 * @since
 */
public class AplLogger {


	private static final String INTERNAL_SCOPE="INTERNAL";
	private static final String INTERNAL_SCOPE_TEXT="Hinemos_Internal";

	private static final String PRIORITY_UNKNOWN = "unknown";
	private static final String PRIORITY_INFO = "info";
	private static final String PRIORITY_WARNING = "warning";
	private static final String PRIORITY_CRITICAL = "critical";
	private String m_pluginID;
	@SuppressWarnings("unused")
	private String m_aplID;
	private String m_aplName;
	private static SendQueue m_sendQueue = null;

	private static Map m_priorityMap = null;

	/** InternalEventのSyslog転送フラグ */
	private static boolean isSyslog = false;
	/** InternalEventのSyslog転送ホスト(複数指定) */
	private static String[] syslogHost = {"127.0.0.1"};
	/** InternalEventのSyslog転送ポート */
	private static int syslogPort = 514;
	/** InternalEventのSyslog転送Facility */
	private static String syslogFacility = "daemon";
	/** InternalEventのSyslog転送Severity */
	private static String syslogSeverity = "alert";
	/** InternalEventのSyslog転送用Utility */
	private SendSyslog sendSyslog = null;

	/** InternalEventのイベント出力フラグ */
	private static boolean isEvent = true;
	/** InternalEventのログファイルフラグ */
	private static boolean isFile = true;

	private static OutPutInfo syslogOutPutInfo = null;
	private static OutPutInfo eventOutPutInfo = null;
	private static OutPutInfo fileOutPutInfo = null;

	private static Log FILE_LOGGER = LogFactory.getLog("hinemos.apllog");
	private static Log log = LogFactory.getLog(AplLogger.class);

	/**
	 * 出力レベルを保持する
	 * 
	 * @author Hinemos
	 *
	 */
	private static class OutPutInfo{

		/** priorityを保持するマップ */
		private HashMap<String, String> priority = null;

		/**
		 * コンストラクタ
		 */
		public OutPutInfo() {
			priority = new HashMap<String, String>();
		}

		/**
		 * 送信を行うかのPriority毎の判定を設定する
		 * @param key unknown/info/warning/critical
		 * @param flg true/false(default true)
		 */
		public void setPriority(String key, String flg){
			priority.put(key, flg);
		}

		/**
		 * 送信を行うかのPriority毎の判定を行う
		 */
		public boolean isPut(String key){
			String flg = priority.get(key);

			if(flg == null || flg.equals("true")){
				return true;
			}
			else if(flg.equals("false")){
				return false;
			}
			else{
				return true;
			}
		}
	}

	/**
	 * static initializer
	 */
	static {
		// hinemos.propertiesから設定値の取得
		initialParameter();
	}

	/**
	 * hinemos.propertiesから設定値の取得
	 */
	private static void initialParameter(){

		try {
			syslogOutPutInfo = new OutPutInfo();
			eventOutPutInfo = new OutPutInfo();
			fileOutPutInfo = new OutPutInfo();



			/////
			// 設定値取得(common.internal.syslog)
			////

			// syslog転送有無:isSyslog
			String syslog = HinemosProperties.getProperty("common.internal.syslog");
			if(syslog != null && syslog.equals("true")){
				isSyslog = true;
			}
			// syslog転送ホスト(複数指定):syslogHost
			String hosts = HinemosProperties.getProperty("common.internal.syslog.host");
			syslogHost = hosts.split(",");
			// syslog転送ポート:syslogPort
			syslogPort = Integer.parseInt(HinemosProperties.getProperty("common.internal.syslog.port"));
			// syslog転送Facility:syslogFacility
			syslogFacility = HinemosProperties.getProperty("common.internal.syslog.facility");
			// syslog転送Severity:syslogSeverity
			syslogSeverity = HinemosProperties.getProperty("common.internal.syslog.severity");
			// syslog転送レベル
			syslogOutPutInfo.setPriority(PRIORITY_CRITICAL, HinemosProperties.getProperty("common.internal.syslog.priority.critical"));
			syslogOutPutInfo.setPriority(PRIORITY_WARNING, HinemosProperties.getProperty("common.internal.syslog.priority.warning"));
			syslogOutPutInfo.setPriority(PRIORITY_INFO, HinemosProperties.getProperty("common.internal.syslog.priority.info"));
			syslogOutPutInfo.setPriority(PRIORITY_UNKNOWN, HinemosProperties.getProperty("common.internal.syslog.priority.unknown"));

			/////
			// 設定値取得(common.internal.event)
			////
			// イベント通知有無:isEvent
			String event = HinemosProperties.getProperty("common.internal.event");
			if(event != null && event.equals("false")){
				isEvent = false;
			}
			// イベント通知レベル
			eventOutPutInfo.setPriority(PRIORITY_CRITICAL, HinemosProperties.getProperty("common.internal.event.priority.critical"));
			eventOutPutInfo.setPriority(PRIORITY_WARNING, HinemosProperties.getProperty("common.internal.event.priority.warning"));
			eventOutPutInfo.setPriority(PRIORITY_INFO, HinemosProperties.getProperty("common.internal.event.priority.info"));
			eventOutPutInfo.setPriority(PRIORITY_UNKNOWN, HinemosProperties.getProperty("common.internal.event.priority.unknown"));

			/////
			// 設定値取得(common.internal.file)
			////
			// ログファイル出力有無:isFile
			String file = HinemosProperties.getProperty("common.internal.file");
			if(file != null && file.equals("false")){
				isFile = false;
			}
			// ログファイル出力レベル
			fileOutPutInfo.setPriority(PRIORITY_CRITICAL, HinemosProperties.getProperty("common.internal.file.priority.critical"));
			fileOutPutInfo.setPriority(PRIORITY_WARNING, HinemosProperties.getProperty("common.internal.file.priority.warning"));
			fileOutPutInfo.setPriority(PRIORITY_INFO, HinemosProperties.getProperty("common.internal.file.priority.info"));
			fileOutPutInfo.setPriority(PRIORITY_UNKNOWN, HinemosProperties.getProperty("common.internal.file.priority.unknown"));

		} catch (Exception e) {
			log.error("initialParameter()",e);
		}

		// debug用出力
		if(log.isDebugEnabled()){
			log.debug("initialParameter() isSyslog = " + isSyslog);
			if(syslogHost != null && syslogHost.length > 0){
				for (int i = 0; i < syslogHost.length; i++) {
					log.debug("initialParameter() syslogHost = " + syslogHost[i]);
				}
			}
			log.debug("initialParameter() syslogPort = " + syslogPort);
			log.debug("initialParameter() syslogFacility = " + syslogFacility);
			log.debug("initialParameter() syslogSeverity = " + syslogSeverity);

			log.debug("initialParameter() isEvent = " + isEvent);
			log.debug("initialParameter() isFile = " + isFile);

			log.debug("initialParameter() syslog level(cri,warn,info,unknown) = (" + syslogOutPutInfo.isPut(PRIORITY_CRITICAL) +
					"," + syslogOutPutInfo.isPut(PRIORITY_WARNING) +
					"," + syslogOutPutInfo.isPut(PRIORITY_INFO) +
					"," + syslogOutPutInfo.isPut(PRIORITY_UNKNOWN) + ")");
			log.debug("initialParameter() event level(cri,warn,info,unknown) = (" + eventOutPutInfo.isPut(PRIORITY_CRITICAL) +
					"," + eventOutPutInfo.isPut(PRIORITY_WARNING) +
					"," + eventOutPutInfo.isPut(PRIORITY_INFO) +
					"," + eventOutPutInfo.isPut(PRIORITY_UNKNOWN) + ")");
			log.debug("initialParameter() file level(cri,warn,info,unknown) = (" + fileOutPutInfo.isPut(PRIORITY_CRITICAL) +
					"," + fileOutPutInfo.isPut(PRIORITY_WARNING) +
					"," + fileOutPutInfo.isPut(PRIORITY_INFO) +
					"," + fileOutPutInfo.isPut(PRIORITY_UNKNOWN) + ")");
		}
	}

	/**
	 * コンストラクタ<BR>
	 * 
	 * @param pluginID	プラグインID
	 * @param aplId		アプリケーションID
	 */
	public AplLogger(String pluginID, String aplId) {
		super();
		m_pluginID = pluginID;
		m_aplID = aplId;

		//メッセージ送信クラス初期化
		if( m_sendQueue == null ){
			m_sendQueue = new SendQueue(QueueConstant.QUEUE_NAME_EVENT);
		}

		//アプリケーション名取得
		m_aplName = PropertyUtil.getString(pluginID + "." + aplId);

		//重要度文字列取得
		if( m_priorityMap == null ){
			m_priorityMap = Collections.synchronizedMap(new HashMap());

			m_priorityMap.put(PRIORITY_CRITICAL, PropertyUtil.getString("priority.critical"));
			m_priorityMap.put(PRIORITY_WARNING, PropertyUtil.getString("priority.warning"));
			m_priorityMap.put(PRIORITY_INFO, PropertyUtil.getString("priority.info"));
			m_priorityMap.put(PRIORITY_UNKNOWN, PropertyUtil.getString("priority.unknown"));

		}

		// Syslog送信用クラスの初期化
		sendSyslog = new SendSyslog();
	}

	/**
	 * メインクラス
	 * @param args
	 */
	public static void main(String[] args) {
		AplLogger apllog = new AplLogger("REP","rep");
		apllog.put("SYS","001");
		apllog.put("USR","001");
		apllog.put("USR","002");
		apllog.put("USR","003");

	}

	/**
	 * 
	 * ログを出力します。<BR>
	 * 
	 * @param moniterID		監視項目ID
	 * @param msgID			メッセージID
	 * @param msgArgs		メッセージ置換項目
	 * @param detailMsg		詳細メッセージ
	 * @since
	 */
	public void put(String moniterID, String msgID, Object[] msgArgs, String detailMsg) {


		//現在日時取得
		Date nowDate = new Date();

		//監視項目IDとプラグインIDでイベントメッセージ（リソースファイル）から
		StringBuffer keyBase = new StringBuffer();
		keyBase.append(m_pluginID);
		keyBase.append(".");
		keyBase.append(moniterID);
		keyBase.append(".");
		keyBase.append(msgID);

		//重要度を取得
		StringBuffer keyPriority = new StringBuffer();
		keyPriority.append(keyBase);
		keyPriority.append(".");
		keyPriority.append("priority");

		String priority = PropertyUtil.getString(keyPriority.toString());

		//メッセージを取得する。
		StringBuffer keyMsg = new StringBuffer();
		keyMsg.append(keyBase);
		keyMsg.append(".");
		keyMsg.append("message");

		//メッセージ項目値が指定されている場合、メッセージの項目を置換
		String msg = null;
		if(msgArgs != null && msgArgs.length != 0){
			msg = PropertyUtil.getString(keyMsg.toString(),msgArgs);
		}else{
			msg = PropertyUtil.getString(keyMsg.toString());
		}

		//メッセージ出力
		if( isEvent && eventOutPutInfo.isPut(priority) ){

			if( !putMsg(nowDate,moniterID,msgID,getPriority(priority),msg,detailMsg) ){
				//メッセージ送信に失敗したら、ファイル出力
				isFile = true;
			}
		}

		//ファイル出力
		if( isFile && fileOutPutInfo.isPut(priority)){
			String priorityStr = (String)m_priorityMap.get(priority);
			if( priorityStr == null ){
				priorityStr = (String)m_priorityMap.get(PRIORITY_UNKNOWN);
			}
			putFile(nowDate,moniterID,msgID,priorityStr,msg,detailMsg);
		}

		//Syslog出力
		if ( isSyslog && syslogOutPutInfo.isPut(priority)){
			String priorityStr = (String)m_priorityMap.get(priority);
			if( priorityStr == null ){
				priorityStr = (String)m_priorityMap.get(PRIORITY_UNKNOWN);
			}
			putSyslog(moniterID,msgID,priorityStr,msg,detailMsg);
		}
	}
	/**
	 * ログを出力します。<BR>
	 * 
	 * @param moniterID		監視項目ID
	 * @param msgID			メッセージID
	 * @since
	 */
	public void put(String moniterID, String msgID) {

		put(moniterID, msgID,null,null);
	}
	/**
	 * 
	 * ログを出力します。<BR>
	 * @param moniterID		監視項目ID
	 * @param msgID			メッセージID
	 * @param msgArgs		メッセージ置換項目
	 * @since
	 */
	public void put(String moniterID, String msgID, Object[] msgArgs) {
		put(moniterID, msgID,msgArgs,null);

	}

	/**
	 * ログファイルへ出力します。<BR>
	 * ファイルへのログを出力します。
	 * 
	 * @param genDate		日時
	 * @param monitorId		監視項目ID
	 * @param msgId			メッセージID
	 * @param priority		重要度文字列
	 * @param msg			メッセージ
	 * @param detailMsg		詳細メッセージ
	 * @since
	 */
	public void putFile(Date genDate,String monitorId, String msgId, String priority, String msg, String detailMsg ) {
		if(log.isDebugEnabled()){
			log.debug("putFile() genDate = " + genDate.toString() +
					", monitorId = " + monitorId + ", msgId = " + msgId +
					", priority = " + priority + ", detailMsg = " + detailMsg);
		}

		/**	ログファイル出力用フォーマット「日付  プラグインID,アプリケーション,監視項目ID,メッセージID,ファシリティID,メッセージ,詳細メッセージ」 */
		MessageFormat logfmt = new MessageFormat("{0,date,yyyy/MM/dd HH:mm:ss}  {1},{2},{3},{4},{5},{6}");

		//メッセージを編集
		Object[] args ={genDate,m_pluginID, m_aplName, monitorId, msgId, priority, msg, detailMsg};
		String logmsg = logfmt.format(args);

		//ファイル出力
		if(log.isDebugEnabled()){
			log.debug("putFile() logmsg = " + logmsg);
		}
		FILE_LOGGER.info(logmsg);

	}

	/**
	 * syslogへ出力します。<BR>
	 * 
	 * @param genDate		日時
	 * @param monitorId		監視項目ID
	 * @param msgId			メッセージID
	 * @param priority		重要度文字列
	 * @param msg			メッセージ
	 * @param detailMsg		詳細メッセージ
	 */
	private void putSyslog(String monitorId, String msgId, String priority, String msg, String detailMsg ) {
		if(log.isDebugEnabled()){
			log.debug("putSyslog() monitorId = " + monitorId +
					", msgId = " + msgId + ", priority = " + priority +
					", msg = " + msg + ", detailMsg = " + detailMsg);
		}

		/**	syslog出力用フォーマット「日付  プラグインID,アプリケーション,監視項目ID,メッセージID,ファシリティID,メッセージ,詳細メッセージ」 */
		MessageFormat syslogfmt = new MessageFormat("hinemos: {0},{1},{2},{3},{4},{5}");

		// メッセージを編集
		Object[] args ={m_pluginID, m_aplName, monitorId, msgId, priority, msg, detailMsg};
		String logmsg = syslogfmt.format(args);

		// メッセージをsyslog送信
		if(syslogHost != null && syslogHost.length > 0){
			for (int i = 0; i < syslogHost.length; i++) {
				if(log.isDebugEnabled()){
					log.debug("putSyslog() syslogHost = " + syslogHost + ", syslogPort = " + syslogPort +
							", syslogFacility = " + syslogFacility + ", syslogSeverity = " + syslogSeverity +
							", logmsg = " + logmsg);
				}

				// syslog送信
				sendSyslog.sendAfterConvertHostname(syslogHost[i], syslogPort, syslogFacility, syslogSeverity, INTERNAL_SCOPE, logmsg);
			}
		}
	}

	/**
	 * イベントを出力します。<BR>
	 * 
	 * 事象をメッセージ情報クラスにセットして、メッセージ送信します。<BR>
	 * 
	 * @param genDate		日時
	 * @param monitorId		監視項目ID
	 * @param msgId			メッセージID
	 * @param priority		重要度
	 * @param msg			メッセージ
	 * @param detailMsg		詳細メッセージ
	 * @return				true:成功/false：失敗
	 * @since
	 */
	private boolean putMsg(Date genDate,String monitorId, String msgId, int priority, String msg, String detailMsg) {
		if(log.isDebugEnabled()){
			log.debug("putMsg() monitorId = " + monitorId +
					", msgId = " + msgId + ", priority = " + priority +
					", msg = " + msg + ", detailMsg = " + detailMsg);
		}

		//    	ログ出力情報
		//    	・プラグインID			:コンストラクタ
		//    	・監視項目ID			:コンストラクタ
		//    	・ファシリティID		:SYSTEM 固定
		//    	・スコープ				:なし？
		//    	・アプリケーション		:コンストラクタ、リソースファイル
		//    	・メッセージID			:パラメータ
		//    	・メッセージ			:パラメータ、リソースファイル（重要度を除く）
		//    	・オリジナルメッセージ	:パラメータ
		//    	・重要度				:パラメータ、リソースファイル
		//    	・発生日時				:現在日時※メソッド呼び出し時に取得したもの
		//    	・出力日時				:現在日時※イベント出力直前に取得したもの

		//メッセージ情報作成
		OutputBasicInfo output = new OutputBasicInfo();
		output.setPluginId(m_pluginID);
		output.setMonitorId(monitorId);
		output.setFacilityId(INTERNAL_SCOPE);
		output.setScopeText(INTERNAL_SCOPE_TEXT);
		output.setApplication(m_aplName);
		output.setMessageId(msgId);
		output.setMessage(msg);
		output.setMessageOrg(detailMsg == null ? "":detailMsg);
		output.setPriority(priority);
		if (genDate != null) {
			output.setGenerationDate(genDate.getTime());
		}

		//メッセージ送信
		return m_sendQueue.put(output);
	}

	/**
	 * 
	 * 文字列から、Priority区分を取得します。<BR>
	 * 
	 * @param priority
	 * @since
	 */
	private int getPriority(String priority) {

		int ret = PriorityConstant.TYPE_UNKNOWN;

		if(priority.equals(PRIORITY_CRITICAL)){
			ret = PriorityConstant.TYPE_CRITICAL;
		}else if(priority.equals(PRIORITY_WARNING)){
			ret = PriorityConstant.TYPE_WARNING;
		}else if(priority.equals(PRIORITY_INFO)){
			ret = PriorityConstant.TYPE_INFO;
		}

		return ret;

	}
}
