package com.clustercontrol.performance.monitor.factory;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;

import javax.ejb.CreateException;
import javax.ejb.FinderException;
import javax.naming.InitialContext;
import javax.naming.NamingException;
import javax.rmi.PortableRemoteObject;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.fault.FacilityNotFound;
import com.clustercontrol.bean.HinemosModuleConstant;
import com.clustercontrol.monitor.run.bean.MonitorInfo;
import com.clustercontrol.performance.monitor.bean.PerfCheckInfo;
import com.clustercontrol.performance.util.PollingDataManager;
import com.clustercontrol.poller.CollectorAlreadyExistException;
import com.clustercontrol.poller.NotInitializedException;
import com.clustercontrol.poller.PollerManager;
import com.clustercontrol.poller.PollingController;
import com.clustercontrol.repository.ejb.session.RepositoryControllerBean;
import com.clustercontrol.repository.ejb.session.RepositoryControllerLocal;
import com.clustercontrol.repository.ejb.session.RepositoryControllerUtil;
import com.clustercontrol.sharedtable.DataTableNotFoundException;
import com.clustercontrol.sharedtable.SharedTable;
import com.clustercontrol.util.apllog.AplLogger;

public class ModifyPollingSchedule {

	private static Log m_log = LogFactory.getLog( ModifyPollingSchedule.class );

	private SharedTable table = null;

	/**
	 * スケジュールに登録します。
	 * 
	 * @param monitorTypeId
	 * @param monitorId
	 * @param parentFacilityId
	 * @param interval
	 * @return
	 */
	public boolean addSchedule(MonitorInfo info) {
		m_log.debug("addSchedule() monitorId = " + info.getMonitorId() + ", parentFacilityId = " + info.getFacilityId());

		String monitorTypeId = info.getMonitorTypeId();
		String monitorId = info.getMonitorId();
		String parentFacilityId = info.getFacilityId();
		int interval = info.getRunInterval();

		if(parentFacilityId != null && !"".equals(parentFacilityId)){

			try{
				// ファシリティIDの配下全ての一覧を取得
				RepositoryControllerLocal repository = RepositoryControllerUtil.getLocalHome().create();
				ArrayList<String> facilityList = repository.getExecTargetFacilityIdList(parentFacilityId);

				if(m_log.isDebugEnabled()){
					for (String facilityId : facilityList) {
						m_log.debug("addSchedule() add schedule target facilityId = " + facilityId);
					}
				}

				// ファシリティ毎に登録
				boolean ret = true;
				for(String facilityId : facilityList){
					m_log.debug("addSchedule() target facilityId = " + facilityId);

					if(facilityId != null && !"".equals(facilityId)){
						ret = ret & addNodeSchedule(facilityId, monitorId, monitorTypeId, interval, info.getPerfCheckInfo());
					}
				}
				return ret;

			}catch(CreateException e){
				AplLogger apllog = new AplLogger("PERF", "perf");
				String[] args = { parentFacilityId };
				apllog.put("SYS", "014", args);
				m_log.error("addSchedule():" + e.getMessage(), e);
			}catch(NamingException e){
				AplLogger apllog = new AplLogger("PERF", "perf");
				String[] args = { parentFacilityId };
				apllog.put("SYS", "014", args);
				m_log.error("addSchedule():" + e.getMessage(), e);
			}catch(FacilityNotFound e){
				AplLogger apllog = new AplLogger("PERF", "perf");
				String[] args = { parentFacilityId };
				apllog.put("SYS", "014", args);
				m_log.error("addSchedule():" + e.getMessage(), e);
			}catch(FinderException e){
				AplLogger apllog = new AplLogger("PERF", "perf");
				String[] args = { parentFacilityId };
				apllog.put("SYS", "014", args);
				m_log.error("addSchedule():" + e.getMessage(), e);
			}
		}
		return false;

	}

	/**
	 * ノード別のポーラを登録する
	 * 
	 * @param facilityId
	 * @param monitorId
	 * @param monitorTypeId
	 * @param interval
	 * @param perfCheckInfo
	 * @return
	 * @throws NamingException
	 * @throws FacilityNotFound
	 * @throws CreateException
	 * @throws FinderException
	 */
	public boolean addNodeSchedule(String facilityId, String monitorId, String monitorTypeId, int interval, PerfCheckInfo perfCheckInfo)
	throws NamingException, FacilityNotFound, CreateException, FinderException{
		m_log.debug("addNodeSchedule() facilityId = " + facilityId + ", monitorId = " + monitorId + ", monitorTypeId = " + monitorTypeId + ", interval = " + interval);

		// 収集値の共有テーブルをルックアップ
		SharedTable sst = getSharedTable();

		// テーブル生成
		// テーブルが存在しない場合は生成する
		// 一部の項目で時系列情報が必要なためページサイズは2を指定する
		if(sst.containsTable(HinemosModuleConstant.PERFORMANCE, facilityId, 2) == false){
			sst.createDataTable(HinemosModuleConstant.PERFORMANCE, facilityId, 2);
		}

		// ポーラを生成する
		m_log.debug("addSchedule() lookup");
		InitialContext ctx = new InitialContext();
		Object obj = ctx.lookup(PollingController.POLLER_MANAGER_JNDI_NAME);
		PollerManager manager =
			(PollerManager)PortableRemoteObject.narrow(obj, PollerManager.class);

		// ポーリングを開始する
		try {
			// ポーラを取得
			PollingController poller = manager.getPoller(monitorTypeId, facilityId);
			// ない場合は生成する
			if(poller == null){
				m_log.debug("addSchedule() create poller");
				poller = manager.createPoller(monitorTypeId, facilityId, false, HinemosModuleConstant.PERFORMANCE, facilityId);
			}
			// ある場合はTableHolderにアクセスして最終参照日時を更新する
			else{
				try{
					sst.getLastDataTables(HinemosModuleConstant.PERFORMANCE, facilityId, monitorId, 2);
					m_log.info("addSchedule() update last reference date : " + HinemosModuleConstant.PERFORMANCE + ", " + facilityId + ", " + monitorId);
				} catch (NotInitializedException e) {
					m_log.debug("addSchedule()", e);
				} catch (DataTableNotFoundException e) {
					m_log.debug("addSchedule()", e);
				}
			}

			// リポジトリ,DBから設定情報を取得する(プラットフォームID, サブプラットフォームID, 収集方法など)
			if(perfCheckInfo == null){
				m_log.error("addSchedule() perfCheckInfo is null");
				return false;
			}
			PollingDataManager dataManager =
				new PollingDataManager(facilityId,
						perfCheckInfo.getItemCode(),
						perfCheckInfo.isBreakdown());

			String collectMethod = dataManager.getCollectMethod();
			List<String> pollingTargets = dataManager.getPollingTargets();

			// 取得したプラットフォームにおいて、リソース監視で何が使われているかを取得
			HashMap<String, List<String>> map = new HashMap<String, List<String>>();
			map.put(collectMethod, pollingTargets);

			m_log.debug("addSchedule() start poller");
			poller.startPolling(monitorId, interval, map);
		} catch (NotInitializedException e) {
			m_log.error(e.getMessage(), e);
			return false;
		} catch (DataTableNotFoundException e) {
			m_log.error(e.getMessage(), e);
			return false;
		} catch (CollectorAlreadyExistException e) {
			m_log.error(e.getMessage(), e);
			return false;
		}

		return true;
	}

	/**
	 * スケジュールの登録を削除する
	 * 
	 * @param monitorTypeId
	 * @param monitorId
	 * @param parentFacilityId
	 * @return
	 */
	public boolean deleteSchedule(String monitorTypeId, String monitorId, String parentFacilityId) {
		m_log.debug("deleteSchedule() monitorId = " + monitorId);

		// ポーラの削除
		if(parentFacilityId != null && !"".equals(parentFacilityId)){

			try{
				// SNMP収集値の共有テーブルをルックアップ
				SharedTable sst = getSharedTable();

				// ファシリティIDの配下全ての一覧を取得
				RepositoryControllerLocal repository = RepositoryControllerUtil.getLocalHome().create();
				ArrayList facilityList = repository.getNodeFacilityIdList(parentFacilityId, RepositoryControllerBean.ALL);

				//配下にノードがないということはノードの可能性があるので指定されたIDをセット
				if (facilityList.size() == 0) {
					if(repository.isNode(parentFacilityId) == true){
						facilityList.add(parentFacilityId);
					}
				}

				// ノード毎にポーリングを停止する
				String facilityId = null;
				for(int index=0; index<facilityList.size(); index++){
					facilityId = (String)facilityList.get(index);

					// ポーラを生成する
					InitialContext ctx = new InitialContext();
					Object obj = ctx.lookup(PollingController.POLLER_MANAGER_JNDI_NAME);
					PollerManager manager =
						(PollerManager)PortableRemoteObject.narrow(obj, PollerManager.class);

					// ポーリングを停止する
					PollingController poller = manager.getPoller(monitorTypeId, facilityId);
					if(poller != null){
						poller.stopPolling(monitorId);
					}

					// テーブルから参照登録を削除する
					obj = ctx.lookup(PollingController.SHARED_TABLE_JNDI_NAME);
					SharedTable sharedTable =
						(SharedTable)PortableRemoteObject.narrow(obj, SharedTable.class);

					try {
						sharedTable.unregisterCollector(HinemosModuleConstant.PERFORMANCE, facilityId, monitorId);
					} catch (DataTableNotFoundException e) {
						m_log.warn(e.getMessage());
					}
				}
				return true;

			} catch (FacilityNotFound e) {
				AplLogger apllog = new AplLogger("PERF", "perf");
				String[] args = { parentFacilityId };
				apllog.put("SYS", "015", args);
				m_log.debug("deleteSchedule():" + e.getMessage(), e);
			}catch(CreateException e){
				AplLogger apllog = new AplLogger("PERF", "perf");
				String[] args = { parentFacilityId };
				apllog.put("SYS", "015", args);
				m_log.debug("deleteSchedule():" + e.getMessage(), e);
			}catch(NamingException e){
				AplLogger apllog = new AplLogger("PERF", "perf");
				String[] args = { parentFacilityId };
				apllog.put("SYS", "015", args);
				m_log.debug("deleteSchedule():" + e.getMessage(), e);
			}
		}
		return false;
	}

	/**
	 * 収集値の共有テーブルをルックアップする
	 * @return
	 * @throws NamingException
	 */
	private SharedTable getSharedTable() throws NamingException {
		m_log.debug("SharedTable()");
		if(table != null){
			m_log.debug("SharedTable() table already exist!");
			return table;
		}

		try {
			InitialContext ctx = new InitialContext();

			// SNMP収集値の共有テーブルをルックアップ
			table = (SharedTable)ctx.lookup(PollingController.SHARED_TABLE_JNDI_NAME);
			return table;
		} catch (NamingException e) {
			m_log.error("getSharedTable:" + e.getMessage());
			throw e;
		}
	}
}
