/*

Copyright (C) 2006 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.

 */

package com.clustercontrol.notify.factory;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.Iterator;

import javax.ejb.CreateException;
import javax.ejb.FinderException;
import javax.naming.NamingException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.fault.HinemosUnknown;
import com.clustercontrol.fault.NotifyNotFound;
import com.clustercontrol.bean.HinemosModuleConstant;
import com.clustercontrol.notify.bean.NotifyCommandInfo;
import com.clustercontrol.notify.bean.NotifyEventInfo;
import com.clustercontrol.notify.bean.NotifyInfo;
import com.clustercontrol.notify.bean.NotifyJobInfo;
import com.clustercontrol.notify.bean.NotifyLogEscalateInfo;
import com.clustercontrol.notify.bean.NotifyMailInfo;
import com.clustercontrol.notify.bean.NotifyStatusInfo;
import com.clustercontrol.notify.ejb.entity.NotifyCommandInfoLocal;
import com.clustercontrol.notify.ejb.entity.NotifyEventInfoLocal;
import com.clustercontrol.notify.ejb.entity.NotifyInfoLocal;
import com.clustercontrol.notify.ejb.entity.NotifyInfoUtil;
import com.clustercontrol.notify.ejb.entity.NotifyJobInfoLocal;
import com.clustercontrol.notify.ejb.entity.NotifyLogEscalateInfoLocal;
import com.clustercontrol.notify.ejb.entity.NotifyMailInfoLocal;
import com.clustercontrol.notify.ejb.entity.NotifyStatusInfoLocal;
import com.clustercontrol.repository.ejb.session.RepositoryControllerLocal;
import com.clustercontrol.repository.ejb.session.RepositoryControllerUtil;
import com.clustercontrol.util.apllog.AplLogger;

/**
 * 通知情報を検索するクラスです。
 *
 * @version 3.0.0
 * @since 1.0.0
 */
public class SelectNotify {

	/** ログ出力のインスタンス。 */
	private static Log m_log = LogFactory.getLog( SelectNotify.class );

	/**
	 * 通知情報を返します。
	 * 
	 * @param notifyId 取得対象の通知ID
	 * @return 通知情報
	 * @throws HinemosUnknown
	 * @throws FinderException
	 * 
	 * @see com.clustercontrol.notify.ejb.entity.NotifyInfoBean
	 * @see com.clustercontrol.notify.ejb.entity.NotifyEventInfoBean
	 */
	public NotifyInfo getNotify(String notifyId) throws NotifyNotFound, HinemosUnknown {
		// 通知情報を取得
		NotifyInfo info = null;
		try{
			NotifyInfoLocal notify = NotifyInfoUtil.getLocalHome().findByPrimaryKey(notifyId);
			info = getNotify(notify);
		} catch (FinderException e){
			AplLogger apllog = new AplLogger("NOTIFY", "notify");
			String[] args = { notifyId };
			apllog.put("SYS", "003", args);
			m_log.debug("delete():" + e.getMessage());
			throw new NotifyNotFound(e.getMessage(), e);
		} catch (NamingException e) {
			AplLogger apllog = new AplLogger("NOTIFY", "notify");
			String[] args = { notifyId };
			apllog.put("SYS", "003", args);
			m_log.debug("delete():" + e.getMessage());
			throw new HinemosUnknown(e.getMessage(), e);
		}

		return info;
	}

	private NotifyInfo getNotify(NotifyInfoLocal notify) throws NamingException {

		NotifyInfo bean = null;
		String notifyId = notify.getNotifyId();

		ArrayList<NotifyCommandInfo> notifyCommands = getNotifyCommand(notifyId,notify);
		ArrayList<NotifyEventInfo> notifyEvents = getNotifyEvent(notifyId,notify);
		ArrayList<NotifyJobInfo> notifyJobs = getNotifyJob(notifyId,notify);
		ArrayList<NotifyLogEscalateInfo> notifyLogEscalates = getNotifyLogEscalate(notifyId,notify);
		ArrayList<NotifyMailInfo> notifyMails = getNotifyMail(notifyId,notify);
		ArrayList<NotifyStatusInfo> notifyStatuses = getNotifyStatus(notifyId,notify);
		Long regDate = notify.getRegDate() == null ? null:notify.getRegDate().getTime();
		Long updateDate = notify.getUpdateDate() == null ? null:notify.getUpdateDate().getTime();

		bean = new NotifyInfo(
				notify.getNotifyId(),
				notify.getDescription(),
				notify.getNotifyType(),
				notify.getInitialCount(),
				notify.getRenotifyType(),
				notify.getRenotifyPeriod(),
				regDate,
				updateDate,
				notify.getRegUser(),
				notify.getUpdateUser(),
				notify.getValidFlg(),
				notifyCommands,
				notifyEvents,
				notifyJobs,
				notifyLogEscalates,
				notifyMails,
				notifyStatuses
		);
		return bean;
	}

	/**
	 * コマンド通知詳細情報を取得します。
	 * 
	 * @param notifyId
	 * @param notify
	 * @return
	 */
	private ArrayList<NotifyCommandInfo> getNotifyCommand(String notifyId, NotifyInfoLocal notify){
		ArrayList<NotifyCommandInfo> notifyEvents = new ArrayList<NotifyCommandInfo>();
		Collection<NotifyCommandInfoLocal> ct = notify.getNotifyCommandInfo();

		for (NotifyCommandInfoLocal local : ct){
			notifyEvents.add(local.getData());
		}
		return notifyEvents;
	}

	/**
	 * イベント通知詳細情報を取得します。
	 * 
	 * @param notifyId
	 * @param notify
	 * @return
	 */
	private ArrayList<NotifyEventInfo> getNotifyEvent(String notifyId, NotifyInfoLocal notify){
		ArrayList<NotifyEventInfo> notifyEvents = new ArrayList<NotifyEventInfo>();
		Collection<NotifyEventInfoLocal> ct = notify.getNotifyEventInfo();

		Iterator<NotifyEventInfoLocal> itr = ct.iterator();
		NotifyEventInfoLocal local = null;
		while(itr.hasNext()){
			local = itr.next();
			NotifyEventInfo event = new NotifyEventInfo(
					notifyId,
					local.getPriority(),
					local.getValidFlg(),
					local.getEventNormalState());

			notifyEvents.add(event);
		}
		return notifyEvents;
	}

	/**
	 * ジョブ通知詳細情報を取得します。
	 * 
	 * @param notifyId
	 * @param notify
	 * @return
	 */
	private ArrayList<NotifyJobInfo> getNotifyJob(String notifyId, NotifyInfoLocal notify){
		ArrayList<NotifyJobInfo> notifyEvents = new ArrayList<NotifyJobInfo>();
		Collection<NotifyJobInfoLocal> ct = notify.getNotifyJobInfo();
		RepositoryControllerLocal repository = null;;
		try {
			repository = RepositoryControllerUtil.getLocalHome().create();
			Iterator<NotifyJobInfoLocal> itr = ct.iterator();
			NotifyJobInfoLocal local = null;
			while(itr.hasNext()){
				local = itr.next();

				String facilityPath = repository.getFacilityPath(local.getJobExecFacility(), null);

				NotifyJobInfo status = new NotifyJobInfo(
						notifyId,
						local.getPriority(),
						local.getValidFlg(),
						local.getJobunitId(),
						local.getJobId(),
						local.getJobFailurePriority(),
						local.getJobExecFacilityFlg(),
						local.getJobExecFacility(),
						facilityPath);

				notifyEvents.add(status);
			}
		} catch (CreateException e) {
			m_log.error("getNotifyJob() : " + e.getMessage(), e);
		} catch (NamingException e) {
			m_log.error("getNotifyJob() : " + e.getMessage(), e);
		}
		return notifyEvents;
	}

	/**
	 * ログエスカレーション通知詳細情報を取得します。
	 * 
	 * @param notifyId
	 * @param notify
	 * @return
	 */
	private ArrayList<NotifyLogEscalateInfo> getNotifyLogEscalate(String notifyId, NotifyInfoLocal notify){
		ArrayList<NotifyLogEscalateInfo> notifyEvents = new ArrayList<NotifyLogEscalateInfo>();
		Collection<NotifyLogEscalateInfoLocal> ct = notify.getNotifyLogEscalateInfo();
		RepositoryControllerLocal repository = null;;
		try {
			repository = RepositoryControllerUtil.getLocalHome().create();
			Iterator<NotifyLogEscalateInfoLocal> itr = ct.iterator();
			NotifyLogEscalateInfoLocal local = null;
			while(itr.hasNext()){
				local = itr.next();

				String facilityPath = repository.getFacilityPath(local.getEscalateFacility(), null);

				NotifyLogEscalateInfo status = new NotifyLogEscalateInfo(
						notifyId,
						local.getPriority(),
						local.getValidFlg(),
						local.getEscalateMessage(),
						local.getSyslogPriority(),
						local.getSyslogFacility(),
						local.getEscalateFacilityFlg(),
						local.getEscalateFacility(),
						facilityPath,
						local.getEscalatePort());

				notifyEvents.add(status);
			}
		} catch (CreateException e) {
			m_log.error("getNotifyLogEscalate() : " + e.getMessage(), e);
		} catch (NamingException e) {
			m_log.error("getNotifyLogEscalate() : " + e.getMessage(), e);
		}
		return notifyEvents;
	}

	/**
	 * メール通知詳細情報を取得します。
	 * 
	 * @param notifyId
	 * @param notify
	 * @return
	 */
	private ArrayList<NotifyMailInfo> getNotifyMail(String notifyId, NotifyInfoLocal notify){
		ArrayList<NotifyMailInfo> notifyEvents = new ArrayList<NotifyMailInfo>();
		Collection<NotifyMailInfoLocal> ct = notify.getNotifyMailInfo();

		Iterator<NotifyMailInfoLocal> itr = ct.iterator();
		NotifyMailInfoLocal local = null;
		while(itr.hasNext()){
			local = itr.next();
			NotifyMailInfo status = new NotifyMailInfo(
					notifyId,
					local.getPriority(),
					local.getMailTemplateId(),
					local.getValidFlg(),
					local.getMailAddress());

			notifyEvents.add(status);
		}
		return notifyEvents;
	}

	/**
	 * ステータス通知詳細情報を取得します。
	 * 
	 * @param notifyId
	 * @param notify
	 * @return
	 */
	private ArrayList<NotifyStatusInfo> getNotifyStatus(String notifyId, NotifyInfoLocal notify){
		ArrayList<NotifyStatusInfo> notifyEvents = new ArrayList<NotifyStatusInfo>();
		Collection<NotifyStatusInfoLocal> ct = notify.getNotifyStatusInfo();

		Iterator<NotifyStatusInfoLocal> itr = ct.iterator();
		NotifyStatusInfoLocal local = null;
		while(itr.hasNext()){
			local = itr.next();
			NotifyStatusInfo status = new NotifyStatusInfo(
					notifyId,
					local.getPriority(),
					local.getValidFlg(),
					local.getStatusInvalidFlg(),
					local.getStatusUpdatePriority(),
					local.getStatusValidPeriod());

			notifyEvents.add(status);
		}
		return notifyEvents;
	}

	/**
	 * 通知情報一覧を返します(障害検知用通知を除く)。
	 * <p>
	 * <ol>
	 * <li>通知IDの昇順に並んだ全ての通知情報を取得します。</li>
	 * <li>１通知情報をテーブルのカラム順（{@link com.clustercontrol.notify.bean.NotifyTableDefine}）に、リスト（{@link ArrayList}）にセットします。</li>
	 * <li>この１通知情報を保持するリストを、通知情報一覧を保持するリスト（{@link ArrayList}）に格納し返します。<BR>
	 *  <dl>
	 *  <dt>通知情報一覧（Objectの2次元配列）</dt>
	 *  <dd>{ 通知情報1 {カラム1の値, カラム2の値, … }, 通知情報2{カラム1の値, カラム2の値, …}, … }</dd>
	 *  </dl>
	 * </li>
	 * </ol>
	 * 
	 * @return 通知情報一覧（Objectの2次元配列）
	 * @throws NotifyNotFound
	 * @throws HinemosUnknown
	 * 
	 * @see com.clustercontrol.notify.bean.NotifyTableDefine
	 * @see #collectionToArray(Collection)
	 */
	public ArrayList<NotifyInfo> getNotifyList() throws NotifyNotFound, HinemosUnknown {
		m_log.debug("getNotifyList() : start");
		ArrayList<NotifyInfo> list = new ArrayList<NotifyInfo>();
		try
		{
			// 通知情報一覧を取得
			Collection<NotifyInfoLocal> ct = NotifyInfoUtil.getLocalHome().findAllOrderByNotifyId();

			// システム機能（障害検知、セルフチェック）の通知IDを除く
			for(NotifyInfoLocal notify : ct){
				String regex1 = "^" + HinemosModuleConstant.SYSYTEM_TROUBLE_DETECTION + ".*";
				String regex2 = "^" + HinemosModuleConstant.SYSYTEM_SELFCHECK + ".*";

				// FIXME : 文字列の比較による障害検知用通知IDの特定方法を変更する
				// 結果用Collectionから削除対象以外のものをリストに追加する
				if(! notify.getNotifyId().matches(regex1) && ! notify.getNotifyId().matches(regex2)){
					list.add(getNotify(notify.getNotifyId()));
				}
			}
		} catch (FinderException e) {
			AplLogger apllog = new AplLogger("NOTIFY", "notify");
			apllog.put("SYS", "006");
			m_log.debug("getNotifyList():" + e.getMessage());
			throw new NotifyNotFound(e.getMessage(),e);
		} catch (NamingException e) {
			AplLogger apllog = new AplLogger("NOTIFY", "notify");
			apllog.put("SYS", "006");
			m_log.debug("getNotifyList():" + e.getMessage());
			throw new HinemosUnknown(e.getMessage(), e);
		}
		return list;
	}

	/**
	 * 通知情報マップを返します。
	 * <p>
	 * <ol>
	 * <li>全ての通知情報を取得します。</li>
	 * <li>通知情報マップに、通知情報をセットします。</li>
	 *  <dl>
	 *   <dt>通知情報マップ</dt>
	 *   <dd>キー：通知ID</dd>
	 *   <dd>値　：通知情報</dd>
	 *  </dl>
	 * </li>
	 * </ol>
	 * 
	 * @return 通知情報マップ
	 * @throws FinderException
	 * @throws NamingException
	 */
	public HashMap<String, NotifyInfo> getNotifyMap() throws NotifyNotFound, NamingException {

		HashMap<String, NotifyInfo> map = new HashMap<String, NotifyInfo>();
		try
		{
			// 通知情報一覧を取得
			Collection ct = NotifyInfoUtil.getLocalHome().findAll();

			Iterator itr = ct.iterator();
			NotifyInfo bean = null;
			NotifyInfoLocal notify = null;

			while(itr.hasNext()){

				notify = (NotifyInfoLocal)itr.next();
				bean = getNotify(notify);

				map.put(notify.getNotifyId(), bean);
			}

		} catch (FinderException e) {
			AplLogger apllog = new AplLogger("NOTIFY", "notify");
			apllog.put("SYS", "006");
			m_log.debug("getNotifyMap():" + e.getMessage());
			throw new NotifyNotFound(e.getMessage(),e);
		} catch (NamingException e) {
			AplLogger apllog = new AplLogger("NOTIFY", "notify");
			apllog.put("SYS", "006");
			m_log.debug("getNotifyMap():" + e.getMessage());
			throw e;
		}
		return map;
	}

	/**
	 * 障害検知用通知情報一覧を返します。
	 * <p>
	 * <ol>
	 * <li>通知IDの昇順に並んだ全ての通知情報を取得します。</li>
	 * <li>１通知情報をテーブルのカラム順（{@link com.clustercontrol.notify.bean.NotifyTableDefine}）に、リスト（{@link ArrayList}）にセットします。</li>
	 * <li>この１通知情報を保持するリストを、通知情報一覧を保持するリスト（{@link ArrayList}）に格納し返します。<BR>
	 *  <dl>
	 *  <dt>通知情報一覧（Objectの2次元配列）</dt>
	 *  <dd>{ 通知情報1 {カラム1の値, カラム2の値, … }, 通知情報2{カラム1の値, カラム2の値, …}, … }</dd>
	 *  </dl>
	 * </li>
	 * </ol>
	 * 
	 * @return 通知情報一覧（Objectの2次元配列）
	 * @throws NotifyNotFound
	 * 
	 * @see com.clustercontrol.notify.bean.NotifyTableDefine
	 * @see #collectionToArray(Collection)
	 */
	public ArrayList<NotifyInfo> getTroubleDetectionNotifyList() throws NotifyNotFound, HinemosUnknown {

		ArrayList<NotifyInfo> list = new ArrayList<NotifyInfo>();
		try
		{
			// 通知情報一覧を取得
			Collection<NotifyInfoLocal> ct = NotifyInfoUtil.getLocalHome().findAllOrderByNotifyId();

			// 障害検知用の通知IDのみを取得
			for(NotifyInfoLocal notify :ct){
				String regex = "^" + HinemosModuleConstant.SYSYTEM_TROUBLE_DETECTION + ".*";

				// FIXME : 文字列の比較による障害検知用通知IDの特定方法を変更する
				// 結果用Collectionから対象のCollectionのみリストに加える
				if(notify.getNotifyId().matches(regex)){
					list.add(getNotify(notify.getNotifyId()));
				}
			}
		} catch (FinderException e) {
			AplLogger apllog = new AplLogger("NOTIFY", "notify");
			apllog.put("SYS", "006");
			m_log.debug("getNotifyList():" + e.getMessage());
			throw new NotifyNotFound(e.getMessage(),e);
		} catch (NamingException e) {
			AplLogger apllog = new AplLogger("NOTIFY", "notify");
			apllog.put("SYS", "006");
			m_log.debug("getNotifyList():" + e.getMessage());
			throw new HinemosUnknown(e.getMessage(), e);
		}
		return list;
	}
}
