/*

Copyright (C) 2006 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.

 */

package com.clustercontrol.monitor.ejb.session;

import java.rmi.RemoteException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

import javax.activation.DataHandler;
import javax.ejb.EJBException;
import javax.ejb.FinderException;
import javax.ejb.RemoveException;
import javax.ejb.SessionContext;
import javax.naming.NamingException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.fault.EventLogNotFound;
import com.clustercontrol.fault.FacilityNotFound;
import com.clustercontrol.fault.HinemosUnknown;
import com.clustercontrol.fault.InvalidSetting;
import com.clustercontrol.fault.MonitorNotFound;
import com.clustercontrol.fault.UsedFacility;
import com.clustercontrol.bean.PluginConstant;
import com.clustercontrol.commons.bean.ViewListInfo;
import com.clustercontrol.commons.util.CommonValidator;
import com.clustercontrol.monitor.bean.EventBatchConfirmInfo;
import com.clustercontrol.monitor.bean.EventDataInfo;
import com.clustercontrol.monitor.bean.EventFilterInfo;
import com.clustercontrol.monitor.bean.ScopeDataInfo;
import com.clustercontrol.monitor.bean.StatusDataInfo;
import com.clustercontrol.monitor.bean.StatusFilterInfo;
import com.clustercontrol.monitor.factory.DeleteStatus;
import com.clustercontrol.monitor.factory.ManageStatus;
import com.clustercontrol.monitor.factory.ModifyEventComment;
import com.clustercontrol.monitor.factory.ModifyEventConfirm;
import com.clustercontrol.monitor.factory.SelectEvent;
import com.clustercontrol.monitor.factory.SelectScope;
import com.clustercontrol.monitor.factory.SelectStatus;
import com.clustercontrol.monitor.run.ejb.entity.MonitorInfoLocal;
import com.clustercontrol.monitor.run.ejb.entity.MonitorInfoUtil;
import com.clustercontrol.nodemap.util.Messages;
import com.clustercontrol.repository.ejb.session.RepositoryControllerBean;

/**
 * 監視管理機能の管理を行う Session Bean です。<BR>
 * クライアントからの Entity Bean へのアクセスは、Session Bean を介して行います。
 * 
 * <!-- begin-xdoclet-definition -->
 * @ejb.bean name="MonitorController"
 *           jndi-name="MonitorController"
 *           type="Stateless"
 *           transaction-type="Container"
 *           view-type="local"
 * 
 * @ejb.transaction type="Required"
 * 
 * @ejb.permission
 *     unchecked="true"
 *     method-intf="LocalHome"
 * 
 * @ejb.permission
 *     unchecked="true"
 *     method-intf="Local"
 * 
 * <!-- end-xdoclet-definition -->
 */
public abstract class MonitorControllerBean implements javax.ejb.SessionBean {

	/** ファシリティの配下全てのエントリ。 */
	public static final int ALL = RepositoryControllerBean.ALL;
	/** ファシリティの直下のエントリ。 */
	public static final int ONE_LEVEL = RepositoryControllerBean.ONE_LEVEL;
	/** ファシリティの自エントリ。 */
	public static final int ONLY = -1;

	/** ログ出力のインスタンス。 */
	private static Log m_log = LogFactory.getLog( MonitorControllerBean.class );

	private SessionContext m_context;

	@Override
	public void setSessionContext(SessionContext ctx) throws EJBException, RemoteException {
		m_context = ctx;
	}


	/**
	 * スコープ情報一覧を取得します。<BR><BR>
	 * 引数で指定されたファシリティの配下全てのファシリティのスコープ情報一覧を返します。<BR>
	 * 各スコープ情報は、ScopeDataInfoのインスタンスとして保持されます。<BR>
	 * 
	 * @ejb.interface-method
	 * 
	 * @jboss.method-attributes
	 *     read-only="true"
	 * 
	 * 
	 * @param facilityId 取得対象の親ファシリティID
	 * @return スコープ情報一覧（ScopeDataInfoが格納されたArrayList）
	 * @throws MonitorNotFound
	 * @throws HinemosUnknown
	 * @throws FacilityNotFound
	 * 
	 * @see com.clustercontrol.monitor.bean.ScopeDataInfo
	 * @see com.clustercontrol.monitor.factory.SelectScope#getScopeList(String)
	 */
	public ArrayList<ScopeDataInfo> getScopeList(String facilityId, boolean statusFlag, boolean eventFlag, boolean orderFlg)
	throws MonitorNotFound, HinemosUnknown, FacilityNotFound{

		//ステータス情報を取得
		SelectScope select = new SelectScope();
		ArrayList<ScopeDataInfo> list;
		try {
			list = select.getScopeList(facilityId, statusFlag, eventFlag, orderFlg);
		} catch (FacilityNotFound e){
			throw e;
		} catch (MonitorNotFound e) {
			throw e;
		} catch (Exception e) {
			m_log.error("getScopeList " + e.getMessage(), e);
			throw new HinemosUnknown(e.getMessage(), e);
		}

		return list;
	}

	/**
	 * 引数で指定された条件に一致するステータス情報一覧を取得します。<BR>
	 * 各ステータス情報は、StatusInfoDataのインスタンスとして保持されます。<BR>
	 * 
	 * @ejb.interface-method
	 * 
	 * @jboss.method-attributes
	 *     read-only="true"
	 * 
	 * @param facilityId 取得対象の親ファシリティID
	 * @param property 検索条件
	 * @return ステータス情報一覧（StatusInfoDataが格納されたArrayList）
	 * @throws MonitorNotFound
	 * @throws HinemosUnknown
	 * 
	 * @see com.clustercontrol.monitor.bean.StatusDataInfo
	 * @see com.clustercontrol.monitor.factory.SelectStatus#getStatusList(String, StatusFilterInfo)
	 */
	public ArrayList<StatusDataInfo> getStatusList(String facilityId, StatusFilterInfo filter)
	throws MonitorNotFound, HinemosUnknown{

		//ステータス情報を取得
		SelectStatus select = new SelectStatus();
		ArrayList<StatusDataInfo> list;
		try {
			list = select.getStatusList(facilityId, filter);
		} catch (MonitorNotFound e) {
			throw e;
		} catch (Exception e) {
			m_log.error("getStatusList " + e.getMessage(), e);
			throw new HinemosUnknown(e.getMessage(), e);
		}

		return list;
	}

	/**
	 * 引数で指定されたステータス情報を削除します。<BR>
	 * 
	 * 引数のlistは、StatusDataInfoが格納されたListとして渡されます。<BR>
	 * 
	 * @ejb.interface-method
	 * 
	 * @param list 削除対象のステータス情報一覧（StatusDataInfoが格納されたList）
	 * @return 削除に成功した場合、</code> true </code>
	 * @throws MonitorNotFound
	 * @throws HinemosUnknown
	 * 
	 * @since 2.0.0
	 * 
	 * @see com.clustercontrol.monitor.bean.StatusInfoData
	 * @see com.clustercontrol.monitor.factory.DeleteStatus#delete(List)
	 */
	public boolean deleteStatus(ArrayList<StatusDataInfo> list) throws MonitorNotFound, HinemosUnknown{

		//ステータス情報を削除
		DeleteStatus status = new DeleteStatus();

		boolean ret = false;

		try {
			ret = status.delete(list);
		}catch (MonitorNotFound e) {
			throw e;
		} catch (Exception e) {
			m_log.error("getStatusList " + e.getMessage(), e);
			throw new HinemosUnknown(e.getMessage(), e);
		}
		return ret;
	}

	/**
	 * 存続期間を経過したステータス情報を削除 または 更新します。<BR><BR>
	 * 
	 * Quartzから呼び出されるコールバックメソッド<BR>
	 * Quartz以外から呼び出さないでください。
	 * 
	 * @ejb.interface-method
	 * 
	 * @throws RemoveException
	 * @throws FinderException
	 * @throws NamingException
	 * 
	 * トランザクション開始はユーザが制御する。
	 * また、追加実装により、トランザクションの入れ子が予期せず生じることを避けるため、Neverを採用する。
	 * @ejb.transaction type="Never"
	 * 
	 * @since 2.0.0
	 *
	 * @see com.clustercontrol.monitor.factory.ManageStatus#execute()
	 * @see #addQuartz(String)
	 * @see #deleteQuartz()
	 */
	public void manageStatus() throws RemoveException, FinderException, NamingException {

		// 期限切れのステータス情報を削除/更新する
		ManageStatus manage = new ManageStatus();
		manage.execute();
	}

	/**
	 * 引数で指定された条件に一致するイベント一覧情報を取得します。(クライアントview用)<BR><BR>
	 * 
	 * 引数のpropertyには、com.clustercontrol.monitor.factory.EventFilterInfoの属性が１つ以上含まれます。<BR>
	 * 各イベント情報は、EventDataInfoインスタンスとして保持されます。<BR>
	 * 戻り値のViewListInfoは、クライアントにて表示用の形式に変換されます。
	 * 
	 * @ejb.interface-method
	 * 
	 * @jboss.method-attributes
	 *     read-only="true"
	 * 
	 * @param facilityId 取得対象の親ファシリティID
	 * @param filter 検索条件
	 * @param messages 表示イベント数
	 * @return ビュー一覧情報
	 * @throws MonitorNotFound
	 * @throws HinemosUnknown
	 * 
	 * @see com.clustercontrol.monitor.bean.EventDataInfo
	 * @see com.clustercontrol.monitor.factory.SelectEvent#getEventList(String, EventFilterInfo, int)
	 */
	public ViewListInfo getEventList(String facilityId, EventFilterInfo filter, int messages) throws MonitorNotFound, HinemosUnknown{

		// イベントログ情報を取得
		SelectEvent select = new SelectEvent();
		ViewListInfo list;
		try {
			list = select.getEventList(facilityId, filter, messages);
		} catch (MonitorNotFound e) {
			throw e;
		} catch (Exception e) {
			m_log.error("getEventList " + e.getMessage(), e);
			throw new HinemosUnknown(e.getMessage(), e);
		}

		return list;
	}

	/**
	 * 引数で指定された条件に一致する帳票出力用イベント情報一覧を返します。<BR><BR>
	 * 
	 * 引数のpropertyには、com.clustercontrol.monitor.factory.StatusFilterPropertyの属性が
	 * １つ以上含まれます。<BR>
	 * 戻り値のArrayListはArrayListのArrayListであり、内部のArrayListには、
	 * com.clustercontrol.monitor.bean.ReportEventInfoがリストとして格納されます。
	 * 
	 * @ejb.interface-method
	 * 
	 * @jboss.method-attributes
	 *     read-only="true"
	 * 
	 * @param facilityId 取得対象の親ファシリティID
	 * @param property 検索条件
	 * @return 帳票出力用イベント情報一覧（{@link com.clustercontrol.monitor.bean.ReportEventInfo}のリスト）
	 * @throws MonitorNotFound
	 * @throws HinemosUnknown
	 * 
	 * @since 2.1.0
	 * 
	 * @see com.clustercontrol.monitor.factory.SelectEvent#getEventListForReport(String, EventFilterInfo)
	 */
	public DataHandler downloadEventFile(String facilityId, EventFilterInfo filter, String filename)
	throws MonitorNotFound, HinemosUnknown{

		// 帳票出力用イベントログ情報を取得
		SelectEvent select = new SelectEvent();
		String username = m_context.getCallerPrincipal().getName();
		try {
			long now = System.currentTimeMillis();
			DataHandler handler = select.getEventFile(facilityId, filter, filename, username);
			long end = System.currentTimeMillis();
			m_log.info("downloadEventFile, time=" + (end - now) + "ms");
			return handler;
		} catch (MonitorNotFound e) {
			throw e;
		} catch (Exception e) {
			m_log.error("getEventList " + e.getMessage(), e);
			throw new HinemosUnknown(e.getMessage(), e);
		}
	}

	/**
	 * 一時ファイルとして作成したイベントファイルの削除。
	 * @ejb.interface-method
	 */
	public void deleteEventFile (String filename) {
		SelectEvent select = new SelectEvent();
		select.deleteEventFile(filename);
	}

	/**
	 * イベント詳細情報を取得します。<BR><BR>
	 * 
	 * @ejb.interface-method
	 * 
	 * @param monitorId 取得対象の監視項目ID
	 * @param pluginId 取得対象のプラグインID
	 * @param facilityId 取得対象のファシリティID
	 * @param outputDate 取得対象の受信日時
	 * @param locale ロケール情報
	 * @return イベント詳細情報
	 * @throws MonitorNotFound
	 * @throws HinemosUnknown
	 * 
	 */
	public EventDataInfo getEventInfo(
			String monitorId,
			String monitorDetailId,
			String pluginId,
			String facilityId,
			Long outputDate) throws MonitorNotFound, HinemosUnknown{

		// イベント情報詳細を取得
		try {
			return SelectEvent.getEventInfo(monitorId, monitorDetailId, pluginId, facilityId, outputDate);
		} catch (MonitorNotFound e) {
			throw e;
		} catch (Exception e) {
			m_log.error("getEventInfo " + e.getMessage(), e);
			throw new HinemosUnknown(e.getMessage(), e);
		}
	}

	/**
	 * 引数で指定されたイベント情報のコメントを更新します。<BR><BR>
	 * コメント追記ユーザとして、コメントユーザを設定します。
	 * 
	 * @ejb.interface-method
	 * 
	 * @param monitorId 更新対象の監視項目ID
	 * @param pluginId 更新対象のプラグインID
	 * @param facilityId 更新対象のファシリティID
	 * @param outputDate 更新対象の受信日時ID
	 * @param confirmDate 確認済み日時（更新値）
	 * @param confirmType 確認タイプ（未／済）（更新値）
	 * @throws HinemosUnknown
	 * @throws EventLogNotFound
	 * @throws InvalidSetting
	 * @throws FinderException
	 * 
	 * @see com.clustercontrol.bean.ConfirmConstant
	 * @see com.clustercontrol.monitor.factory.ModifyEventComment#modifyComment(String, String, String, Date, String, Date, String)
	 */
	public void modifyComment(
			String monitorId,
			String monitorDetailId,
			String pluginId,
			String facilityId,
			Long outputDate,
			String comment,
			Long commentDate,
			String commentUser
	) throws HinemosUnknown, EventLogNotFound, InvalidSetting {

		// コメントの文字数が2048文字より多い場合は、コメント変更不可
		CommonValidator.validateString(Messages.getString("comment"),
				comment, false, 0, 2048);

		commentUser = m_context.getCallerPrincipal().getName();

		// イベント通知のコメント状態を更新する
		ModifyEventComment modify = new ModifyEventComment();

		try{
			modify.modifyComment(monitorId, monitorDetailId, pluginId, facilityId, outputDate, comment, commentDate, commentUser);
		}catch(EventLogNotFound e){
			m_log.warn("modifyComment() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw e;
		}catch(Exception e){
			m_log.warn("modifyComment() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw new HinemosUnknown(e.getMessage(), e);
		}
	}


	/**
	 * 引数で指定されたイベント情報の確認を更新します。<BR><BR>
	 * 確認ユーザとして、操作を実施したユーザを設定します。
	 * 
	 * @ejb.interface-method
	 * 
	 * @param monitorId 更新対象の監視項目ID
	 * @param pluginId 更新対象のプラグインID
	 * @param facilityId 更新対象のファシリティID
	 * @param outputDate 更新対象の受信日時ID
	 * @param confirmDate 確認済み日時（更新値）
	 * @param confirmType 確認タイプ（未／済）（更新値）
	 * @throws MonitorNotFound
	 * @throws HinemosUnknown
	 * 
	 * @see com.clustercontrol.bean.ConfirmConstant
	 * @see com.clustercontrol.monitor.factory.ModifyEventConfirm#modifyConfirm(String, String, String, Date, Date, int)
	 * 
	 */
	public void modifyConfirm(
			String monitorId,
			String monitorDetailId,
			String pluginId,
			String facilityId,
			int priority,
			Long generateDate,
			Long outputDate,
			Long confirmDate,
			int confirmType
	) throws MonitorNotFound, HinemosUnknown{

		String confirmUser = m_context.getCallerPrincipal().getName();

		// イベントの確認状態を更新する
		ModifyEventConfirm modify = new ModifyEventConfirm();

		try{
			modify.modifyConfirm(monitorId, monitorDetailId, pluginId, facilityId, priority, generateDate, outputDate, confirmDate, confirmType, confirmUser);
		} catch (MonitorNotFound e) {
			m_log.warn("modifyConfirm() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw e;
		} catch (Exception e) {
			m_log.warn("modifyConfirm() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw new HinemosUnknown(e.getMessage(), e);
		}
	}

	/**
	 * 引数で指定されたイベント情報一覧の確認を更新します。<BR><BR>
	 * 確認ユーザとして、操作を実施したユーザを設定します。<BR>
	 * 複数のイベント情報を更新します。
	 * 
	 * @ejb.interface-method
	 * 
	 * @param list 更新対象のイベント情報一覧（EventDataInfoが格納されたArrayList）
	 * @param confirmType 確認タイプ（未／済）（更新値）
	 * @throws MonitorNotFound
	 * @throws HinemosUnknown
	 * 
	 * @see com.clustercontrol.bean.ConfirmConstant
	 * @see com.clustercontrol.monitor.factory.ModifyEventConfirm#modifyConfirm(List, int)
	 */
	public void modifyConfirm(ArrayList<EventDataInfo> list,
			int confirmType
	) throws MonitorNotFound, HinemosUnknown{

		String confirmUser = m_context.getCallerPrincipal().getName();

		// イベントの確認状態を更新する
		ModifyEventConfirm modify = new ModifyEventConfirm();

		try{
			modify.modifyConfirm(list, confirmType, confirmUser);
		} catch(MonitorNotFound e){
			m_log.warn("modifyConfirm() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw e;
		} catch (Exception e) {
			m_log.warn("modifyConfirm() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw new HinemosUnknown(e.getMessage(), e);
		}
	}

	/**
	 * 引数で指定された条件に一致するイベント情報の確認を一括更新します。<BR><BR>
	 * 確認ユーザとして、操作を実施したユーザを設定します。<BR>
	 * 
	 * @ejb.interface-method
	 * 
	 * @param confirmType 確認タイプ（未／済）（更新値）
	 * @param facilityId 更新対象の親ファシリティID
	 * @param property 更新条件
	 * @throws HinemosUnknown
	 * 
	 * @see com.clustercontrol.bean.ConfirmConstant
	 * @see com.clustercontrol.monitor.factory.ModifyEventConfirm#modifyBatchConfirm(int, String, EventBatchConfirmInfo)
	 */
	public void modifyBatchConfirm(int confirmType, String facilityId, EventBatchConfirmInfo info) throws HinemosUnknown{

		String confirmUser = m_context.getCallerPrincipal().getName();

		// イベントの確認状態を一括更新する
		ModifyEventConfirm modify = new ModifyEventConfirm();

		try{
			modify.modifyBatchConfirm(confirmType, facilityId, info, confirmUser);
		}catch(Exception e){
			m_log.warn("modifyBatchConfirm() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw new HinemosUnknown(e.getMessage(), e);
		}
	}

	/**
	 * ファシリティが利用されているか確認する。
	 * @ejb.interface-method
	 * 
	 */
	public void isUseFacilityId(String facilityId) throws UsedFacility {
		try {
			Collection infoCollection = MonitorInfoUtil.getLocalHome().findAll();
			for (Object o : infoCollection) {
				MonitorInfoLocal info = (MonitorInfoLocal)o;
				m_log.info("isUseFacilityId " + facilityId + ", " + info.getFacilityId());
				if (facilityId.equals(info.getFacilityId())) {
					m_log.info("isUseFacilityId,[" + facilityId + "], [" + info.getFacilityId() + "]");
					throw new UsedFacility(PluginConstant.TYPE_MONITOR);
				}
			}
		} catch (FinderException e) {
			m_log.warn("isUseFacilityId " + e.getMessage());
		} catch (NamingException e) {
			m_log.warn("isUseFacilityId " + e.getMessage());
		}
	}
}
