/*

Copyright (C) 2006 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.

 */

package com.clustercontrol.collectiverun.factory;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Date;
import java.util.Iterator;
import java.util.Locale;

import javax.ejb.FinderException;
import javax.naming.NamingException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.fault.CollectiveRunNotFound;
import com.clustercontrol.bean.StatusConstant;
import com.clustercontrol.commons.bean.ViewListInfo;
import com.clustercontrol.collectiverun.bean.DetailInfo;
import com.clustercontrol.collectiverun.bean.HistoryFilterInfo;
import com.clustercontrol.collectiverun.bean.HistoryInfo;
import com.clustercontrol.collectiverun.ejb.entity.CRunCmdMstLocal;
import com.clustercontrol.collectiverun.ejb.entity.CRunCmdMstUtil;
import com.clustercontrol.collectiverun.ejb.entity.CRunSessionDetailLocal;
import com.clustercontrol.collectiverun.ejb.entity.CRunSessionDetailUtil;
import com.clustercontrol.collectiverun.ejb.entity.CRunSessionLocal;
import com.clustercontrol.collectiverun.ejb.entity.CRunSessionUtil;
import com.clustercontrol.util.Messages;

/**
 * 履歴検索クラス
 *
 * @version 1.0.0
 * @since 1.0.0
 */
public class SelectSession {
	/** ログ出力のインスタンス */
	private static Log m_log = LogFactory.getLog( SelectSession.class );

	private final static int MAX_DISPLAY_NUMBER = 500;

	/**
	 * 履歴一覧取得
	 * 
	 * @param histories
	 * @return
	 * @throws NamingException
	 * @throws CollectiveRunNotFound
	 */
	@SuppressWarnings("unchecked")
	public ViewListInfo getHistoryList(int histories) throws NamingException, CollectiveRunNotFound {

		ViewListInfo list = new ViewListInfo();
		ArrayList<HistoryInfo> history = new ArrayList<HistoryInfo>();
		int total = 0;

		if(histories <= 0){
			histories = MAX_DISPLAY_NUMBER;
		}
		Integer limit = Integer.valueOf(histories + 1);

		//全セッションを取得
		Collection sessionList = null;
		try {
			sessionList = CRunSessionUtil.getLocalHome().selectHistoryFilter(null, null, null, null, null, null, limit);
		} catch (FinderException e) {
			m_log.warn(e.getMessage(), e);
			throw new CollectiveRunNotFound("SelectSession.getHistoryList() find error : " + e.getMessage());
		} catch (NamingException e) {
			throw e;
		}

		if(sessionList != null){

			// 履歴数をカウント
			if(sessionList.size() > histories){
				// 最大表示件数より大きい場合
				Collection allHistory = null;
				try {
					allHistory = CRunSessionUtil.getLocalHome().selectHistoryFilter(null, null, null, null, null, null, null);
				} catch (FinderException e) {
					m_log.warn(e.getMessage(), e);
					throw new CollectiveRunNotFound("SelectSession.getHistoryList() find error : " + e.getMessage());
				} catch (NamingException e) {
					throw e;
				}
				total = allHistory.size();

			} else {
				total = sessionList.size();
			}

			Iterator itr = sessionList.iterator();
			while(itr.hasNext()){
				//セッションを取得
				CRunSessionLocal session = (CRunSessionLocal)itr.next();
				CRunCmdMstLocal cmd = null;
				try {
					cmd = CRunCmdMstUtil.getLocalHome().findByPrimaryKey(session.getCommand_id());
				} catch (FinderException e) {
					m_log.info("command (command_id=" + session.getCommand_id() + ") not found in session (session_id=" + session.getSession_id() + ")");
				}

				//履歴一覧の１行を作成
				HistoryInfo info = new HistoryInfo();
				info.setStatus(session.getStatus());
				info.setSessionId(session.getSession_id());
				if(session.getStart_date() != null){
					info.setStartDate(session.getStart_date().getTime());
				}
				if(session.getEnd_date() != null){
					info.setEndDate(session.getEnd_date().getTime());
				}

				String name = null;
				try{
					name = SelectMaster.typeIdToString(session.getType_id(), Locale.getDefault());
				}catch(FinderException e){
					m_log.info("type (type_id=" + session.getType_id() + ") not found in session (session_id=" + session.getSession_id() + ")");
				}

				if(name != null){
					info.setType(name);
				}
				else{
					info.setType("");
				}

				if(cmd != null){
					info.setName(Messages.getString(cmd.getName_id(), Locale.getDefault()));
				}
				else{
					info.setName("");
				}
				info.setFacilityId(session.getFacility_id());
				info.setScopeText(session.getScope_text());
				info.setNodeCount(session.getNode_cnt());
				info.setEndNormalCount(session.getEnd_nomal_cnt());
				info.setEndWarningCount(session.getEnd_warning_cnt());
				info.setEndAbnormalCount(session.getEnd_abnomal_cnt());
				info.setWaitCount(session.getWait_cnt());
				info.setRunningCount(session.getRunning_cnt());
				history.add(info);

				//取得した履歴を最大表示件数まで格納したら終了
				if(history.size() >= histories)
					break;
			}
		}
		list.setTotal(total);
		list.setCollectiverunHistoryList(history);

		return list;
	}

	/**
	 * 履歴一覧取得
	 * 
	 * @param property
	 * @param histories
	 * @return
	 * @throws NamingException
	 * @throws CollectiveRunNotFound
	 */
	@SuppressWarnings("unchecked")
	public ViewListInfo getHistoryList(HistoryFilterInfo filter, int histories) throws NamingException, CollectiveRunNotFound {

		Date startFromDate = null;
		Date startToDate = null;
		Date endFromDate = null;
		Date endToDate = null;
		String type = null;
		Integer status = null;
		int total = 0;
		ArrayList<HistoryInfo> history = null;
		ViewListInfo list = null;

		//検索条件に該当するセッションを取得
		Collection<CRunSessionLocal> sessionList = null;
		try {
			//開始・再実行日時（自）取得

			if(filter.getStartFromDate() != null){
				startFromDate = new Date(filter.getStartFromDate());
			}
			//開始・再実行日時（至）取得

			if(filter.getStartToDate() != null){
				startToDate = new Date(filter.getStartToDate());
			}
			//終了・中断日時（自）取得

			if(filter.getEndFromDate() != null){
				endFromDate = new Date(filter.getEndFromDate());
			}
			//終了・中断日時（至）取得

			if(filter.getEndToDate() != null){
				endToDate = new Date(filter.getEndToDate());
			}

			//管理種別取得

			if(filter.getType() instanceof String && filter.getType().length() > 0){
				type = SelectMaster.stringToTypeId(filter.getType(), Locale.getDefault());
			}
			//状態取得

			if(filter.getStatus() instanceof String){
				status = Integer.valueOf(StatusConstant.stringToType(filter.getStatus()));
			}

			list = new ViewListInfo();
			history = new ArrayList<HistoryInfo>();

			if(histories <= 0){
				histories = MAX_DISPLAY_NUMBER;
			}

			sessionList = CRunSessionUtil.getLocalHome()
			.selectHistoryFilter(startFromDate, startToDate, endFromDate, endToDate, type, status, histories);
		} catch (FinderException e) {
			throw new CollectiveRunNotFound("SelectSession.getHistoryList() : "
					+ "startFromDate=" + startFromDate
					+ ", startToDate=" + startToDate
					+ ", endFromDate=" + endFromDate
					+ ", endToDate=" + endToDate
					+ ", type=" + type
					+ ", status=" + status
					+ ", histories=" + histories + " not found.");

		} catch (NamingException e) {
			throw e;
		}

		if(sessionList != null){
			// 履歴数を取得
			total = sessionList.size();

			for(CRunSessionLocal session : sessionList){
				CRunCmdMstLocal cmd = null;
				try {
					cmd = CRunCmdMstUtil.getLocalHome().findByPrimaryKey(session.getCommand_id());
				} catch (FinderException e) {
					m_log.info("command (command_id=" + session.getCommand_id() + ") not found in session (session_id=" + session.getSession_id() + ")");
				}

				//履歴一覧の１行を作成
				HistoryInfo info = new HistoryInfo();
				info.setStatus(session.getStatus());
				info.setSessionId(session.getSession_id());
				if(session.getStart_date() != null){
					info.setStartDate(session.getStart_date().getTime());
				}
				if(session.getEnd_date() != null){
					info.setEndDate(session.getEnd_date().getTime());
				}

				String name = null;
				try{
					name = SelectMaster.typeIdToString(session.getType_id(), Locale.getDefault());
				}catch(FinderException e){
					m_log.info("type (type_id=" + session.getType_id() + ") not found in session (session_id=" + session.getSession_id() + ")");
				}

				if(name != null){
					info.setName(Messages.getString(cmd.getName_id(), Locale.getDefault()));
				}
				else{
					info.setName("");
				}

				if(cmd != null){
					info.setType(name);
				}
				else{
					info.setType("");
				}
				info.setFacilityId(session.getFacility_id());
				info.setScopeText(session.getScope_text());
				info.setNodeCount(session.getNode_cnt());
				info.setEndNormalCount(session.getEnd_nomal_cnt());
				info.setEndWarningCount(session.getEnd_warning_cnt());
				info.setEndAbnormalCount(session.getEnd_abnomal_cnt());
				info.setWaitCount(session.getWait_cnt());
				info.setRunningCount(session.getRunning_cnt());
				history.add(info);

				//取得した履歴を最大表示件数まで格納したら終了
				if(history.size() >= histories)
					break;
			}
		}
		list.setTotal(total);
		list.setCollectiverunHistoryList(history);

		return list;
	}

	/**
	 * 一括制御詳細情報の一覧を取得します。
	 * 
	 * @param sessionId
	 * @return 一括制御詳細情報のリスト
	 * @throws CollectiveRunNotFound
	 * @throws NamingException
	 */
	@SuppressWarnings("unchecked")
	public ArrayList<DetailInfo> getDetailList(String sessionId) throws CollectiveRunNotFound, NamingException {
		ArrayList<DetailInfo> list = new ArrayList<DetailInfo>();

		//セッションディテールをセッションIDで検索し取得
		Collection<CRunSessionDetailLocal> detailList = null;
		try {
			detailList = CRunSessionDetailUtil.getLocalHome().findBySessionId(sessionId);
		} catch (FinderException e) {
			m_log.warn(e.getMessage(), e);
			throw new CollectiveRunNotFound("SelectSession.getDetailList() find error : sessionId=" + sessionId);
		} catch (NamingException e) {
			throw e;
		}
		if(detailList != null){
			for(CRunSessionDetailLocal detail : detailList){
				//履歴一覧の１行を作成
				DetailInfo info = new DetailInfo();
				info.setStatus(detail.getStatus());
				info.setEndStatus(detail.getEnd_status());
				info.setEndValue(detail.getEnd_value());
				info.setFacilityId(detail.getFacility_id());
				info.setFacilityName(detail.getFacility_name());
				if(detail.getStart_date() != null){
					info.setStartDate(detail.getStart_date().getTime());
				}
				if(detail.getEnd_date() != null){
					info.setEndDate(detail.getEnd_date().getTime());
				}
				StringBuffer message = new StringBuffer();
				if(detail.getMessage() != null){
					message.append(detail.getMessage());
				}
				if(detail.getError_message() != null){
					message.append(detail.getError_message());
				}

				info.setMessage(message.toString());
				list.add(info);
			}
		}
		return list;
	}
}
