/*

Copyright (C) 2011 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.

 */

package com.clustercontrol.agent.job;

import java.nio.charset.Charset;
import java.util.Date;
import java.util.Hashtable;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.agent.SendQueue;
import com.clustercontrol.agent.util.AgentProperties;
import com.clustercontrol.jobmanagement.bean.RunStatusConstant;
import com.clustercontrol.util.CommandCreator;
import com.clustercontrol.util.CommandExecutor;
import com.clustercontrol.util.CommandExecutor.CommandResult;
import com.clustercontrol.ws.jobmanagement.RunInstructionInfo;
import com.clustercontrol.ws.jobmanagement.RunResultInfo;

/**
 * コマンドを実行するスレッドクラス<BR>
 * 
 * ジョブ実行の際にプロセスを生成して、 終了まで、状態を監視するクラスです。
 * 
 */
public class CommandThread extends AgentThread {

	// ロガー
	static private Log m_log = LogFactory.getLog(CommandThread.class);

	private String m_resultMsg = "";
	private String m_errMsg = "";
	private int m_exitValue;

	// ジョブ実行結果を受け取る際のエンコーディング
	private String m_inputEncoding = null;

	/**
	 * デバッグ用メイン処理
	 * 
	 * @param args
	 */
	public static void main(String[] args) {

		CommandExecutor executor1 = new CommandExecutor(new String[] { "hostname" });
		System.out.println(executor1.execute().stdout);

	}

	/**
	 * デバッグ用コンストラクタ
	 */
	public CommandThread() {
		super();
	}

	/**
	 * コンストラクタ
	 * 
	 * @param info
	 *            実行指示
	 * @param sendQueue
	 *            実行応答用メッセージ送信クラス
	 * @param runHistory
	 *            実行履歴
	 */
	public CommandThread(RunInstructionInfo info, SendQueue sendQueue,
			Hashtable<String, Date> runHistory) {
		super(info, sendQueue, runHistory);

		// ログファイルのエンコーディングを設定
		m_inputEncoding =  AgentProperties.getProperty("input.encoding");
		if(m_inputEncoding == null){
			m_inputEncoding = System.getProperty("file.encoding");
		}
		m_log.info("input.encoding = " + m_inputEncoding);
	}

	/**
	 * ジョブ（コマンド・スクリプト）を実行するクラス<BR>
	 * 
	 * ReceiveTopicで受け取ったジョブの指示が実行の場合に このメソッドが実行されます。
	 * 
	 */
	/*
	 * (non-Javadoc)
	 * 
	 * @see java.lang.Runnable#run()
	 */
	@Override
	public void run() {
		m_log.debug("run start");

		Date startDate = new Date();

		// 実行履歴に追加
		RunHistoryUtil.addRunHistory(m_info, m_runHistory, startDate);

		// ---------------------------
		// -- 開始メッセージ送信
		// ---------------------------

		// メッセージ作成
		RunResultInfo info = new RunResultInfo();
		info.setSessionId(m_info.getSessionId());
		info.setJobunitId(m_info.getJobunitId());
		info.setJobId(m_info.getJobId());
		info.setFacilityId(m_info.getFacilityId());
		info.setCommand(m_info.getCommand());
		info.setCommandType(m_info.getCommandType());
		info.setStatus(RunStatusConstant.START);
		info.setTime(startDate.getTime());

		m_log.info("run SessionID=" + m_info.getSessionId() + ", JobID="
				+ m_info.getJobId());

		// 送信
		m_sendQueue.put(info);

		// ---------------------------
		// -- コマンド作成(OSへ渡す形式)
		// ---------------------------
		String[] cmd = null;
		String mode = AgentProperties.getProperty("command.create.mode");

		/** 指定されたモードでコマンド生成の処理を切り替える */
		try {
			CommandCreator.PlatformType platform = CommandCreator.convertPlatform(mode);
			cmd = CommandCreator.createCommand(m_info.getUser(), m_info.getCommand(), platform);
		} catch (Exception e) {

			// 実行履歴削除メッセージ送信
			m_log.error(e.getMessage());

			info.setStatus(RunStatusConstant.ERROR);
			info.setTime(new Date().getTime());
			info.setErrorMessage(e.getMessage());
			info.setMessage("");
			m_sendQueue.put(info);
			sendDeleteHistory(info);

			m_log.info("run end");
			return;
		}

		// ---------------------------
		// -- コマンド実行
		// ---------------------------
		for (int i = 0; i < cmd.length; i++) {
			m_log.info("Command Execute [" + i + "] : " + cmd[i]);
		}
		if (cmd.length == 0) {
			m_log.warn("Command Execute : cmd.length=0");
		}

		CommandExecutor cmdExec = new CommandExecutor(cmd, Charset.forName(m_inputEncoding), CommandExecutor._disableTimeout, m_limit_jobmsg);
		CommandResult cmdResult = cmdExec.execute();

		// ---------------------------
		// -- 終了メッセージ送信
		// ---------------------------

		if (cmdResult.exitCode != null) {

			info.setStatus(RunStatusConstant.END);
			info.setEndValue(cmdResult.exitCode);

		} else {

			info.setStatus(RunStatusConstant.ERROR);

		}

		// 終了を送信
		info.setTime(new Date().getTime());
		info.setErrorMessage(cmdResult.stderr);
		info.setMessage(cmdResult.stdout);
		m_sendQueue.put(info);

		// 実行履歴削除メッセージ送信
		sendDeleteHistory(info);

		m_log.debug("run end");
	}
}
