/*
 
Copyright (C) 2006 NTT DATA Corporation
 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation, version 2.
 
This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.
 
*/

package com.clustercontrol.sql.ejb.session;

import java.rmi.RemoteException;
import java.security.Principal;
import java.util.ArrayList;
import javax.ejb.EJBException;
import javax.ejb.SessionBean;
import javax.ejb.SessionContext;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.HinemosUnknownException;
import com.clustercontrol.MonitorDuplicateException;
import com.clustercontrol.MonitorNotFoundException;
import com.clustercontrol.bean.HinemosModuleConstant;
import com.clustercontrol.monitor.run.bean.MonitorInfo;
import com.clustercontrol.monitor.run.bean.MonitorTypeConstant;
import com.clustercontrol.monitor.run.factory.AddMonitor;
import com.clustercontrol.monitor.run.factory.DeleteMonitor;
import com.clustercontrol.monitor.run.factory.ModifyMonitor;
import com.clustercontrol.monitor.run.factory.SelectMonitor;
import com.clustercontrol.sql.bean.JdbcDriverInfo;
import com.clustercontrol.sql.factory.AddMonitorSql;
import com.clustercontrol.sql.factory.AddMonitorSqlString;
import com.clustercontrol.sql.factory.DeleteMonitorSql;
import com.clustercontrol.sql.factory.ModifyMonitorSql;
import com.clustercontrol.sql.factory.ModifyMonitorSqlString;
import com.clustercontrol.sql.factory.SelectMonitorSql;
import com.clustercontrol.sql.util.JdbcDriverUtil;

/**
 * SQL監視を制御するSession Bean <BR>
 * 
 * @ejb.bean name="MonitorSqlController"
 *           jndi-name="MonitorSqlController"
 *           type="Stateless"
 *           transaction-type="Container"
 * 
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=MonitorSqlInfoLocal"
 * 
 * @ejb.permission
 *     unchecked="true"
 *     method-intf="Home"
 *     
 * @ejb.permission
 *     unchecked="true"
 *     method-intf="LocalHome"
 *     
 * @ejb.permission
 *     unchecked="true"
 *     method-intf="Local"
 * 
 */
public abstract class MonitorSqlControllerBean implements SessionBean {
	
	protected static Log m_log = LogFactory.getLog( MonitorSqlControllerBean.class );
	
	@SuppressWarnings("unused")
	private SessionContext m_context;
	
	public void setSessionContext(SessionContext ctx) throws EJBException, RemoteException {
		m_context = ctx;
	}
	
	/**
	 * SQL監視情報をマネージャに登録します。<BR>
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
     *     role-name="SqlMonitorWrite"
     *     method-intf="Remote"
	 * 
	 * @param info 監視情報
	 * @return 登録に成功した場合、true
	 * @throws MonitorDuplicateException
	 * @throws HinemosUnknownException 
	 * @see com.clustercontrol.monitor.run.bean.MonitorInfo
	 */
	public boolean addSql(MonitorInfo info) throws MonitorDuplicateException, HinemosUnknownException{
		
		Principal pri = m_context.getCallerPrincipal();
		
		// SQL監視情報を登録
		AddMonitor sql = null;
		if(info.getMonitorType() == MonitorTypeConstant.TYPE_NUMERIC){
			// 数値監視の場合
			sql = new AddMonitorSql();
		}
		else if(info.getMonitorType() == MonitorTypeConstant.TYPE_STRING){
			// 文字列監視の場合
			sql = new AddMonitorSqlString();
		}
		else{
			return false;
		}
		boolean flag;
		try {
			flag = sql.add(info, pri.getName());
		} catch (MonitorDuplicateException e) {
			m_log.warn("addSql() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw e;
		} catch (Exception e) {
			m_log.error("addSql() : " + e.getClass().getSimpleName() +
					", " + e.getMessage(), e);
			m_context.setRollbackOnly();
			throw new HinemosUnknownException(e.getMessage(), e);
		}
		return flag;
	}
	
	/**
	 * マネージャ上のSQL監視情報を更新します。<BR>
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
     *     role-name="SqlMonitorWrite"
     *     method-intf="Remote"
	 * 
	 * @param info 監視情報
	 * @return 変更に成功した場合、true
	 * @throws MonitorNotFoundException 
	 * @throws HinemosUnknownException
	 * @see com.clustercontrol.monitor.run.bean.MonitorInfo
	 */
	public boolean modifySql(MonitorInfo info) throws MonitorNotFoundException, HinemosUnknownException{
		
		Principal pri = m_context.getCallerPrincipal();

		// SQL監視情報を更新
		ModifyMonitor sql = null;
		if(info.getMonitorType() == MonitorTypeConstant.TYPE_NUMERIC){
			// 数値監視の場合
			sql = new ModifyMonitorSql();
		}
		else if(info.getMonitorType() == MonitorTypeConstant.TYPE_STRING){
			// 文字列監視の場合
			sql = new ModifyMonitorSqlString();
		}
		else{
			return false;
		}
		boolean flag;
		try {
			flag = sql.modify(info, pri.getName());
		} catch (MonitorNotFoundException e) {
			m_log.warn("modifySql() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw e;
		} catch (Exception e) {
			m_log.error("modifySql() : " + e.getClass().getSimpleName() +
					", " + e.getMessage(), e);
			m_context.setRollbackOnly();
			throw new HinemosUnknownException(e.getMessage(), e);
		}
		return flag;
	}
	
	/**
	 * SQL監視情報をマネージャから削除します。<BR>
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
     *     role-name="SqlMonitorWrite"
     *     method-intf="Remote"
	 * 
	 * @param monitorId 監視項目ID
	 * @return 削除に成功した場合、true
	 * @throws MonitorNotFoundException
	 * @throws HinemosUnknownException 
	 */
	public boolean deleteSql(String monitorId) throws MonitorNotFoundException, HinemosUnknownException {

		// SQL監視情報を削除
		DeleteMonitor sql = new DeleteMonitorSql();
		boolean flag;
		try {
			flag = sql.delete(HinemosModuleConstant.MONITOR_SQL, monitorId);
		} catch (MonitorNotFoundException e) {
			m_log.warn("deleteSql() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw e;
		} catch (Exception e) {
			m_log.error("deleteSql() : " + e.getClass().getSimpleName() +
					", " + e.getMessage(), e);
			m_context.setRollbackOnly();
			throw new HinemosUnknownException(e.getMessage(), e);
		}
		return flag;
	}
	
	/**
	 * SQL監視情報を取得します。<BR>
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
     *     role-name="SqlMonitorRead"
     *     method-intf="Remote"
	 * 
	 * @jboss.method-attributes
	 *     read-only="true"
	 * 
	 * @param monitorId 監視項目ID
	 * @return SQL監視情報
	 * @throws MonitorNotFoundException
	 * @throws HinemosUnknownException
	 * @see com.clustercontrol.monitor.run.bean.MonitorInfo
	 */
	public MonitorInfo getSqlInfo(String monitorId) throws MonitorNotFoundException, HinemosUnknownException{

		// SQL監視情報を取得
		SelectMonitor sql = new SelectMonitorSql();
		
		MonitorInfo info = null;
		try {
			info = sql.getMonitor(HinemosModuleConstant.MONITOR_SQL, monitorId);
		} catch (MonitorNotFoundException e) {
			throw e;
		} catch (Exception e) {
			m_log.error("getSqlInfo " + e.getMessage(), e);
			throw new HinemosUnknownException(e.getMessage(), e);
		}
		return info;
	}

	/**
	 * SQL監視一覧リストを取得します。
	 * 
	 * 戻り値はArrayListのArrayListであり内部のArrayListは以下のように
	 * 値が格納される。
	 * 
	 * 
	 * 
	 * <Pre>
	 * ArrayList info = new ArrayList();
	 *  info.add(monitor.getMonitorId());
	 *  info.add(monitor.getDescription());
	 *  info.add(facilityPath);
	 *  info.add(MonitorTypeConstant.typeToString(monitor.getMonitorType().intValue()));
	 *  info.add(monitor.getCalendarId());
	 *  info.add(RunIntervalConstant.typeToString(monitor.getRunInterval().intValue()));
	 *  info.add(schedule.getValid(monitor.getMonitorTypeId(), monitor.getMonitorId()));
	 *  info.add(monitor.getRegUser());
	 *  info.add(monitor.getRegDate() == null ? null:new Date(monitor.getRegDate().getTime()));
	 *  info.add(monitor.getUpdateUser());
	 *  info.add(monitor.getUpdateDate() == null ? null:new Date(monitor.getUpdateDate().getTime()));
	 *
	 *  list.add(info);
	 * </Pre>
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
     *     role-name="SqlMonitorRead"
     *     method-intf="Remote"
     *     
     * @jboss.method-attributes
	 *     read-only="true"
	 * 
	 * @return Objectの2次元配列
	 * @throws MonitorNotFoundException 
	 * @throws HinemosUnknownException
	 */
	public ArrayList<MonitorInfo> getSqlList() throws MonitorNotFoundException, HinemosUnknownException{

		// SQL監視一覧を取得
		SelectMonitorSql sql = new SelectMonitorSql();
		ArrayList<MonitorInfo> list = null;
		try {
			list = sql.getMonitorList(HinemosModuleConstant.MONITOR_SQL);
		} catch (MonitorNotFoundException e) {
			throw e;
		} catch (Exception e) {
			m_log.error("getSqlList " + e.getMessage(), e);
			throw new HinemosUnknownException(e.getMessage(), e);
		}
		return list;
	}
	
	
	/**
	 * JDBC定義一覧をリストで返却します。<BR>
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
	 *     unchecked="true"
	 *     method-intf="Remote"
	 * 
	 * @return JDBC定義のリスト
	 */
	public ArrayList<JdbcDriverInfo> getJdbcDriverList(){
		JdbcDriverUtil util = new JdbcDriverUtil();
		return util.getJdbcDriver();
	}
}
