/*
 
Copyright (C) 2006 NTT DATA Corporation
 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation, version 2.
 
This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.
 
*/

package com.clustercontrol.snmp.ejb.session;

import java.rmi.RemoteException;
import java.security.Principal;
import java.util.ArrayList;
import javax.ejb.EJBException;
import javax.ejb.SessionBean;
import javax.ejb.SessionContext;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.HinemosUnknownException;
import com.clustercontrol.MonitorDuplicateException;
import com.clustercontrol.MonitorNotFoundException;
import com.clustercontrol.bean.HinemosModuleConstant;
import com.clustercontrol.monitor.run.bean.MonitorInfo;
import com.clustercontrol.monitor.run.bean.MonitorTypeConstant;
import com.clustercontrol.monitor.run.factory.AddMonitor;
import com.clustercontrol.monitor.run.factory.DeleteMonitor;
import com.clustercontrol.monitor.run.factory.ModifyMonitor;
import com.clustercontrol.monitor.run.factory.SelectMonitor;
import com.clustercontrol.snmp.factory.AddMonitorSnmp;
import com.clustercontrol.snmp.factory.AddMonitorSnmpString;
import com.clustercontrol.snmp.factory.DeleteMonitorSnmp;
import com.clustercontrol.snmp.factory.ModifyMonitorSnmp;
import com.clustercontrol.snmp.factory.ModifyMonitorSnmpString;
import com.clustercontrol.snmp.factory.SelectMonitorSnmp;

/**
 * SNMP監視を制御するSession Bean <BR>
 * @ejb.bean name="MonitorSnmpController"
 *           jndi-name="MonitorSnmpController"
 *           type="Stateless"
 *           transaction-type="Container"
 * 
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=MonitorSnmpInfoLocal"
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=MonitorSnmpValueLocal"
 * 
 * @ejb.permission
 *     unchecked="true"
 *     method-intf="Home"
 *     
 * @ejb.permission
 *     unchecked="true"
 *     method-intf="LocalHome"
 *     
 * @ejb.permission
 *     unchecked="true"
 *     method-intf="Local"
 * 
 */
public abstract class MonitorSnmpControllerBean implements SessionBean {
	
	protected static Log m_log = LogFactory.getLog( MonitorSnmpControllerBean.class );
	
	@SuppressWarnings("unused")
	private SessionContext m_context;

	public void setSessionContext(SessionContext ctx) throws EJBException, RemoteException {
		m_context = ctx;
	}
	
	/**
	 * SNMP監視情報をマネージャに登録します。<BR>
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
     *     role-name="SnmpMonitorWrite"
     *     method-intf="Remote"
	 * 
	 * @param info 監視情報
	 * @return 登録に成功した場合、true
	 * @throws MonitorDuplicateException 
	 * @throws HinemosUnknownException
	 * @see com.clustercontrol.monitor.run.bean.MonitorInfo
	 */
	public boolean addSnmp(MonitorInfo info) throws MonitorDuplicateException, HinemosUnknownException{
		
		Principal pri = m_context.getCallerPrincipal();

		// SNMP監視情報を登録
		AddMonitor snmp = null;
		if(info.getMonitorType() == MonitorTypeConstant.TYPE_NUMERIC){
			// 数値監視の場合
			snmp = new AddMonitorSnmp();
		}
		else if(info.getMonitorType() == MonitorTypeConstant.TYPE_STRING){
			// 文字列監視の場合
			snmp = new AddMonitorSnmpString();
		}
		else{
			return false;
		}
		boolean flag;
		try {
			flag = snmp.add(info, pri.getName());
		} catch (MonitorDuplicateException e) {
			m_log.warn("addSnmp() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw e;
		} catch (Exception e) {
			m_log.error("addSnmp() : " + e.getClass().getSimpleName() +
					", " + e.getMessage(), e);
			m_context.setRollbackOnly();
			throw new HinemosUnknownException(e.getMessage(), e);
		}
		return flag;
	}
	
	/**
	 * マネージャ上のSNMP監視設定報を更新します。<BR>
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
     *     role-name="SnmpMonitorWrite"
     *     method-intf="Remote"
	 * 
	 * @param info 監視情報
	 * @return 変更に成功した場合、true
	 * @throws MonitorNotFoundException 
	 * @throws HinemosUnknownException
	 * @ see com.clustercontrol.monitor.run.bean.MonitorInfo
	 */
	public boolean modifySnmp(MonitorInfo info) throws MonitorNotFoundException, HinemosUnknownException{
		
		Principal pri = m_context.getCallerPrincipal();

		// SNMP監視情報を更新
		ModifyMonitor snmp = null;
		if(info.getMonitorType() == MonitorTypeConstant.TYPE_NUMERIC){
			// 数値監視の場合
			snmp = new ModifyMonitorSnmp();
		}
		else if(info.getMonitorType() == MonitorTypeConstant.TYPE_STRING){
			// 文字列監視の場合
			snmp = new ModifyMonitorSnmpString();
		}
		else{
			return false;
		}
		boolean flag;
		try {
			flag = snmp.modify(info, pri.getName());
		} catch (MonitorNotFoundException e) {
			m_log.warn("modifySnmp() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw e;
		} catch (Exception e) {
			m_log.error("modifySnmp() : " + e.getClass().getSimpleName() +
					", " + e.getMessage(), e);
			m_context.setRollbackOnly();
			throw new HinemosUnknownException(e.getMessage(), e);
		}
		return flag;
	}

	/**
	 * SNMP監視設定情報をマネージャから削除します。<BR>
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
     *     role-name="SnmpMonitorWrite"
     *     method-intf="Remote"
	 * 
	 * @param monitorId 監視項目ID
	 * @return 削除に成功した場合、true
	 * @throws MonitorNotFoundException 
	 * @throws HinemosUnknownException
	 */
	public boolean deleteSnmp(String monitorId) throws MonitorNotFoundException, HinemosUnknownException {

		// SNMP監視情報を削除
		DeleteMonitor snmp = new DeleteMonitorSnmp();
		boolean flag;
		try {
			flag = snmp.delete(HinemosModuleConstant.MONITOR_SNMP, monitorId);
		} catch (MonitorNotFoundException e) {
			m_log.warn("deleteSnmp() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw e;
		} catch (Exception e) {
			m_log.error("deleteSnmp() : " + e.getClass().getSimpleName() +
					", " + e.getMessage(), e);
			m_context.setRollbackOnly();
			throw new HinemosUnknownException(e.getMessage(), e);
		}
		return flag;
	}
	
	/**
	 * SNMP監視情報を取得します。<BR>
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
     *     role-name="SnmpMonitorRead"
     *     method-intf="Remote"
	 * 
	 * @jboss.method-attributes
	 *     read-only="true"
	 * 
	 * @param monitorId 状態監視ID
	 * @return SNMP監視情報
	 * @throws MonitorNotFoundException 
	 * @throws HinemosUnknownException
	 * @see com.clustercontrol.monitor.run.bean.MonitorInfo
	 */
	public MonitorInfo getSnmp(String monitorId) throws MonitorNotFoundException, HinemosUnknownException{
		
		// SNMP監視情報を取得
		SelectMonitor snmp = new SelectMonitorSnmp();
		
		MonitorInfo info = null;
		try {
			info = snmp.getMonitor(HinemosModuleConstant.MONITOR_SNMP, monitorId);
		} catch (MonitorNotFoundException e) {
			throw e;
		} catch (Exception e) {
			m_log.error("getSnmp " + e.getMessage(), e);
			throw new HinemosUnknownException(e.getMessage(), e);
		}
		return info;
	}

	/**
	 * SNMP監視一覧リストを取得します。<BR>
	 * 
	 * 戻り値はArrayListのArrayListであり内部のArrayListは以下のように
	 * 値が格納される。
	 * 
	 * 
	 * 
	 * <Pre>
	 * ArrayList info = new ArrayList();
	 *  info.add(monitor.getMonitorId());
	 *  info.add(monitor.getDescription());
	 *  info.add(facilityPath);
	 *  info.add(MonitorTypeConstant.typeToString(monitor.getMonitorType().intValue()));
	 *  info.add(monitor.getCalendarId());
	 *  info.add(RunIntervalConstant.typeToString(monitor.getRunInterval().intValue()));
	 *  info.add(schedule.getValid(monitor.getMonitorTypeId(), monitor.getMonitorId()));
	 *  info.add(monitor.getRegUser());
	 *  info.add(monitor.getRegDate() == null ? null:new Date(monitor.getRegDate().getTime()));
	 *  info.add(monitor.getUpdateUser());
	 *  info.add(monitor.getUpdateDate() == null ? null:new Date(monitor.getUpdateDate().getTime()));
	 * 
	 *  list.add(info);
	 * </Pre>
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
     *     role-name="SnmpMonitorRead"
     *     method-intf="Remote"
     *     
     * @jboss.method-attributes
	 *     read-only="true"
	 * 
	 * @return Objectの2次元配列
	 * @throws MonitorNotFoundException 
	 * @throws HinemosUnknownException
	 */
	public ArrayList<MonitorInfo> getSnmpList() throws MonitorNotFoundException, HinemosUnknownException {

		// SNMP監視一覧を取得
		SelectMonitorSnmp snmp = new SelectMonitorSnmp();
		ArrayList<MonitorInfo> list = null;
		try {
			list = snmp.getMonitorList(HinemosModuleConstant.MONITOR_SNMP);
		} catch (MonitorNotFoundException e) {
			throw e;
		} catch (Exception e) {
			m_log.error("getSnmpList " + e.getMessage(), e);
			throw new HinemosUnknownException(e.getMessage(), e);
		}

		return list;
	}
}
