/*
                                                                                                
Copyright (C) 2008 NTT DATA Corporation
 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.
 
This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.
 
*/

package com.clustercontrol.sharedtable;

import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.Iterator;
import java.util.List;
import java.util.Set;
import java.util.TreeSet;
import java.util.concurrent.ConcurrentHashMap;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.poller.NotInitializedException;

public class SharedTable {
	private static Log m_log = LogFactory.getLog( SharedTable.class );
	
	private Object lock = new Object();

	private static SharedTable m_instance = null;
	
	// 参照されなくなったデータテーブルの保持期間
	private long m_keepAlive = 60000L;  // 60秒間
	
	/**
	 * データグループ名とデータテーブル名をキーに
	 * データテーブルの集合を管理するマップ
	 * 
	 * Key   : テーブルグループ名とテーブル名を保持するクラス
	 * Value : データテーブルの集合
	 */
	private ConcurrentHashMap<TableInfo, DataTableCollection> m_tableMap;

	private SharedTable(){
		m_tableMap = new ConcurrentHashMap<TableInfo, DataTableCollection>();
	}
	
	/**
	 * インスタンスを返します。
	 * シングルトンであるため返されるインスタンスは常に同じものとなります。
	 * @return SharedTableクラスのインスタンス
	 */
	static public synchronized SharedTable getInstance(){
		if(m_instance == null){
			m_instance = new SharedTable();
		}
		
		return m_instance;
	}

	/**
	 * 参照されなくなったデータテーブルの保持期間を設定します。
	 * @param keepAlive 参照されなくなったデータテーブルの保持期間
	 */
	public void setKeepAlive(long keepAlive) {
		m_keepAlive = keepAlive;
	}

	/**
	 * 参照されなくなったデータテーブルの保持期間を返します。
	 * @return 参照されなくなったデータテーブルの保持期間
	 */
	public long getKeepAlive() {
		return m_keepAlive;
	}
	
	/**
	 * データテーブルを生成します。（既に存在する場合は何もしません）
	 * @param tableGroup テーブルグループ名
	 * @param tableName テーブル名
	 * @param pageSize ページ数
	 */
	public void createDataTable(String tableGroup, String tableName, int pageSize){
		synchronized (lock) {
			m_log.debug("creat table : " + tableName + "(" + tableGroup + ")");
			
			// ページサイズが1以下の場合はエラーとする
			if(pageSize < 1){
				throw new IllegalArgumentException();
			}
			
			TableInfo info = new TableInfo(tableGroup, tableName);

			// 既に存在するかを調べる
			DataTableCollection dtc = m_tableMap.get(info);
			if(dtc != null){
				// 指定のものと同じページサイズのものが既にある場合は、新規には生成しない
				if(dtc.getPageSize() == pageSize){
					return;
				}
			}
			
			// データテーブルの集合を管理するクラスを生成し、マップに登録する
			dtc = new DataTableCollection(pageSize, m_keepAlive);
			m_tableMap.put(info, dtc);
		}
	}
	
	/**
	 * データテーブルを削除します。
	 * @param tableGroup テーブルグループ名
	 * @param tableName テーブル名
	 */
	public void removeDataTable(String tableGroup, String tableName){
		synchronized (lock) {
			m_log.debug("remove table : " + tableName + "(" + tableGroup + ")");
			TableInfo info = new TableInfo(tableGroup, tableName);

			// データテーブルの集合を管理するクラスをマップから削除
			m_tableMap.remove(info);
		}
	}

	/**
	 * 指定の収集名でデータテーブルにアクセス可能なように設定する
	 * @param tableGroup テーブルグループ
	 * @param tableName テーブル名
	 * @param collectorName 収集名
	 * @param interval 収集間隔
	 * @throws DataTableNotFoundException
	 */
	public void registerCollector(String tableGroup, String tableName, String collectorName, int interval) throws DataTableNotFoundException{
		TableInfo info = new TableInfo(tableGroup, tableName);
		
		DataTableCollection dtc = m_tableMap.get(info);
		
		if(dtc == null){
			throw new DataTableNotFoundException(tableGroup, tableName);
		}

		dtc.creatDataTableHolder(collectorName, interval);
	}
	
	/**
	 * 指定の収集名でデータテーブルにアクセス不能なように設定する
	 * @param tableGroup テーブルグループ
	 * @param tableName テーブル名
	 * @param collectorName 収集名
	 * @throws DataTableNotFoundException
	 */
	public void unregisterCollector(String tableGroup, String tableName, String collectorName)
	throws DataTableNotFoundException{
		TableInfo info = new TableInfo(tableGroup, tableName);
		
		DataTableCollection dtc = m_tableMap.get(info);
		
		if(dtc == null){
			throw new DataTableNotFoundException(tableGroup, tableName);
		}

		dtc.removeCollectorName(collectorName);
	}
	
	/**
	 * データテーブルを挿入します。
	 * checkKeyは、テーブルホルダで管理されているページのキーと同じキーを指定する必要があります。
	 * キーが異なる場合は、テーブルホルダの全てのページがクリアされ、
	 * 今回与えられたデータテーブルのみが挿入された状態となります。
	 * @param tableGroup データグループ名
	 * @param tableName データテーブル名
	 * @param interval 収集間隔
	 * @param dataTable データテーブル
	 * @param checkKey テーブルホルダにデータを格納する際のキー
	 * @throws DataTableNotFoundException
	 */
	public void insertDataTable(String tableGroup, String tableName, int interval, DataTable table, String checkKey)
	throws DataTableNotFoundException{
		TableInfo info = new TableInfo(tableGroup, tableName);
		
		DataTableCollection dtc = m_tableMap.get(info);
		
		if(dtc == null){
			// エラー処理
			throw new DataTableNotFoundException(tableGroup, tableName);
		}
		
		try {
//			m_log.debug("insert DataTable : " + tableGroup + ", " + tableName);
			dtc.insertDataTable(interval, table, checkKey);
		} catch (NotInitializedException e) {
			// エラー処理
			throw new DataTableNotFoundException(tableGroup, tableName);
		}
	}
	
	/**
	 * 指定のデータ格納テーブルが存在するか確認します。
	 * @param tableGroup テーブルグループ
	 * @param tableName テーブル名
	 * @return 存在する場合はtrue
	 */
	public boolean containsDataTable(String tableGroup, String tableName){
		TableInfo info = new TableInfo(tableGroup, tableName);
		
		return m_tableMap.containsKey(info);
	}
	
	/**
	 * 指定のデータ格納テーブルが存在するか確認します。
	 * @param tableGroup データグループ名
	 * @param tableName データテーブル名
	 * @return 存在する場合はtrue
	 */
	public boolean containsTable(
			String tableGroup, 
			String tableName,
			int pageSize){
		TableInfo info = new TableInfo(tableGroup, tableName);
		
		DataTableCollection dtc = m_tableMap.get(info);
		
		if(dtc == null){
			return false;
		}
		
		return (dtc.getPageSize() == pageSize);
	}
	
	/**
	 * 指定の収集名用のデータ格納テーブルが存在するか確認します。
	 * @param tableGroup データグループ名
	 * @param tableName データテーブル名
	 * @param collectorName 収集名
	 * @return 存在する場合はtrue
	 * @throws DataTableNotFoundException 
	 */
	public boolean containsCollectorName(
			String tableGroup, 
			String tableName, 
			String collectorName) throws DataTableNotFoundException{
		TableInfo info = new TableInfo(tableGroup, tableName);
		
		DataTableCollection dtc = m_tableMap.get(info);
		
		if(dtc == null){
			throw new DataTableNotFoundException(tableGroup, tableName);
		}
		
		return dtc.containsCollectorName(collectorName);
	}
	
	/**
	 * データテーブルは、10秒間隔での監視用、30秒間隔での監視用...
	 * と監視間隔ごとに複数のテーブルを保持する
	 * 
	 * 指定のデータテーブルが保持しているテーブルホルダのうち
	 * 存在する収集間隔のセットを返す。
	 * 
	 * @param tableGroup データグループ名
	 * @param tableName データテーブル名
	 * @return 収集間隔のセット
	 * @throws DataTableNotFoundException 
	 */
	public Set<Integer> getIntervals(String tableGroup,	String tableName)
	throws DataTableNotFoundException{
		TableInfo info = new TableInfo(tableGroup, tableName);
		
		DataTableCollection dtc = m_tableMap.get(info);
		
		if(dtc == null){
			throw new DataTableNotFoundException(tableGroup, tableName);
		}
		
		return dtc.getIntervals();
	}
	
	/**
	 * 設定されている収集名のセットを取得します
	 * @return 設定されている収集名のセット
	 * @throws DataTableNotFoundException 
	 */
	public Set<String> getCollectorNames(String tableGroup,	String tableName) 
	throws DataTableNotFoundException{
		TableInfo info = new TableInfo(tableGroup, tableName);
		
		DataTableCollection dtc = m_tableMap.get(info);
		
		if(dtc == null){
			throw new DataTableNotFoundException(tableGroup, tableName);
		}
		
		return dtc.getCollectorNames();
	}
	
	/**
	 * データテーブルを取得します。
	 * @param tableGroup データグループ名
	 * @param tableName データテーブル名
	 * @param collectorName 収集名
	 * @param pageIndex ページ番号
	 * @return データテーブル
	 * @throws DataTableNotFoundException 
	 * @throws NotInitializedException 
	 */
	public DataTable getDataTable(
			String tableGroup, 
			String tableName, 
			String collectorName, 
			int pageIndex)
	throws DataTableNotFoundException, NotInitializedException{
		TableInfo info = new TableInfo(tableGroup, tableName);
		
		DataTableCollection dtc = m_tableMap.get(info);
		
		if(dtc == null){
			throw new DataTableNotFoundException(tableGroup, tableName);
		}
		
		DataTableHolder tableHolder = dtc.getDataTableHolder(collectorName);
		
		if(tableHolder == null){
			throw new NotInitializedException(
					collectorName + " is not registered in " + tableName + "(" + tableGroup + ")");
		}
		
		return tableHolder.get(pageIndex);
	}
	
	
	/**
	 * データテーブルを取得します。
	 * @param tableGroup データグループ名
	 * @param tableName データテーブル名
	 * @param collectorName 収集名
	 * @param pageSize ページ数
	 * @return データテーブル
	 * @throws DataTableNotFoundException 
	 * @throws NotInitializedException 
	 */
	public List<DataTable> getLastDataTables(
			String tableGroup, 
			String tableName, 
			String collectorName, 
			int pageSize)
	throws DataTableNotFoundException, NotInitializedException{
		TableInfo info = new TableInfo(tableGroup, tableName);
		
		DataTableCollection dtc = m_tableMap.get(info);
		
		if(dtc == null){
			throw new DataTableNotFoundException(tableGroup, tableName);
		}
		
		DataTableHolder tableHolder = dtc.getDataTableHolder(collectorName);
		
		if(tableHolder == null){
			throw new DataTableNotFoundException(tableGroup, tableName);
		}
		
		long insertCount = tableHolder.getInsertCount();
		
		if(m_log.isDebugEnabled()){
			m_log.debug(collectorName + ", " + tableName + "(" + tableGroup + ")  "
					+ "request page size=" + pageSize + " insert page="+ insertCount);
		}
		
		// 指定のページ数分まだDataTableが挿入されていない場合
		if(insertCount < pageSize){
			throw new NotInitializedException(
					" Not enough page. " + collectorName + ", " + tableName + "(" + tableGroup + ")");
		}
		
		return tableHolder.getLast(pageSize);
	}
	
	/**
	 * 管理しえているデータテーブル全てをチェックし、
	 * 一定期間（収集間隔 ＋ 生存期間（m_keepAlive））以上経過しているものは削除する
	 */
	public void checkUnnecessaryTable() {
		long now = System.currentTimeMillis();
			
		// チェックを実行
		Iterator<DataTableCollection> itr = m_tableMap.values().iterator();
			
		while(itr.hasNext()){
			itr.next().checkAlive(now);
		}
	}
	
	/**
	 * 管理しているテーブルの情報を文字列で返します。
	 * @return 管理しているテーブルの情報
	 */
	public String getTableListDebugInfo(){
		SimpleDateFormat formatter = new SimpleDateFormat("MM-dd HH:mm:ss.SSS");
		
		String debugStr = "";
		
		// 各テーブルに関しての情報を下記フォーマットで出力
		// テーブルグループ    テーブル名    ページサイズ    (収集間隔, 収集間隔, 収集間隔...)
		//     収集名    収集間隔    最終参照日時
		//     収集名    収集間隔    最終参照日時
		//     収集名    収集間隔    最終参照日時
		//     ...
		for(TableInfo info : m_tableMap.keySet()){
			DataTableCollection col = m_tableMap.get(info);
			// 取得できなかった場合は、次のテーブルに処理を進める
			if(col == null){
				break;
			}
			
			Set<String> collectorNames = col.getCollectorNames();
		
			String txt = 
				info.getTableGroup() + ", " + info.getTableName() + ", page size = " + col.getPageSize();
			
			// 保持しているテーブルの収集間隔を出力
			txt = txt + ", interval = (";
			TreeSet<Integer> intervals = new TreeSet<Integer>(col.getIntervals());
			Iterator<Integer> itrInter = intervals.iterator();
			while(itrInter.hasNext()){
				int interval = itrInter.next();
				txt = txt + interval;
				if(itrInter.hasNext()){
					txt = txt + ", ";
				}
			}
			txt = txt + ")\n";
			
			// 各収集名に関しての情報を下記フォーマットで出力
			//     収集名    収集間隔    最終参照日時
			String collectorNameList = "";
			for(String collectorName : collectorNames){
				DataTableHolder holder = col.getDataTableHolder(collectorName);
				
				collectorNameList = collectorNameList + "\t"
				+ collectorName + "\t" 
				+ col.getInterval(collectorName) + "\t" + " last ref \t"
				+ formatter.format(new Date(col.getLastReference(collectorName)))
				+ "\t insert:" + holder.getInsertCount()
				+ "\n";

				// 各収集間隔ごとに管理されているDataTableHolderの各ページの更新時刻を出力
				String holderInfoTxt = "";
				if(holder != null){
					// 各ページの最終更新時刻を取得します
					for(int i=0; i<holder.getPageSize(); i++){
						DataTable table = holder.get(i);
						if(table != null){
							holderInfoTxt = holderInfoTxt + "\t\t"
							+ "page:" + i 
							+ "\t" + " create \t" + formatter.format(new Date(table.getCreateTime()))
							+ "\t" + " last modify \t" + formatter.format(new Date(table.getLastModify()))
							+ "\n";
						}
					}
				}
				
				collectorNameList = collectorNameList + holderInfoTxt;
			}
			
			txt = txt + collectorNameList + "\n";
			
			debugStr = debugStr + txt;
		}
		
		return debugStr;
	}
	
	/**
	 * 収集値を格納する対象テーブルを特定する情報を保持するクラス
	 * HashMapのキーとして利用できるように equalsメソッドと hashCodeメソッドを実装
	 */
	private class TableInfo{
		private String m_tableGroup;
		private String m_tableName;
		
		public TableInfo(String tableGroup, String tableName){
			m_tableGroup = tableGroup;
			m_tableName = tableName;
		}

		public String getTableGroup() {
			return m_tableGroup;
		}

		public String getTableName() {
			return m_tableName;
		}
		
		@Override
		public boolean equals(Object other) {
			if (other instanceof TableInfo) {
				TableInfo info = (TableInfo)other;
				
				if (this.m_tableGroup == null && this.m_tableName == null){
					if (info.m_tableGroup == null && info.m_tableName == null){
						return true;
					}
				} else if (this.m_tableGroup == null && this.m_tableName != null){
					if (info.m_tableGroup == null && this.m_tableName.equals(info.m_tableName)){
						return true;
					}
				} else if (this.m_tableGroup != null && this.m_tableName == null){
					if (this.m_tableGroup.equals(info.m_tableGroup) && info.m_tableName == null){
						return true;
					}
				} else {
					if (this.m_tableGroup.equals(info.m_tableGroup)){
						return this.m_tableName.equals(info.m_tableName);
					}
				}
				return false;
			} else {
				return false;
			}
		}

		@Override
		public int hashCode() {
			int result = 17;

			result = 37 * result + ((this.m_tableGroup != null) ? this.m_tableGroup.hashCode() : 0);

			result = 37 * result + ((this.m_tableName != null) ? this.m_tableName.hashCode() : 0);

			return result;
		}
	}
}
	
