/*
 
Copyright (C) 2006 NTT DATA Corporation
 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation, version 2.
 
This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.
 
*/

package com.clustercontrol.repository.ejb.session;

import java.net.UnknownHostException;
import java.rmi.RemoteException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Date;
import java.util.HashMap;
import java.util.Locale;

import javax.ejb.DuplicateKeyException;
import javax.ejb.EJBException;
import javax.ejb.SessionContext;
import javax.naming.NamingException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.FacilityDuplicateException;
import com.clustercontrol.FacilityNotFoundException;
import com.clustercontrol.UsedFacilityException;
import com.clustercontrol.HinemosUnknownException;
import com.clustercontrol.bean.FacilityTreeItem;
import com.clustercontrol.repository.bean.FacilitySortOrderConstant;
import com.clustercontrol.repository.bean.NodeInfo;
import com.clustercontrol.repository.bean.RepositoryTableInfo;
import com.clustercontrol.repository.bean.ScopeInfo;
import com.clustercontrol.repository.bean.TopicConstant;
import com.clustercontrol.repository.ejb.entity.CollectorPlatformMstLocal;
import com.clustercontrol.repository.ejb.entity.CollectorPlatformMstUtil;
import com.clustercontrol.repository.ejb.entity.DeviceInfoData;
import com.clustercontrol.repository.ejb.entity.FacilityInfoData;
import com.clustercontrol.repository.ejb.entity.FacilityLocal;
import com.clustercontrol.repository.ejb.entity.FacilityUtil;
import com.clustercontrol.repository.ejb.entity.NodeLocal;
import com.clustercontrol.repository.factory.FacilityModifier;
import com.clustercontrol.repository.factory.NodeProperty;
import com.clustercontrol.repository.factory.ScopeProperty;
import com.clustercontrol.repository.factory.SearchNodeBySNMP;
import com.clustercontrol.repository.factory.FacilitySelector;
import com.clustercontrol.repository.util.RepositoryUpdateTime;
import com.clustercontrol.repository.util.SendTopic;
import com.clustercontrol.vm.ejb.entity.VmProtocolMstUtil;
import com.clustercontrol.vm.ejb.entity.VmSolutionMstLocal;
import com.clustercontrol.vm.ejb.entity.VmSolutionMstUtil;

/**
 *
 * <!-- begin-user-doc --> リポジトリ情報（ノード、スコープ）の生成、変更、削除、
 * 参照を行うSessionBean<BR>　
 * クライアントからの Entity Bean へのアクセスは、Session Bean を介して行います。
 * <!-- end-user-doc --> *
 *
 * <!-- begin-xdoclet-definition --> 
 * @ejb.bean name="RepositoryController"	
 *           jndi-name="RepositoryController"
 *           type="Stateless" 
 *           transaction-type="Container"
 * 
 * @ejb.transaction
 *   type="Required"
 * 
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=FacilityLocal"
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=FacilityRelationLocal"
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=NodeLocal"
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=NodeHostnameLocal"
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=NodeNoteLocal"
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=NodeDeviceLocal"
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=NodeFilesystemLocal"
 * 
 * @ejb.permission
 *     unchecked="true"
 *     method-intf="Home"
 *     
 * @ejb.permission
 *     unchecked="true"
 *     method-intf="LocalHome"
 *     
 * @ejb.permission
 *     unchecked="true"
 *     method-intf="Local"
 *           
 * <!-- end-xdoclet-definition --> 
 * @generated
 */
public abstract class RepositoryControllerBean implements javax.ejb.SessionBean {
	
	protected static Log m_log = LogFactory.getLog( RepositoryControllerBean.class ); 
	
	private SessionContext m_context;
	
	public static final int ALL = 0;
	public static final int ONE_LEVEL = 1;
	
	public void setSessionContext(SessionContext ctx) throws EJBException, RemoteException {
		m_context = ctx;
	}
	
	
	/**********************
	 * ファシリティツリーのメソッド群
	 **********************/
	
	/**
	 * ファシリティツリー（スコープツリー）取得を取得します。
	 * <BR>
	 * 取得したファシリティツリーには割り当てられたノードを含みます。<BR>
	 * このメソッドはクライアントの画面情報を作成するために
	 * 呼び出されます。クライアントのロケールを引数をして必要とします。<BR>
	 * （最上位のスコープという表記をscopeをいう表記を切り替えるため。）
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
	 *     role-name="RepositoryRead"
	 *     method-intf="Remote"
	 * 
	 * @version 1.0.0
	 * @since 1.0.0
	 * 
	 * @param locale クライアントのロケール
	 * @return FacilityTreeItemの階層オブジェクト
	 */
	public FacilityTreeItem getFacilityTree(Locale locale) {
		/** メイン処理 */
		return FacilitySelector.getFacilityTree(locale, false, null);
	}
	
	/**
	 * ファシリティツリー（スコープツリー）取得を取得します。(有効なノードのみ)
	 * <BR>
	 * 取得したファシリティツリーには割り当てられたノードを含みます。<BR>
	 * このメソッドはクライアントの画面情報を作成するために
	 * 呼び出されます。クライアントのロケールを引数をして必要とします。<BR>
	 * （最上位のスコープという表記をscopeをいう表記を切り替えるため。）
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
	 *     role-name="RepositoryRead"
	 *     method-intf="Remote"
	 * 
	 * @version 1.0.0
	 * @since 1.0.0
	 * 
	 * @param locale クライアントのロケール
	 * @return FacilityTreeItemの階層オブジェクト
	 */
	public FacilityTreeItem getExecTargetFacilityTree(Locale locale) {
		/** メイン処理 */
		return FacilitySelector.getFacilityTree(locale, false, Boolean.TRUE);
	}
	
	
	/**********************
	 * ノードのメソッド群(getter)
	 **********************/
	/**
	 * ノード一覧を取得します。<BR>
	 * リポジトリに登録されているすべてのノードを取得します。<BR>
	 * 戻り値はNodeInfoのArrayListで、NodeInfoには
	 * ノードの"ファシリティID"、"ファシリティ名"、"説明"のみ格納されています。
	 * getNodeFacilityIdListを利用すること。（getNodeと組み合わせて利用する。）
	 * 
	 * @ejb.interface-method
	 *     
	 * @ejb.permission
	 *     role-name="RepositoryRead"
	 *     method-intf="Remote"
	 * 
	 * @version 1.0.0
	 * @since 1.0.0
	 * 
	 * @return NodeInfoの配列
	 */
	@Deprecated
	public ArrayList<NodeInfo> getNodeList() {
		/** メイン処理 */
		return FacilitySelector.getNodeList();
	}
	
	/**
	 * ノード一覧を取得します。<BR>
	 * 
	 * クライアントなどで検索した場合に呼ばれ、該当するノード一覧を取得します。<BR>
	 * 引数はNodeInfoであり、"ファシリティID"、"ファシリティ名"、"説明"、
	 * "IPアドレス"、"OS名"、"OSリリース"、"管理者"、"連絡先"が１つ以上含まれており、
	 * その条件を元に該当するノードを戻り値とします。<BR>
	 * 戻り値はNodeInfoのArrayListで、NodeInfoには
	 * ノードの"ファシリティID"、"ファシリティ名"、"説明"のみ格納されています。<BR>
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
	 *     role-name="RepositoryRead"
	 *     method-intf="Remote"
	 * 
	 * @version 1.0.0
	 * @since 1.0.0
	 * 
	 * @param property　検索条件のプロパティ
	 * @return NodeInfoの配列
	 */
	public ArrayList<NodeInfo> getFilterNodeList(NodeInfo property) {
		/** メイン処理 */
		return FacilitySelector.getFilterNodeList(property);
	}
	
	/**
	 * 
	 * 監視・ジョブ等の処理を実行する対象となる、ファシリティIDのリストを取得します。
	 * 引数で指定されたファシリティIDが、ノードかスコープによって、以下のようなリストを取得します。
	 * 
	 * ノードの場合
	 *   引数で指定されたfacilityIdが格納されたArrayList
	 *   ただし、管理対象（有効/無効フラグが真）の場合のみ
	 * 
	 * スコープの場合
	 *   配下に含まれるノードのファシリティIDが格納されたArrayList
	 *   ただし、管理対象（有効/無効フラグが真）のみ
	 * 
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
	 *     role-name="RepositoryRead"
	 *     method-intf="Remote"
	 * 
	 * @version 3.0.0
	 * @since 3.0.0
	 * 
	 * 
	 * @param facilityId 処理を実行する対象となるファシリティID
	 * @return 有効なノードのリスト（有効なノードがひとつも含まれない場合は空のリスト）
	 */
	public ArrayList<String> getExecTargetFacilityIdList(String facilityId) {
		/** メイン処理 */
		return FacilitySelector.getNodeFacilityIdList(facilityId, RepositoryControllerBean.ALL, false, true);
	}
	
	/**
	 * ノードのファシリティIDリストを取得します。<BR>
	 * 
	 * リポジトリにあるすべてのノードのリストを取得します。<BR>
	 * 戻り値は ファシリティID(String)のArrayList<BR>
	 * 
	 * getNodeList() との違いはこちらの戻り値はArrayListの２次元ではなく、
	 * 単純にファシリティID（String）のみのArrayList
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
	 *     role-name="RepositoryRead"
	 *     method-intf="Remote"
	 * 
	 * @version 1.0.0
	 * @since 1.0.0
	 * 
	 * @return ファシリティIDの配列
	 */
	public ArrayList<String> getNodeFacilityIdList() {
		/** メイン処理 */
		return FacilitySelector.getNodeFacilityIdList(false);
	}
	
	/**
	 * ノードのファシリティIDリストを取得します。<BR>
	 * 
	 * リポジトリにあるすべてのノードのリストを取得します。<BR>
	 * 戻り値は ファシリティID(String)のArrayList
	 * 引数のsortにtrueをセットした場合には、listがCollator.compare()にしたがってソートされる。<BR>
	 * 
	 * getNodeList() との違いはこちらの戻り値はNodeInfoのArrayListではなく、
	 * 単純にファシリティID（String）のみのArrayList
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
	 *     role-name="RepositoryRead"
	 *     method-intf="Remote"
	 * 
	 * @version 2.1.0
	 * @since 2.1.0
	 * 
	 * @param sort sort ソートするか？(する:true しない:false)
	 * @return ファシリティIDの配列
	 */
	public ArrayList<String> getNodeFacilityIdList(boolean sort) {
		/** メイン処理 */
		return FacilitySelector.getNodeFacilityIdList(sort);
	}
	
	/**
	 * ノードの詳細プロパティを取得します。<BR>
	 * 
	 * faciliyIDで指定されるノードの詳細プロパティを取得します。<BR>
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
	 *     role-name="RepositoryRead"
	 *     method-intf="Remote"
	 * 
	 * @version 1.0.0
	 * @since 1.0.0
	 * 
	 * @param facilityId ファシリティID
	 * @param locale クライアントのロケール
	 * @return ノード情報プロパティ
	 * @throws FacilityNotFoundException 
	 */
	public NodeInfo getNode(String facilityId) throws FacilityNotFoundException {
		/** メイン処理 */
		return NodeProperty.getProperty(facilityId);
	}
	
	
	/**
	 * ファシリティパスを取得します。<BR>
	 * 
	 * 第一引数がノードの場合は、パスではなく、ファシリティ名。<BR>
	 * 第一引数がスコープの場合は、第二引数との相対的なファシリティパスを取得します。<BR>
	 * (例　○○スコープ>××システム>DBサーバ)<BR>
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
	 *     role-name="RepositoryRead"
	 *     method-intf="Remote"
	 *     
	 * @version 1.0.0
	 * @since 1.0.0
	 * 
	 * @param facilityId ファシリティID
	 * @param parentFacilityId 上位のファシリティID
	 * @return String ファシリティパス
	 */
	public String getFacilityPath(String facilityId, String parentFacilityId) {
		/** メイン処理 */
		return FacilitySelector.getNodeScopePath(parentFacilityId, facilityId);
	}
	
	
	/**
	 * SNMPを利用してノードの情報を取得します。<BR>
	 * 
	 * 
	 * クライアントからSNMPで検出を行った際に呼び出されるメソッドです。<BR>
	 * SNMPポーリングにより、ノード詳細プロパティをセットし、クライアントに返す。
	 * 戻り値はNodeInfo
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
	 * role-name="RepositoryRead"
	 * method-intf="Remote"
	 * 
	 * @version 2.1.2
	 * @since 2.1.2
	 * 
	 * @param ポーリング対象のIPアドレス、コミュニティ名、バージョン、ポート
	 * @param locale クライアントのロケール
	 * @return ノード情報
	 * @throws UnknownHostException 
	 * @throws FacilityNotFoundException 
	 * @throws HinemosUnknownException 
	 */
	public NodeInfo getNodePropertyBySNMP(String ipAddress, int port, String community, String version)
			throws UnknownHostException, FacilityNotFoundException, HinemosUnknownException {
		/** メイン処理 */
		NodeInfo property = null;
		try {
			property = SearchNodeBySNMP.searchNode(ipAddress, port, community, version);
		} catch (UnknownHostException e) {
			throw e;
		} catch (NamingException e) {
			m_log.error("getNodePropertyBySNMP " + e.getMessage());
			throw new HinemosUnknownException(e.getMessage(), e);
		}
		return property;
	}

	/**
	 * 条件のHashMapに該当するノードのファシリティIDのリストを返却する。<BR>
	 * このメソッドは性能が低いため、要注意。
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
	 *     role-name="RepositoryRead"
	 *     method-intf="Remote"
	 * 
	 * @version 3.1.0
	 * @since 3.1.0
	 * 
	 * @return ArrayList<String>
	 */
	@Deprecated
	public ArrayList<String> findByCondition(HashMap<String, String> condition){
		/** メイン処理 */
		return FacilitySelector.getNodeFacilityIdListByCondition(condition);
	}
	
	/**
	 * ファシリティIDリストを取得します。<BR>
	 * 
	 *  引数のホスト名（ノード名）またはIPアドレスに対応するノードのファシリティIDのリストを
	 *  取得します。<BR>
	 *  戻り値はファシリティID(String)のArrayList
	 * getNodeList(NodeInfo)を利用すること。
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
	 *     role-name="RepositoryRead"
	 *     method-intf="Remote"
	 * 
	 * @version 1.0.0
	 * @since 1.0.0
	 * 
	 * @param hostName ホスト名（ノード名）
	 * @param ipAddress　IPアドレス(v4)
	 * @return ファシリティIDの配列
	 */
	@Deprecated
	public ArrayList<String> getFacilityIdList(String hostName, String ipAddress) {
		/** メイン処理 */
		return FacilitySelector.getFacilityIdList(hostName, ipAddress);
	}
	
	/**
	 * ファシリティIDより対象ノードのノードプロパティ情報をDataオブジェクトで取得する。<BR>
	 * このメソッドは使用せず、getNodePropertyを利用すること！
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
	 * role-name="RepositoryRead"
	 * method-intf="Local"
	 * 
	 * @param facilityId ノードのファシリティID
	 * @return ノードのプロパティ情報一式
	 * 
	 * @version 3.1.0
	 * @since 3.1.0
	 * 
	 */
	@Deprecated
	public FacilityInfoData  getNodeFacilityData(String facilityId) {
		/** メイン処理 */
		return FacilitySelector.getNodeFacilityData(facilityId);
	}
	
	
	/**
	 * ファシリティの表示ソート順位を取得する。<BR>
	 * getNodeを利用すること！
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
	 *     role-name="RepositoryRead"
	 *     method-intf="Remote"
	 * 
	 * @version 3.1.0
	 * @since 3.1.0
	 * 
	 * @return Property
	 * @throws FacilityNotFoundException 
	 * @throws HinemosUnknownException 
	 */
	@Deprecated
	public int getDisplaySortOrder(String facilityId) throws FacilityNotFoundException, HinemosUnknownException {
		/** メイン処理 */
		try {
			return FacilitySelector.getDisplaySortOrder(facilityId);
		} catch (NamingException e) {
			m_log.error("getDisplaySortOrder " + e.getMessage());
			throw new HinemosUnknownException(e.getMessage(), e);
		}
	}
	
	
	/**
	 * 条件のHashMapに該当するノードのリストを返却する。<BR>
	 * このメソッドは性能が低いため、要注意。
	 * 
	 * このメソッドを利用せず、findByConditionとgetNodePropertyを利用すること。
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
	 *     role-name="RepositoryRead"
	 *     method-intf="Remote"
	 *     
	 * @version 3.2.0
	 * @since 3.2.0
	 * @return ArrayList<NodeLocal>
	 */
	@Deprecated
	public ArrayList<NodeLocal> getNodeByCondition(HashMap<String, String> condition){
		return FacilitySelector.getNodeByCondition(condition);
	}
	

	/**
	 * デバイス情報リストを取得します。<BR>
	 * 指定したファシリティIDのデバイスのリストを取得します。<BR>
	 * 戻り値はcom.clustercontrol.performanceMGR.bean.DeviceDataのArrayList
	 *
	 * @see com.clustercontrol.performanceMGR.bean.DeviceData
	 * このメソッドは使用せず、getNodeを利用すること！
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
	 *     role-name="RepositoryRead"
	 *     method-intf="Remote"
	 * 
	 * @version 1.0.0
	 * @since 1.0.0
	 * 
	 * @param facilityId
	 * @return DeviceDataの配列
	 */
	@Deprecated
	public ArrayList<DeviceInfoData> getDeviceList(String facilityId) {
		/** メイン処理 */
		return FacilitySelector.getDeviceList(facilityId);
	}
	
	/**
	 * ノード詳細情報を取得します。<BR>
	 * 
	 * ファシリティIDに対応するノードの詳細情報を取得します。<BR>
	 * 引数のattributesには、com.clustercontrol.repository.bean.FacilityAttributeConstantに
	 * 定義される属性を1つ以上定義します。<BR>
	 * 戻り値のHashMapには属性とその時の値セットが格納されます。<BR>
	 * 
	 * このメソッドは使用せず、getNodeを利用すること！
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
	 *     role-name="RepositoryRead"
	 *     method-intf="Remote"
	 * 
	 * @version 1.0.0
	 * @since 1.0.0
	 * 
	 * @param facilityId　ファシリティID
	 * @param attributes  取得したい属性のリスト
	 * @return 属性名と値のHashMapとして返す
	 * @throws FacilityNotFoundException 
	 */
	@Deprecated
	public HashMap<String, Object> getNodeDetail(String facilityId, ArrayList<String> attributes) throws FacilityNotFoundException {
		/** メイン処理 */
		return FacilitySelector.getNodeDetail(facilityId, attributes);
	}
	
	/**
	 *　複数ノードのノード詳細を一度に取得します。<BR>
	 * 
	 * 指定したファシリティIDのリストに対応する属性の集合を取得します。<BR>
	 * 戻り値は ファシリティIDとその属性のHashMap
	 * HashMapからファシリティIDをkeyにして取り出したObjectはさらにHashMap
	 * そのHashMapの中身は引数に指定したattributesとそれに対応する値
	 * 
	 * このメソッドは使用せず、getNodeを利用すること！
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
	 *     role-name="RepositoryRead"
	 *     method-intf="Remote"
	 * 
	 * @version 1.0.0
	 * @since 1.0.0
	 * 
	 * @param facilityIdList
	 * @param attributes
	 * @return ファシリティIDと属性値(属性名と値のHashMap)のHashMap
	 * @throws FacilityNotFoundException 
	 */
	@Deprecated
	public HashMap<String, HashMap<String, Object>> getNodeDetail(ArrayList<String> facilityIdList, ArrayList<String> attributes) throws FacilityNotFoundException {
		/** メイン処理 */
		return FacilitySelector.getNodeDetail(facilityIdList, attributes);
	}
	
	/**
	 * ノード詳細を設定します。<BR>
	 * 
	 * ファシリティIDに対応するノードに属性attributesを設定します。<BR>
	 * 引数のattributesには、com.clustercontrol.repository.bean.FacilityAttributeConstantで指定される
	 * 属性と設定したいの値のHashMap
	 * 
	 * このメソッドは使用せず、modifyNodeを利用すること！
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
	 *     role-name="RepositoryWrite"
	 *     method-intf="Remote"
	 * 
	 * @version 1.0.0
	 * @since 1.0.0
	 * 
	 * @param facilityId　ファシリティID
	 * @param attributes 属性名と値のHashMap
	 */
	@Deprecated
	public void setNodeDetail(String facilityId, HashMap<String, Object> attributes) {
		/** メイン処理 */
		FacilityModifier.setNodeDetail(facilityId, attributes);
	}
	
	
	/**********************
	 * ノードのメソッド群(getter以外)
	 **********************/
	
	/**
	 * ノードを新規に追加します。<BR>
	 * またこのメソッドで組み込みスコープである"登録済みノード"スコープにも
	 * 割り当てが行われます。
	 * 
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
	 *     role-name="RepositoryWrite"
	 *     method-intf="Remote"
	 * 
	 * @version 3.1.0
	 * @since 1.0.0
	 * 
	 * @param nodeinfo 追加するノード情報のプロパティ
	 * @throws FacilityDuplicateException 
	 */
	public void addNode(NodeInfo nodeInfo) throws FacilityDuplicateException {
		/** メイン処理 */
		try {
			FacilityModifier.addNode(nodeInfo, m_context.getCallerPrincipal().getName(), FacilitySortOrderConstant.DEFAULT_SORT_ORDER_NODE, true);
		} catch (DuplicateKeyException e) {
			m_log.warn("addNode() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw new FacilityDuplicateException(e.getMessage(), e);
		}
		// リポジトリ更新時刻リフレッシュ
		RepositoryUpdateTime.refleshUpdateTime();
	}
	
	/**
	 * ノードを新規に追加します。（リポジトリ更新TOPIC未送信選択可能）<BR>
	 * またこのメソッドで組み込みスコープである"登録済みノード"スコープにも
	 * 割り当てが行われます。
	 * 
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
	 *     role-name="RepositoryWrite"
	 *     method-intf="Remote"
	 * 
	 * @version 3.1.0
	 * @since 1.0.0
	 * 
	 * @param property 追加するノード情報のプロパティ
	 * @throws FacilityDuplicateException 
	 */
	public void addNode(NodeInfo property, boolean topicSendFlg) throws FacilityDuplicateException {
		/** メイン処理 */
		try {
			FacilityModifier.addNode(property, m_context.getCallerPrincipal().getName(), FacilitySortOrderConstant.DEFAULT_SORT_ORDER_NODE, true);
		} catch (DuplicateKeyException e) {
			m_log.warn("addNode() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw new FacilityDuplicateException(e.getMessage(), e);
		}
		// リポジトリ更新時刻リフレッシュ
		RepositoryUpdateTime.refleshUpdateTime();
	}
	
	/**
	 * ノードを変更します。<BR>
	 * 引数のpropertyには変更する属性のみを設定してください。<BR>
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
	 *     role-name="RepositoryWrite"
	 *     method-intf="Remote"
	 * 
	 * @version 2.0.0
	 * @since 1.0.0
	 * 
	 * @param property　変更するノード情報のプロパティ
	 */
	public void modifyNode(NodeInfo property) {
		/** メイン処理 */
		FacilityModifier.modifyNode(property, m_context.getCallerPrincipal().getName(), true);
		
		// リポジトリ更新時刻リフレッシュ
		RepositoryUpdateTime.refleshUpdateTime();
	}
	
	/**
	 * ノード詳細情報を一括変更します。<BR>
	 * クライアントにあるスコープを指定して変更を行うためのメソッド
	 * 引数のpropertyには変更する属性のみを設定してください。<BR>
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
	 *     role-name="RepositoryWrite"
	 *     method-intf="Remote"
	 * 
	 * @version 2.0.0
	 * @since 1.0.0
	 * 
	 * @param property　変更するノード情報のプロパティ
	 * @param parentFacilityId　スコープのファシリティID
	 */
	public void modifyNode(NodeInfo property, String parentFacilityId) {
		/** メイン処理 */
		FacilityModifier.modifyAllNode(property, parentFacilityId, m_context.getCallerPrincipal().getName(), true);
		
		// リポジトリ更新時刻リフレッシュ
		RepositoryUpdateTime.refleshUpdateTime();
	}
	
	/**
	 * ノード情報を削除します。<BR>
	 * 
	 * faciityIDで指定されたノードをリポジトリから削除します。
	 * 
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
	 *     role-name="RepositoryWrite"
	 *     method-intf="Remote"
	 * 
	 * @version 1.0.0
	 * @since 1.0.0
	 * 
	 * @param facilityId ファシリティID
	 * @throws UsedFacilityException 
	 */
	public void deleteNode(String facilityId) throws UsedFacilityException {
		/** メイン処理 */
		try {
			FacilityModifier.deleteNode(facilityId, m_context.getCallerPrincipal().getName(), true);
		} catch (UsedFacilityException e) {
			m_log.warn("deleteNode() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw e;
		}
		
		// リポジトリ更新時刻リフレッシュ
		RepositoryUpdateTime.refleshUpdateTime();
	}
	


	
	/**********************
	 * スコープのメソッド群
	 **********************/
	/**
	 * ファシリティID一覧を取得します。<BR>
	 * あるスコープを指定してその直下にあるスコープとノードを取得します。<BR>
	 * このメソッドは引数としてそのスコープのファシリティIDを要求します。<BR>
	 * 戻り値はScopeInfoのArrayListで中のScopeInfoには子スコープの
	 * "ファシリティID"、"ファシリティ名"、"説明"のみ格納されています。<BR>
	 * TODO 返り値をScopeInfoからFacilityInfoに変更。
	 * TODO メソッド名をgetFacilityListに変更。
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
	 *     role-name="RepositoryRead"
	 *     method-intf="Remote"
	 * 
	 * @version 1.0.0
	 * @since 1.0.0
	 * 
	 * @param parentFacilityId ファシリティID
	 * @return ScopeInfoの配列
	 */
	@Deprecated
	public ArrayList<ScopeInfo> getScopeList(String parentFacilityId) {
		/** メイン処理 */
		return FacilitySelector.getFacilityListAssignedScope(parentFacilityId);
	}
	
	/**
	 * スコープ用プロパティ情報を取得します。<BR>
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
	 *     role-name="RepositoryRead"
	 *     method-intf="Remote"
	 * 
	 * @version 1.0.0
	 * @since 1.0.0
	 * 
	 * @param facilityId ファシリティID
	 * @param locale クライアントのロケール
	 * @return スコープのプロパティ情報（ファシリティID、ファシリティ名、説明）
	 * @throws FacilityNotFoundException 
	 * @throws HinemosUnknownException
	 */
	public ScopeInfo getScopeProperty(String facilityId) throws FacilityNotFoundException, HinemosUnknownException {
		/** メイン処理 */
		ScopeInfo property = null;
		try {
			property = ScopeProperty.getProperty(facilityId);
		} catch (NamingException e) {
			m_log.error("NamingException " + e.getMessage(), e);
			throw new HinemosUnknownException(e.getMessage(), e);
		}
		return property;
	}
	
	/**
	 * スコープを新規に追加します。<BR>
	 * 
	 * parentFacilityIdで指定されるスコープの下にpropertyで指定されるスコープを
	 * 追加します。<BR>
	 * 引数propertyには、"ファシリティID"、"ファシリティ名"、"説明"（任意）を含める必要があります。
	 * 
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
	 *     role-name="RepositoryWrite"
	 *     method-intf="Remote"
	 * 
	 * @version 3.1.0
	 * @since 1.0.0
	 * 
	 * @param parentFacilityId
	 * @param property
	 * @throws FacilityDuplicateException 
	 */
	public void addScope(String parentFacilityId, ScopeInfo property) throws FacilityDuplicateException {
		/** メイン処理 */
		try {
			FacilityModifier.addScope(parentFacilityId, property, m_context.getCallerPrincipal().getName(), FacilitySortOrderConstant.DEFAULT_SORT_ORDER_SCOPE, true);
		} catch (DuplicateKeyException e) {
			m_log.warn("addScope() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw new FacilityDuplicateException(e.getMessage(), e);
		}
		
		// リポジトリ更新時刻リフレッシュ
		RepositoryUpdateTime.refleshUpdateTime();
	}
	
	/**
	 * スコープを新規に追加します(表示順指定)。<BR>
	 * 
	 * parentFacilityIdで指定されるスコープの下にpropertyで指定されるスコープを
	 * 追加します。<BR>
	 * 引数propertyには、"ファシリティID"、"ファシリティ名"、"説明"（任意）を含める必要があります。
	 * 
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
	 *     role-name="RepositoryWrite"
	 *     method-intf="Remote"
	 * 
	 * @version 3.1.0
	 * @since 1.0.0
	 * 
	 * @param parentFacilityId
	 * @param property
	 * @param sortOrder
	 * @throws FacilityDuplicateException 
	 */
	public void addScope(String parentFacilityId, ScopeInfo property, int displaySortOrder) throws FacilityDuplicateException {
		/** メイン処理 */
		try {
			FacilityModifier.addScope(parentFacilityId, property, m_context.getCallerPrincipal().getName(), displaySortOrder, true);
		} catch (DuplicateKeyException e) {
			m_log.warn("addScope() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw new FacilityDuplicateException(e.getMessage(), e);
		}
		
		// リポジトリ更新時刻リフレッシュ
		RepositoryUpdateTime.refleshUpdateTime();
	}
	
	/**
	 * スコープを新規に追加します(表示順指定、リポジトリ更新TOPIC未送信選択可能)。<BR>
	 * 
	 * parentFacilityIdで指定されるスコープの下にpropertyで指定されるスコープを
	 * 追加します。<BR>
	 * 引数propertyには、"ファシリティID"、"ファシリティ名"、"説明"（任意）を含める必要があります。
	 * 
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
	 *     role-name="RepositoryWrite"
	 *     method-intf="Remote"
	 * 
	 * @version 3.1.0
	 * @since 1.0.0
	 * 
	 * @param parentFacilityId
	 * @param property
	 * @param sortOrder
	 * @throws FacilityDuplicateException 
	 */
	public void addScope(String parentFacilityId, ScopeInfo property, int displaySortOrder, boolean topicSendFlg)
			throws FacilityDuplicateException {
		/** メイン処理 */
		try {
			FacilityModifier.addScope(parentFacilityId, property, m_context.getCallerPrincipal().getName(), displaySortOrder, topicSendFlg);
		} catch (DuplicateKeyException e) {
			m_log.warn("addScope() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw new FacilityDuplicateException(e.getMessage(), e);
		}
		
		// リポジトリ更新時刻リフレッシュ
		RepositoryUpdateTime.refleshUpdateTime();
	}
	
	/**
	 * スコープの情報を変更します。<BR>
	 * 
	 * 引数propertyで指定した内容でスコープ情報を更新します。<BR>
	 * 引数propertyには、"ファシリティID"、"ファシリティ名"、"説明"（任意）を含める必要があります。
	 * propertyに含まれるファシリティIDに対応するスコープの情報が変更されます。<BR>
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
	 *     role-name="RepositoryWrite"
	 *     method-intf="Remote"
	 * 
	 * @version 1.0.0
	 * @since 1.0.0
	 * 
	 * @param property
	 */
	public void modifyScope(ScopeInfo property) {
		/** メイン処理 */
		FacilityModifier.modifyScope(property, m_context.getCallerPrincipal().getName(), true);
		
		// リポジトリ更新時刻リフレッシュ
		RepositoryUpdateTime.refleshUpdateTime();
	}
	
	/**
	 * スコープ情報を削除します。<BR>
	 * 
	 * faciityIDで指定されたスコープをリポジトリから削除します。
	 * 
	  * @ejb.interface-method
	 * 
	 * @ejb.permission
	 *     role-name="RepositoryWrite"
	 *     method-intf="Remote"
	 * 
	 * @version 1.0.0
	 * @since 1.0.0
	 * 
	 * @param facilityId ファシリティID
	 * @throws UsedFacilityException 
	 */
	public void deleteScope(String facilityId) throws UsedFacilityException {
		/** メイン処理 */
		try {
			FacilityModifier.deleteScope(facilityId, m_context.getCallerPrincipal().getName(), true);
		} catch (UsedFacilityException e) {
			m_log.warn("deleteScope() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw e;
		}
		
		// リポジトリ更新時刻リフレッシュ
		RepositoryUpdateTime.refleshUpdateTime();
	}
	
	
	
	/**********************
	 * ノード割り当てのメソッド群
	 **********************/
	/**
	 * 割当ノード一覧を取得します。<BR>
	 * 
	 * あるファシリティIDの配下または直下のノード一覧を取得します。<BR>
	 * このメソッドでは、引数levelで直下または配下を制御します。<BR>
	 * 戻り値はNodeInfoのArrayListで、NodeInfoには
	 * ノードの"ファシリティID"、"ファシリティ名"、"説明"のみ格納されています。<BR>
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
	 *     role-name="RepositoryRead"
	 *     method-intf="Remote"
	 * 
	 * @version 1.0.0
	 * @since 1.0.0
	 * 
	 * @param parentFacilityId
	 * @param level　取得レベル 0:ALL(配下) 1:ONE_LEVEL（直下）
	 * @return NodeInfoの配列
	 */
	public ArrayList<NodeInfo> getNodeList(String parentFacilityId, int level) {
		/** メイン処理 */
		return FacilitySelector.getNodeList(parentFacilityId, level);
	}
	
	
	/**
	 * 割当スコープ一覧を取得します。<BR>
	 * 割り当てスコープ一覧とは、あるノードが属しているスコープすべてを
	 * 一覧表示したものです。
	 * クライアントの割り当てスコープビューの表示データとなります。
	 * 戻り値はArrayListのArrayListで中のArrayListには"スコープ"が最上位からの
	 * スコープパス表記で（Stringで）格納されています。
	 * 外のArrayListには、そのレコードが順に格納されています。
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
	 *     role-name="RepositoryRead"
	 *     method-intf="Remote"
	 * 
	 * @version 1.0.0
	 * @since 1.0.0
	 * 
	 * @param facilityId ノードのファシリティID
	 * @return Stringの配列
	 */
	public ArrayList<String> getNodeScopeList(String facilityId) {
		/** メイン処理 */
		return FacilitySelector.getNodeScopeList(facilityId);
	}
	
	/**
	 * ファシリティIDリストを取得します。<BR>
	 * 
	 * 引数に指定した親ファシリティIDの配下または直下のファシリティ（スコープ、ノード）の
	 * リストを取得します。<BR>
	 * 戻り値は ファシリティID（String）のArrayList
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
	 *     role-name="RepositoryRead"
	 *     method-intf="Remote"
	 * 
	 * @version 2.1.0
	 * @since 1.0.0
	 * 
	 * @param parentFacilityId ファシリティID
	 * @param level　取得レベル 0:ALL(配下) 1:ONE_LEVEL（直下）
	 * @return ファシリティIDの配列
	 */
	public ArrayList<String> getFacilityIdList(String parentFacilityId, int level) {
		/** メイン処理 */
		return FacilitySelector.getFacilityIdList(parentFacilityId, level, false);
	}
	
	/**
	 * ファシリティIDリストを取得します。<BR>
	 * 
	 * 引数に指定した親ファシリティIDの配下または直下のファシリティ（スコープ、ノード）の
	 * リストを取得します。<BR>
	 * 戻り値は ファシリティID（String）のArrayList
	 * 引数のsortにtrueをセットした場合には、listがCollator.compare()にしたがってソートされる。
	 * 
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
	 *     role-name="RepositoryRead"
	 *     method-intf="Remote"
	 * 
	 * @version 2.1.0
	 * @since 2.1.0
	 * 
	 * @param parentFacilityId ファシリティID
	 * @param level 取得レベル 0:ALL(配下) 1:ONE_LEVEL（直下）
	 * @param sort ソートするか？(する:true しない:false)
	 * @return ファシリティIDの配列
	 */
	public ArrayList<String> getFacilityIdList(String parentFacilityId, int level, boolean sort) {
		/** メイン処理 */
		return FacilitySelector.getFacilityIdList(parentFacilityId, level, sort);
	}
	
	/**
	 * ノードのファシリティIDリストを取得します。<BR>
	 * 
	 * 引数に指定した親ファシリティIDの配下または直下のファシリティ（ノード）の
	 * リストを取得します。<BR>
	 * 戻り値は ファシリティID（String）のArrayList
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
	 *     role-name="RepositoryRead"
	 *     method-intf="Remote"
	 * 
	 * @version 2.1.0
	 * @since 1.0.0
	 * 
	 * @param parentFacilityId ファシリティID
	 * @param level  取得レベル 0:ALL(配下) 1:ONE_LEVEL（直下）
	 * @return ファシリティIDの配列
	 */
	public ArrayList<String> getNodeFacilityIdList(String parentFacilityId, int level) {
		/** メイン処理 */
		return FacilitySelector.getNodeFacilityIdList(parentFacilityId, level, false, true);
	}
	
	/**
	 * ノードのファシリティIDリスト取得<BR>
	 * 引数に指定した親ファシリティIDの配下または直下のファシリティ（ノード）の
	 * リストを取得します。<BR>
	 * 戻り値は ファシリティID（String）のArrayList
	 * 引数のsortにtrueをセットした場合には、listがCollator.compare()にしたがってソートされる。
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
	 *     role-name="RepositoryRead"
	 *     method-intf="Remote"
	 * 
	 * @version 2.1.0
	 * @since 2.1.0
	 * 
	 * @param parentFacilityId
	 * @param level   取得レベル 0:ALL(配下) 1:ONE_LEVEL（直下）
	 * @param sort sort ソートするか？(する:true しない:false)
	 * @return ファシリティIDの配列
	 */
	public ArrayList<String> getNodeFacilityIdList(String parentFacilityId, int level, boolean sort, Boolean valid) {
		/** メイン処理 */
		return FacilitySelector.getNodeFacilityIdList(parentFacilityId, level, sort, valid);
	}
	
	/**
	 * スコープへのノードの割り当てを行います。（リポジトリ更新TOPIC未送信選択可能）<BR>
	 * 
	 * parentFacilityIdで指定されるスコープにfacilityIdsで指定されるノード群を
	 * 割り当てます。
	 * 
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
	 *     role-name="RepositoryWrite"
	 *     method-intf="Remote"
	 * 
	 * @version 3.1.0
	 * @since 1.0.0
	 * 
	 * @param parentFacilityId　ノードを割り当てるスコープ
	 * @param facilityIds 割り当てさせるノード(群)
	 */
	public void assignNodeScope(String parentFacilityId, String[] facilityIds, boolean topicSendFlg) {
		/** メイン処理 */
		FacilityModifier.assignFacilityToScope(parentFacilityId, facilityIds, topicSendFlg);
		
		// リポジトリ更新時刻リフレッシュ
		RepositoryUpdateTime.refleshUpdateTime();
	}
	
	/**
	 * スコープへのノードの割り当てを行います。<BR>
	 * 
	 * parentFacilityIdで指定されるスコープにfacilityIdsで指定されるノード群を
	 * 割り当てます。
	 * 
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
	 *     role-name="RepositoryWrite"
	 *     method-intf="Remote"
	 * 
	 * @version 3.1.0
	 * @since 1.0.0
	 * 
	 * @param parentFacilityId　ノードを割り当てるスコープ
	 * @param facilityIds 割り当てさせるノード(群)
	 */
	public void assignNodeScope(String parentFacilityId, String[] facilityIds) {
		/** メイン処理 */
		FacilityModifier.assignFacilityToScope(parentFacilityId, facilityIds, true);
		
		// リポジトリ更新時刻リフレッシュ
		RepositoryUpdateTime.refleshUpdateTime();
	}
	
	/**
	 * ノードをスコープから削除します。（割り当てを解除します。）<BR>
	 * parentFacilityIdで指定されるスコープからfacilityIdsで指定されるノード群を
	 * 削除（割り当て解除）します。
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
	 *     role-name="RepositoryWrite"
	 *     method-intf="Remote"
	 * 
	 * @version 3.1.0
	 * @since 1.0.0
	 * 
	 * @param parentFacilityId ノードを取り除くスコープ
	 * @param facilityIds 取り除かれるノード（群）
	 */
	public void releaseNodeScope(String parentFacilityId, String[] facilityIds) {
		/** メイン処理 */
		FacilityModifier.releaseNodeFromScope(parentFacilityId, facilityIds, m_context.getCallerPrincipal().getName(), true);
		
		// リポジトリ更新時刻リフレッシュ
		RepositoryUpdateTime.refleshUpdateTime();
	}
	
	
	/**
	 * ノードをスコープから削除します。（割り当てを解除します。リポジトリ更新TOPIC未送信選択可能）<BR>
	 * parentFacilityIdで指定されるスコープからfacilityIdsで指定されるノード群を
	 * 削除（割り当て解除）します。
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
	 *     role-name="RepositoryWrite"
	 *     method-intf="Remote"
	 * 
	 * @version 3.1.0
	 * @since 1.0.0
	 * 
	 * @param parentFacilityId ノードを取り除くスコープ
	 * @param facilityIds 取り除かれるノード（群）
	 */
	public void releaseNodeScope(String parentFacilityId, String[] facilityIds, boolean topicSendFlg) {
		/** メイン処理 */
		FacilityModifier.releaseNodeFromScope(parentFacilityId, facilityIds, m_context.getCallerPrincipal().getName(), topicSendFlg);
		
		// リポジトリ更新時刻リフレッシュ
		RepositoryUpdateTime.refleshUpdateTime();
	}
	
	
	/**********************
	 * その他のメソッド群
	 **********************/
	/**
	 * ファシリティがノードかどうかをチェックします。<BR>
	 * 
	 * ファシリティIDに対応するものがノードかチェックし、結果をbooleanで返します。
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
	 *     role-name="RepositoryRead"
	 *     method-intf="Remote"
	 * 
	 * @version 1.0.0
	 * @since 1.0.0
	 * 
	 * @param facilityId　ファシリティID
	 * @return true：ノード　false:ノードではない（スコープ）
	 * @throws FacilityNotFoundException 指定されたIDに該当するファシリティが存在しない場合
	 */
	public boolean isNode(String facilityId) throws FacilityNotFoundException {
		/** メイン処理 */
		try {
			return FacilitySelector.isNode(facilityId);
		} catch (FacilityNotFoundException e) {
			m_log.warn("isNode() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw e;
		}
	}
	
	/**
	 * セパレータ文字列を取得します。<BR>
	 * 
	 * セパレータ文字列はスコープパス表示の際のスコープを区切る文字列
	 * 
	 * @ejb.interface-method
	 * @generated
	 * 
	 * @ejb.permission
	 *     unchecked="true"
	 *     method-intf="Remote"
	 * 
	 * @version 1.0.0
	 * @since 1.0.0
	 * 
	 * @return セパレータ文字列
	 */
	public String getSeparator() {
		/** メイン処理 */
		return FacilitySelector.SEPARATOR;
	}
	
	/**
	 * リポジトリ更新のTOPICを送信する。<BR>
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
	 *     role-name="RepositoryWrite"
	 *     method-intf="Remote"
	 *     
	 * @version 3.1.0
	 * @since 3.1.0
	 * 
	 */
	public void sendRepositoryUpdateTopic(){
		/** メイン処理 */
		SendTopic.sendTopic(TopicConstant.TOPIC_NAME_EXECUTE);
	}
	

	/**
	 * ノード作成変更時に、利用可能プラットフォームを表示するためのメソッド。
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
	 *     role-name="RepositoryRead"
	 *     method-intf="Remote"
	 *     
	 * @version 3.2.0
	 * @since 3.2.0
	 * @return ArrayList<RepositoryTableInfo>
	 * @throws HinemosUnknownException 
	 */
	public ArrayList<RepositoryTableInfo> getPlatformList() throws HinemosUnknownException{
		ArrayList<RepositoryTableInfo> list = new ArrayList<RepositoryTableInfo>();
		try {
			for (CollectorPlatformMstLocal bean : (Collection<CollectorPlatformMstLocal>)CollectorPlatformMstUtil.getLocalHome().findAll()) {
				list.add(new RepositoryTableInfo(bean.getPlatformId(), bean.getPlatformName()));
			}
		} catch (Exception e) {
			m_log.error("getPlatformList " + e.getMessage(), e);
			throw new HinemosUnknownException(e.getMessage(), e);
		}
		return list;
	}
	
	/**
	 * ノード作成変更時に、仮想化種別hostのファシリティIDを表示するためのメソッド。
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
	 *     role-name="RepositoryRead"
	 *     method-intf="Remote"
	 *     
	 * @version 3.2.0
	 * @since 3.2.0
	 * @return ArrayList
	 * @throws HinemosUnknownException 
	 */
	public ArrayList<RepositoryTableInfo> getHostList() throws HinemosUnknownException{
		ArrayList<RepositoryTableInfo> list = new ArrayList<RepositoryTableInfo>();
		try {
			for (FacilityLocal bean : (Collection<FacilityLocal>)FacilityUtil.getLocalHome().findAll()) {
				if (bean.isNode()) {
					if ("host".equals(bean.getNode().getVirtualizationNodeType()) ||
							"controller".equals(bean.getNode().getVirtualizationNodeType())) {
						list.add(new RepositoryTableInfo(bean.getFacilityId(), bean.getFacilityName()));
					}
				}
			}
		} catch (Exception e) {
			m_log.error("getHostList " + e.getMessage(), e);
			throw new HinemosUnknownException(e.getMessage(), e);
		}
		return list;
	}
	
	/**
	 * ノード作成変更時に、利用可能な仮想化ソリューションを表示するためのメソッド。
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
	 *     role-name="RepositoryRead"
	 *     method-intf="Remote"
	 *     
	 * @version 3.2.0
	 * @since 3.2.0
	 * @return ArrayList
	 * @throws HinemosUnknownException 
	 */
	public ArrayList<RepositoryTableInfo> getVmSolutionMstList() throws HinemosUnknownException{
		ArrayList<RepositoryTableInfo> list = new ArrayList<RepositoryTableInfo>();
		try {
			for (VmSolutionMstLocal bean : (Collection<VmSolutionMstLocal>)VmSolutionMstUtil.getLocalHome().findAll()) {
				list.add(new RepositoryTableInfo(bean.getVmSolutionId(), bean.getVmSolutionName()));
			}
		} catch (Exception e) {
			m_log.error("getVmSolutionMstList " + e.getMessage(), e);
			throw new HinemosUnknownException(e.getMessage(), e);
		}
		return list;
	}
	
	/**
	 * ノード作成変更時に、利用可能な仮想化プロトコルを表示するためのメソッド。
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
	 *     role-name="RepositoryRead"
	 *     method-intf="Remote"
	 *     
	 * @version 3.2.0
	 * @since 3.2.0
	 * @return ArrayList
	 * @throws HinemosUnknownException 
	 */
	public ArrayList<String> getVmProtocolMstList() throws HinemosUnknownException{
		ArrayList<String> list = new ArrayList<String>();
		try {
			for (String protocol : (Collection<String>)VmProtocolMstUtil.getLocalHome().findDistinctProtocol()) {
				list.add(protocol);
			}
		} catch (Exception e) {
			m_log.error("getPlatformList " + e.getMessage(), e);
			throw new HinemosUnknownException(e.getMessage(), e);
		}
		return list;
	}
	/**
	 * リポジトリの最終更新時刻を取得
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
	 *     unchecked="true"
	 *     method-intf="Remote"
	 *     
	 * @return
	 */
	public Date getLastUpdate(){
		Date updateTime = RepositoryUpdateTime.getUpdateTime();
		m_log.debug("getLastUpdate() lastUpdate = " + updateTime.toString());
		return updateTime;
	}
}